package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.HasClickHandlers;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.InternalTable;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableDetailViewEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class DetailFeatureHandler<K, R extends Record<K>> extends
        AbstractFeatureHandler<K, R, DetailFeature<K, R>> {

	/* Widgets */
	private ButtonWidget createButton, deleteButton;
	private DetailFeatureWidget<K, R> widget;

	/* Anchor information */
	private static final String CREATE_VIEW = "Create";
	private static final String EDIT_VIEW = "Edit";
	private static final String RECORD_ID_TOKEN = "id";

	/**
	 * Constructor
	 * 
	 * @param table The table
	 * @param detailFeature The detail feature
	 */
	public DetailFeatureHandler(InternalTable<K, R> table, DetailFeature<K, R> detailFeature) {
		super(table, detailFeature);

		/* Build the required widgets */
		this.buildWidgets();
	}

	/*
	 * Build the initial widgets required by the feature. If there are widgets that are rebuilt
	 * based on triggered events, do no build those widgets here.
	 */
	private void buildWidgets() {
		this.createButton = new ButtonWidget("Create", new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleCreateButtonClick(event);
			}
		});
		createButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
		getTable().registerWidgetOnActionContainer(createButton);

		this.deleteButton = new ButtonWidget("Delete", new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleDeleteButtonClick(event);
			}
		});
		deleteButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		getTable().registerWidgetOnActionContainer(deleteButton);

		updateButtons();
	}

	// ========================================================================
	// ===================== FEATURE HANDLER METHODS
	// ========================================================================

	@Override
	public void refresh() {
		/* Update the detail link for each row in the table */
		for (Column<?, R> column : getInternalTable().getColumns()) {
			if (getFeature().hasDetailLink(column)) {
				final List<Widget> widgetList = getInternalTable().getWidgetsForColumn(column);
				if (widgetList == null)
					continue;

				/* Update all clickable widgets to be able to show the detail feature */
				for (Widget widget : widgetList) {
					if (widget instanceof HasClickHandlers) {
						((HasClickHandlers) widget).addClickHandler(new ClickHandler() {
							public void onClick(ClickEvent event) {
								handleDetailLinkClicked(event);
							}
						});
						widget.setStyleName(TableConstants.STYLE_DATA_LINK);
					}
				}
			}
		}

		updateButtons();
	}

	/* Update the create and delete button functionality */
	private void updateButtons() {
		boolean canCreate = getFeature().showCreateButton();
		createButton.setVisible(canCreate);
		createButton.setEnabled(canCreate);

		boolean canDelete = getFeature().showDeleteButton();
		deleteButton.setVisible(canDelete);
		deleteButton.setEnabled(canDelete);
	}

	@Override
	public Map<String, String> getAnchorTokens() {
		Map<String, String> tokens = super.getAnchorTokens();
		if (isDetailViewShowing()) {
			if (!widget.isNewRecord()) {
				K key = widget.getRecord().getKey();
				String keyValue = getFeature().getKeySerializer().toString(key);
				if (!isBlank(keyValue))
					tokens.put(RECORD_ID_TOKEN, keyValue);
			}
		}

		return tokens;
	}

	@Override
	public void setAnchor(TableAnchor anchor) {
		String viewName = anchor.getViewName();
		if (CREATE_VIEW.equals(viewName)) {
			/* Use the requester to show the create page */
			getInternalTable().showView(new LabelWidget(""), viewName);
			showCreateView();
		} else if (EDIT_VIEW.equals(viewName)) {
			/* Use the requester to show the detail page */
			getInternalTable().showView(new LabelWidget(""), viewName);
			String id = anchor.getTokens().get(RECORD_ID_TOKEN);
			K key = null;
			try {
				key = getFeature().getKeySerializer().fromString(id);
			} catch (Exception e) {
				handleShowDetailPageFailure(e);
				return;
			}

			/* Get the record to display */
			getInternalTable().getRecordForKey(key, new NeptuneAsyncCallback<R>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					handleShowDetailPageFailure(caught);
				}

				@Override
				public void onNeptuneSuccess(R result) {
					showDetailView(result);
				};
			});
		}
	}

	/**
	 * Show the create view
	 */
	public void showCreateView() {
		performCreateAction(false);
	}

	/**
	 * Show the detail view for the given record
	 */
	public void showDetailView(R record) {
		performShowDetailViewAction(record, false);
	}

	/*
	 * Determine if the detail view is currently on display
	 */
	private boolean isDetailViewShowing() {
		return (this.widget != null && this.getInternalTable().isShowingView(widget));
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the button click for the 'create' button
	 */
	private void handleCreateButtonClick(ClickEvent event) {
		performCreateAction(true);
	}

	/*
	 * The create action
	 */
	private void performCreateAction(final boolean userEvent) {
		getFeature().getNew(new NeptuneAsyncCallback<R>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				Window.alert(caught.getLocalizedMessage());
			}

			@Override
			public void onNeptuneSuccess(R result) {
				showDetailView(result, true, userEvent);
			}
		});
	}

	/*
	 * Handle the button click for the 'delete' button
	 */
	private void handleDeleteButtonClick(ClickEvent event) {
		getFeature().delete(getTable().getSelectedRecords(), new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				Window.alert(caught.getLocalizedMessage());
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				fireFeatureEvent(new FeatureEvent(false));
			}
		});
	}

	/*
	 * Handle the click event on the widget linking to the detail view
	 */
	private void handleDetailLinkClicked(ClickEvent event) {
		Widget sender = (Widget) event.getSource();
		R record = getInternalTable().getRecordForWidget(sender);
		performShowDetailViewAction(record, true);
	}

	/*
	 * Perform the 'show detail view' action. The user event flag indicates if the this action is a
	 * UI event triggered by the user or a event triggered through code by the requester
	 */
	private void performShowDetailViewAction(R record, boolean userEvent) {
		showDetailView(record, false, userEvent);
	}

	/*
	 * Cancel action in the detail view, return to table list view
	 */
	void handleViewCancel() {
		showListView();
	}

	/*
	 * Save success in the detail view, return to the table list view
	 */
	void handleViewSaveSuccess() {
		showListView();
	}

	/*
	 * Save failure in the detail view, do nothing
	 */
	void handleViewSaveFailure(Throwable error) {
	}

	/*
	 * Revert action in the detail view, do nothing
	 */
	void handleViewRevert() {
	}

	/*
	 * Handle the error when showing the detail page for an existing record
	 */
	private void handleShowDetailPageFailure(Throwable error) {
		Window.alert(error.getLocalizedMessage());
	}

	/*
	 * Show the detail view for the table widget
	 */
	private void showDetailView(final R record, boolean newRecord, final boolean userEvent) {
		this.widget = new DetailFeatureWidget<K, R>(this, record, newRecord);
		String viewName = newRecord ? CREATE_VIEW : EDIT_VIEW;
		getInternalTable().showView(widget, viewName);
		getInternalTable().fireDetailViewEvent(
		        new TableDetailViewEvent<K, R>(newRecord, record));
		this.fireFeatureEvent(new FeatureEvent(userEvent));
	}

	/*
	 * Return to the list view for the table widget
	 */
	private void showListView() {
		/* Return to the first page of the list view */
		if (getTable().hasPageFeature()) {
			getTable().getPageFeature().setPageNumber(1);
		}

		/* Show the list view and fire the feature event */
		getInternalTable().showListView();
		fireFeatureEvent(new FeatureEvent());
	}
}
