/*
 * Created on May 27, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.datatype;

import com.google.gwt.event.dom.client.HasBlurHandlers;
import com.google.gwt.event.dom.client.HasFocusHandlers;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Focusable;
import com.google.gwt.user.client.ui.HasValue;

/**
 * A Widget that can display a value for a specific data type.
 * 
 * @author Vijay Silva
 */
public abstract class DataTypeWidget<DataType> extends Composite implements HasValue<DataType>,
        Focusable, HasBlurHandlers, HasFocusHandlers {

	/* Properties */
	protected String toolTip;
	protected String errorToolTip;
	protected boolean validationError = false;
	protected String validationToolTip;

	/* The default dependent style name to use for widgets in error state */
	protected static final String ERROR_STYLE = "dataTypeError";

	/**
	 * A Composite widget that maintains a typed value
	 */
	protected DataTypeWidget() {
	}

	/**
	 * Checks if the value displayed on the widget is valid based on the data type. Does not update
	 * the widget state.
	 * 
	 * @return true if the value is valid, false otherwise
	 */
	public abstract boolean isValidValue();

	/**
	 * Get the string value displayed by the widget
	 * 
	 * @return The string value displayed by the widget
	 */
	public abstract String getTextValue();

	/**
	 * Set the text value for the widget where the text value may not be valid based on the data
	 * type.
	 * 
	 * @param textValue The text value
	 */
	public abstract void setTextValue(String textValue);

	/**
	 * Set the widget to be in error state to show a validation error. Specifies the validation
	 * message tool tip. A validation error can occur even when the value is valid based on the data
	 * type.
	 * 
	 * @param toolTip The tool tip containing the validation message. Use null to not change the
	 *        existing tool tip.
	 */
	public void showValidationError(String toolTip) {
		this.validationToolTip = toolTip;
		this.validationError = true;
		this.updateWidgetState();
	}

	/**
	 * Clear any previously set validation error for this widget, updating the widget state in the
	 * process.
	 */
	public void clearValidationError() {
		this.validationToolTip = null;
		this.validationError = false;
		this.updateWidgetState();
	}

	/**
	 * Check if the widget has a validation error that is being displayed
	 * 
	 * @return true if the widget has a validation error, false otherwise
	 */
	public boolean hasValidationError() {
		return this.validationError;
	}

	/**
	 * Determine if this widget is read-only
	 * 
	 * @return true if read-only, false otherwise
	 */
	public abstract boolean isReadOnly();

	/**
	 * Sets whether this widget is read-only. If read-only, the value specified by the widget cannot
	 * be changed / edited.
	 * 
	 * @param readOnly true if read-only, false otherwise
	 */
	public abstract void setReadOnly(boolean readOnly);

	/**
	 * Determine if this widget is enabled
	 * 
	 * @return true if enabled, false otherwise
	 */
	public abstract boolean isEnabled();

	/**
	 * Sets whether the widget is enabled.
	 * 
	 * @param enabled true to enable, false otherwise
	 */
	public abstract void setEnabled(boolean enabled);

	/**
	 * Get the tool tip message for this widget in normal state
	 * 
	 * @return The tool tip message
	 */
	public String getToolTip() {
		return this.toolTip;
	}

	/**
	 * Set the tool tip message for this widget in normal state
	 * 
	 * @param toolTip The tool tip message
	 */
	public void setToolTip(String toolTip) {
		this.toolTip = toolTip;
		this.updateToolTip();
	}

	/**
	 * Get the error tool tip message for this widget when the value does not obey the data type
	 * 
	 * @return The data type error tool tip
	 */
	public String getErrorToolTip() {
		return this.errorToolTip;
	}

	/**
	 * Set the data type error tool tip message for this widget
	 * 
	 * @param errorToolTip The data type error tool tip message
	 */
	public void setErrorToolTip(String errorToolTip) {
		this.errorToolTip = errorToolTip;
		this.updateToolTip();
	}

	/**
	 * Determine if this widget is in error state. The widget is in error state if there is a
	 * validation error or if the value displayed is invalid based on the data type.
	 * 
	 * @return true if in error state, false otherwise
	 */
	protected boolean isInErrorState() {
		return ((!this.isValidValue()) || this.validationError);
	}

	/**
	 * Update the state of the widget since one of the widget error states have changed.
	 */
	protected void updateWidgetState() {
		updateStyle();
		updateToolTip();
	}

	/**
	 * Update the widget style (for error state style)
	 */
	protected abstract void updateStyle();

	protected void updateToolTip() {
		String currentToolTip = null;

		/* Check for validation error */
		if (this.validationError)
			currentToolTip = validationToolTip;

		/* Check if the value obeys the data type and the current tool tip is not set */
		if (!isValidValue() && isBlank(currentToolTip))
			currentToolTip = errorToolTip;

		/* Show default tool tip if no tool tip is set */
		if (isBlank(currentToolTip))
			currentToolTip = toolTip;

		this.setCurrentToolTip(currentToolTip);
	}

	/*
	 * Determine if the value is blank
	 */
	private boolean isBlank(String value) {
		return (value == null || (value.trim().length() == 0));
	}

	protected abstract void setCurrentToolTip(String toolTip);

	/**
	 * Get the current tool tip message displayed for the widget
	 */
	public abstract String getCurrentToolTip();
}
