/*
 * Created on Sep 14, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table;

import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATACELL;
import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATACELL_CHECKBOX;
import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATA_EVEN_ROW;
import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATA_ODD_ROW;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.client.ui.FlexTable.FlexCellFormatter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.ExpandingFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.CheckBox;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.ExpandCollapseButton;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;

/**
 * The Table Column Panel widget represents the table with header and body data.
 * 
 * @author Vijay Silva
 */
class TableColumnPanel<K, R extends Record<K>> extends Composite {

	/* Properties */
	private final DataProvider<K, R> dataProvider;
	private final FlexTableContainer table = new FlexTableContainer();
	private int colIndex;
	private List<R> records;
	private CheckBox<R> headerRowCheckbox;
	private List<CheckBox<R>> dataRowsCheckBoxes;
	private List<Column<?, R>> visibleColumns;
	private Map<Column<?, R>, List<Widget>> visibleColumnToWidgetListMap;
	private Map<Widget, R> widgetToRecordMap;
	private Map<Column<?, R>, Label> visibleColumnToHeaderLabel;
	private Map<Column<?, R>, HorizontalPanel> visibleColumnToHeaderColPanel;
	private ExpandingFeature<K,R> expandingFeature;
	private List<SelectionChangeHandler<K, R>> selectionChangeHandlers;
	
	/**
	 * Constructor
	 * 
	 * @param dataProvider The data provider
	 */
	public TableColumnPanel(DataProvider<K, R> dataProvider) {
		this( dataProvider, null );
	}
	
	/**
	 * Constructor
	 * 
	 * @param dataProvider The data provider
	 * @param expandingFeature The expanding feature
	 */
	public TableColumnPanel(DataProvider<K, R> dataProvider, ExpandingFeature<K,R> expandingFeature) {
		this.dataProvider = dataProvider;
		this.expandingFeature = expandingFeature;
		selectionChangeHandlers = new ArrayList<SelectionChangeHandler<K, R>>();
		initializeWidget();
	}

	// ========================================================================
	// ===================== WIDGET INITIALIZATION
	// ========================================================================

	/*
	 * Initialize the table widget
	 */
	private void initializeWidget() {
		/* Initialize the table */
		this.initWidget(table);

		/* Initialize the cache state */
		records = new ArrayList<R>();
		headerRowCheckbox = new CheckBox<R>();
		dataRowsCheckBoxes = new ArrayList<CheckBox<R>>();
		visibleColumns = new ArrayList<Column<?, R>>();
		visibleColumnToWidgetListMap = new HashMap<Column<?, R>, List<Widget>>();
		widgetToRecordMap = new HashMap<Widget, R>();
		visibleColumnToHeaderColPanel = new HashMap<Column<?, R>, HorizontalPanel>();
		visibleColumnToHeaderLabel = new HashMap<Column<?, R>, Label>();

		/* set style */
		setStylePrimaryName(TableConstants.STYLE_TABLE_COLUMN_PANEL);

		/* create header row */
		table.getRowFormatter().setStyleName(0, TableConstants.STYLE_HEADER_ROW);
		colIndex = 0;
		if (dataProvider.isCheckboxEnabled()) {
			headerRowCheckbox.addClickHandler(new ClickHandler() {
				public void onClick(ClickEvent event) {
					handleHeaderCheckBoxClick(event);
				}
			});

			table.getColumnFormatter().setWidth(colIndex, TableConstants.CHECK_BOX_COL_WIDTH);
			headerRowCheckbox.setStyleName(STYLE_DATACELL_CHECKBOX);  /* Added for ECMS  */
			table.setWidget(0, colIndex++, headerRowCheckbox);
		}
		if ( hasExpandingFeature() ) {
			table.setWidget(0, colIndex++, new Label("+") );
		}

		/* Create header row columns */
		for (final Column<?, R> column : dataProvider.getColumns()) {
			addColumn(column);
		}
	}

	/* Add a column to the header row */
	private void addColumn(final Column<?, R> column) {
		visibleColumns.add(column);
		HorizontalPanel headerColPanel = new HorizontalPanel();
		headerColPanel.setSpacing(3);
		headerColPanel.setVerticalAlignment(HorizontalPanel.ALIGN_MIDDLE);
		Label headerLabel = new Label(column.getDisplayName());
		headerColPanel.add(headerLabel);
		
		headerLabel.setStyleName(TableConstants.STYLE_HEADER_NONSORTABLE_COL_TEXT);
		if(column.getColumnWidth() != null) {
			table.getColumnFormatter().setWidth(colIndex, column.getColumnWidth());
		}
		table.setWidget(0, colIndex++, headerColPanel);
		visibleColumnToHeaderLabel.put(column, headerLabel);
		visibleColumnToHeaderColPanel.put(column, headerColPanel);
	}

	// ========================================================================
	// ===================== API
	// ========================================================================

	public void clearRecords() {
		records.clear();
		headerRowCheckbox.setValue(false);
		visibleColumnToWidgetListMap.clear();
		widgetToRecordMap.clear();
		dataRowsCheckBoxes.clear();

		// clean out existing rows
		final int rowCount = table.getRowCount();
		for (int i = rowCount - 1; i > 0; --i)
			table.removeRow(i);
	}

	public void updateRecords(List<R> result) {
		clearRecords();

		/* Add the records to render */
		if (result != null)
			records.addAll(result);
		
	
		int rowIndex = 1;
		for (final R record : records) {
			String styleName = (rowIndex % 2 == 0) ? STYLE_DATA_EVEN_ROW : STYLE_DATA_ODD_ROW;
			table.getRowFormatter().setStyleName(rowIndex, styleName);
			int colIndex = 0;
			if (dataProvider.isCheckboxEnabled()) {
				final CheckBox<R> dataRowCheckBox = new CheckBox<R>();
				//dataRowCheckBox.addStyleName(TableConstants.STYLE_DATA_CHECKBOX);
				dataRowCheckBox.setRecord(record);
				dataRowCheckBox.addClickHandler(new ClickHandler() {
					public void onClick(ClickEvent event) {
						handleDataCheckBoxClick(dataRowCheckBox, event);
					}
				});
				dataRowsCheckBoxes.add(dataRowCheckBox);
				table.setWidget(rowIndex, colIndex, dataRowCheckBox);
				/* table.getCellFormatter().addStyleName(rowIndex, colIndex, STYLE_DATACELL); */ /* Modified for ECMS  */
				table.getCellFormatter().addStyleName(rowIndex, colIndex, STYLE_DATACELL_CHECKBOX);
				colIndex++;
			}
			
			if ( hasExpandingFeature() ) {
				SimplePanel expandedPanel = new SimplePanel();
				expandedPanel.setVisible( false );
				final ExpandCollapseButton<K,R> expandCollapseButton = new ExpandCollapseButton<K,R>( expandedPanel, expandingFeature );
				expandCollapseButton.setRecord( record );
				expandCollapseButton.addClickHandler( new ClickHandler() {
					public void onClick(ClickEvent event) {
						handleExpandCollapseClick(expandCollapseButton, event);
					}
				});
				table.setWidget(rowIndex, colIndex, expandCollapseButton);
				/* table.getCellFormatter().addStyleName(rowIndex, colIndex, STYLE_DATACELL);*/ /* Modified for ECMS  */
				table.getCellFormatter().addStyleName(rowIndex, colIndex, STYLE_DATACELL_CHECKBOX);
				colIndex++;
				
				// Creating expanded row				
				table.setWidget( rowIndex + 1 , 1, expandedPanel );				
			}

			for (Column<?, R> field : visibleColumns) {
				table.getCellFormatter().addStyleName(rowIndex, colIndex, STYLE_DATACELL);
				String cellStyle = field.getCellStyle();
				if (cellStyle != null)
					table.getCellFormatter().addStyleName(rowIndex, colIndex, cellStyle);

				final View<?> listView = field.getView(record);
				String viewStyleName = listView.getStyleName();
				if (viewStyleName == null)
					viewStyleName = TableConstants.STYLE_DATA_TEXT;
				Widget widget = listView.getWidget();
				widget.setStyleName(viewStyleName);
				table.setWidget(rowIndex, colIndex++, widget);

				// add widget to list
				getInitializedWidgetList(field).add(widget);
				widgetToRecordMap.put(widget, record);
			}
			rowIndex++;
			if ( hasExpandingFeature() ) {
				((FlexCellFormatter) table.getCellFormatter()).setColSpan( rowIndex, 1, colIndex-1);
				rowIndex++;
			}
		}
	}

	/*
	 * Get the list of widgets for a column, or initialize if the list is not present
	 */
	private List<Widget> getInitializedWidgetList(Column<?, R> visibleColumn) {
		List<Widget> list = visibleColumnToWidgetListMap.get(visibleColumn);
		if (list == null) {
			list = new ArrayList<Widget>();
			visibleColumnToWidgetListMap.put(visibleColumn, list);
		}
		return list;
	}

	public List<Column<?, R>> listVisibleColumns() {
		return visibleColumns;
	}

	public List<Widget> listWidgetsForVisibleColumn(Column<?, R> column) {
		return visibleColumnToWidgetListMap.get(column);
	}

	public R getRecordForWidget(Widget widget) {
		return widgetToRecordMap.get(widget);
	}

	public List<R> getRecords() {
		List<R> visibleRecords = new ArrayList<R>();
		visibleRecords.addAll(this.records);
		return visibleRecords;
	}

	public List<R> listSelectedRecords() {
		List<R> records = new ArrayList<R>();
		for (CheckBox<R> checkBox : dataRowsCheckBoxes)
			if (checkBox.getValue())
				records.add(checkBox.getRecord());
		return records;
	}

	public int getDataRowCount() {
		return records.size();
	}
	
	/**
	 * Add a listener for selection change events triggered by the table
	 * 
	 * @param handler the event handler
	 */
	public void addSelectionChangeHandler(SelectionChangeHandler<K,R> handler) {
		if ( ( handler != null ) && ( !this.selectionChangeHandlers.contains( handler ) )) {
			selectionChangeHandlers.add( handler );
		}
	}

	/**
	 * Remove a listener for selection change events triggered by the table
	 * 
	 * @param handler the event handler
	 */
	public void removeSelectionChangeHandler(SelectionChangeHandler<K,R> handler) {
		selectionChangeHandlers.remove(handler);
	}
	
	

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/* Handle the click event on the header check box */
	private void handleHeaderCheckBoxClick(ClickEvent event) {
		/* selecting all data row check boxes when header row check box is selected */
		Boolean headerValue = headerRowCheckbox.getValue();
		for (CheckBox<R> dataRowCB : dataRowsCheckBoxes) {
			dataRowCB.setValue(headerValue);
			changeChildrenSelection( dataRowCB );
		}
		notifySelectionChangeHandlers();
	}

	/* Handle the click event on the data row check box */
	private void handleDataCheckBoxClick(CheckBox<R> source, ClickEvent event) {
		if (headerRowCheckbox.getValue()) {
			if (!source.getValue()) {
				headerRowCheckbox.setValue(false);				
			}
		}
		changeChildrenSelection( source );	
		notifySelectionChangeHandlers();
	}
	
	private void notifySelectionChangeHandlers() {
		for ( SelectionChangeHandler<K,R> handler : selectionChangeHandlers ) {
			handler.onSelectionChange( new SelectionChangeEvent<K, R>( this, listSelectedRecords()) );
		}
	}
	
	private void changeChildrenSelection(CheckBox<R> dataRowCB) {
		if ( hasExpandingFeature() ) {
			expandingFeature.getExpandingProvider().parentSelectionChanged( dataRowCB.getRecord(), dataRowCB.getValue() );
		}		
	}

	/* Handle the click event on the expand/collapse button */
	private void handleExpandCollapseClick(ExpandCollapseButton<K,R> source, ClickEvent event) {
		source.changeState();
	}

	// ========================================================================
	// ===================== ACCESSORS
	// ========================================================================

	/**
	 * Get the header column panel for the column
	 */
	public HorizontalPanel getHeaderColumnPanel(Column<?, R> column) {
		return visibleColumnToHeaderColPanel.get(column);
	}

	/**
	 * Get the header column label for the column
	 * 
	 * @return header column label
	 */
	public Label getHeaderColumnLabel(Column<?, R> column) {
		return visibleColumnToHeaderLabel.get(column);
	}

	/**
	 * Get the header row check box
	 * 
	 * @return The header row check box
	 */
	public CheckBox<R> getHeaderRowCheckBox() {
		return headerRowCheckbox;
	}
	
	/**
	 * Get the data rows checkboxes
	 * 
	 * @return The data rows checkboxes
	 */
	public List<CheckBox<R>> getDataRowsCheckBoxes() {
		return dataRowsCheckBoxes;
	}

	/**
	 * Set the expanding feature
	 * 
	 * @param expandingFeature
	 */
	public void setExpandingFeature(ExpandingFeature<K, R> expandingFeature) {
		this.expandingFeature = expandingFeature;
	}

	/**
	 * Get the expanding feature
	 */
	public ExpandingFeature<K, R> getExpandingFeature() {
		return expandingFeature;
	}
	
	/**
	 * Determine if the expanding feature is installed
	 * 
	 * @return true if the expanding feature is installed, false otherwise
	 */
	public boolean hasExpandingFeature() {
		return expandingFeature != null;
	}
}
