/**
 * 
 */
package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl;

import java.util.HashSet;
import java.util.Set;

import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;

public class SortFeatureImpl<K, R extends Record<K>> implements SortFeature<K, R> {

	/* Properties */
	private final Column<?, R> defaultSortColumn;
	private final SortOrder defaultSortOrder;
	private Column<?, R> sortColumn;
	private SortOrder sortOrder = SortOrder.ASCENDING;
	private final Set<Column<?, R>> sortableColumns;

	public SortFeatureImpl(Column<?, R> defaultSortColumn, SortOrder defaultSortOrder) {
		if (defaultSortOrder == null)
			defaultSortOrder = SortOrder.ASCENDING;

		/* Set the default values */
		this.defaultSortColumn = defaultSortColumn;
		this.defaultSortOrder = defaultSortOrder;

		/* Create the set of sortable columns, add the default column (if not null) */
		sortableColumns = new HashSet<Column<?, R>>();
		if (this.defaultSortColumn != null)
			this.sortableColumns.add(this.defaultSortColumn);
		reset();
	}

	@Override
	public boolean isColumnSortable(Column<?, R> column) {
		return (column != null && this.sortableColumns.contains(column));
	}

	@Override
	public Set<Column<?, R>> getSortableColumns() {
		return sortableColumns;
	}

	/**
	 * Add a new column to the list of sortable columns
	 * 
	 * @param column The column to add
	 */
	public void addSortableColumn(Column<?, R> column) {
		sortableColumns.add(column);
	}

	@Override
	public SortOrder getDefaultSortOrder() {
		return this.defaultSortOrder;
	}

	@Override
	public Column<?, R> getDefaultSortColumn() {
		return (isColumnSortable(this.defaultSortColumn)) ? this.defaultSortColumn : null;
	}

	@Override
	public SortOrder getSortOrder() {
		return sortOrder;
	}

	@Override
	public void setSortOrder(SortOrder sortOrder) {
		if (sortOrder == null)
			sortOrder = getDefaultSortOrder();

		this.sortOrder = sortOrder;
	}

	@Override
	public Column<?, R> getSortColumn() {
		return sortColumn;
	}

	@Override
	public void setSortColumn(Column<?, R> column) {
		if (!isColumnSortable(column))
			column = getDefaultSortColumn();

		this.sortColumn = column;
	}

	/**
	 * Determine if the sort order is ascending
	 * 
	 * @return true if ascending, false otherwise
	 */
	public boolean isAscending() {
		return SortOrder.ASCENDING.equals(getSortOrder());
	}

	/**
	 * Get the name of the column to sort
	 * 
	 * @return the sort column name, or null if no column is set
	 */
	public String getSortColumnName() {
		Column<?, R> column = getSortColumn();
		return (column != null) ? column.getName() : null;
	}

	/* Reset the state to the default values */
	@Override
	public void reset() {
		setSortOrder(getDefaultSortOrder());
		setSortColumn(getDefaultSortColumn());
	}
}
