/**
 * File Name	:	ResizableContainer.java
 * Created By	:	Sanjay Khattar (sanjay.khattar@ericsson.com)
 * Date Created	:	Aug 27, 2012 1:01:15 PM
 * Purpose		:	
 *
 * (c) Ericsson Television Inc. 
 */
package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * A <code>ResizableContainer</code> is a re-sizable composite widget with a {@link MessageArea} 
 * on top of a Tabs container {@link TabContainer}. Multiple tabs can be added to the
 * Tab container. Each tab has a {@link Widget} <code>content</code> within a
 * {@link ScrollContainer} and a {@link HorizontalContainer} at the bottom, 
 * to which any number of widgets can be added. Typically, the latter will be used 
 * as a Buttons Panel to perform operations on the <code>content</code>.
 * <p>
 * The <code>ResizableContainer</code> also supports independently scrollable 
 * left and right contents in the tab.
 * <p>
 * The <code>ResizableContainer</code> resizes itself and it's children upon 
 * resizing of the container browser Window. When the size of the 
 * {@link MessageArea} or the bottom {@link HorizontalContainer} changes, 
 * because of messages or widgets becoming visible/invisible, the 
 * {@link ScrollContainer} size changes to accommodate the change keeping
 * the bottom panel fully visible and pinned at the bottom of the browser 
 * window.  
 * 
 * @author esakhat (sanjay.khattar@ericsson.com)
 *
 */
public class ResizableContainer extends Composite implements IResizable, IMessageListener {

	public static final String ANCHOR_TAB_PORTION_PARAMETER_NAME = "tab";
	
	protected static final int CMS_HEADER_MENU_SIZE_PX = NeptuneApplication.CMS_HEADER_HEIGHT_PX;
	protected static final int BOTTOM_OFFSET_PX = 20;
	protected static final int HORIZONTAL_MARGIN_PX = 40;
	protected static final int RESIZABLE_CONTAINER_MIN_WIDTH = 975;
	
	protected static final String STYLE_NAME_MAIN_CONTAINER = "nwt-reszc";
	protected static final String STYLE_NAME_MESSAGE_AREA = "nwt-reszc-messageArea";
	protected static final String STYLE_NAME_TAB_CONTAINER = "nwt-reszc-tabContainer";
	protected static final String STYLE_NAME_SCROLL_CONTAINER = "nwt-reszc-scrollContainer";
	protected static final String STYLE_NAME_BUTTONS_CONTAINER = "nwt-reszc-buttonsContainer";
	
	protected static final String STYLE_NAME_HORIZONTAL_CONTAINER = "nwt-reszc-horizontalContainer";
	protected static final String STYLE_NAME_LEFT_SCROLL_CONTAINER = "nwt-reszc-leftScrollContainer";
	protected static final String STYLE_NAME_RIGHT_SCROLL_CONTAINER = "nwt-reszc-rightScrollContainer";
	
	protected VerticalContainer mainContainer;
	protected MessageArea messageArea;
	protected TabContainer tabContainer;
	protected Widget content;
	protected ScrollContainer nonTabScrollContainer;
	protected HorizontalContainer nonTabButtonsContainer;
	
	protected HandlerRegistration windowRegistration = null;

	protected String tabTitle;
	protected String anchorPrefix;
	protected int offsetHeight = 0;
	protected int offsetWidth = 0;
	protected boolean hasTabs = false;
	
	protected List<ResizableTab> resizableTabs  = new ArrayList<ResizableTab>();
	protected int selectedTabIndex = 0;
	
	protected List<Widget> heightOffsettingWidgets = new ArrayList<Widget>();
	protected List<Widget> widthOffsettingWidgets = new ArrayList<Widget>();

	/**
	 * Create a <code>ResizableContainer</code> given a {@link Widget} 
	 * <code>content</code>, a <code>tabTitle</code>, and an 
	 * <code>anchorPrefix</code>.
	 * <p>
	 * If <code>null</code> or empty <code>tabTitle</code> is provided, 
	 * a non-tabbed container is created, which is re-sizable just like 
	 * the tabbed one.
	 * <p>
	 * This constructor lets the <code>ResizableContainer</code> use it's 
	 * own internal {@link MessageArea} rather than requiring the client to 
	 * provide it.
	 * 
	 * @param content
	 * @param tabTitle
	 * @param anchorPrefix
	 */
	public ResizableContainer(Widget content, String tabTitle, String anchorPrefix) {	
		this(content, tabTitle, anchorPrefix, null);
	}
	
	/**
	 * Create a <code>ResizableContainer</code> given a {@link Widget} 
	 * <code>content</code>, a <code>tabTitle</code>, an 
	 * <code>anchorPrefix</code> and a {@link MessageArea}.
	 * <p>
	 * If <code>null</code> or empty <code>tabTitle</code> is provided, 
	 * a non-tabbed container is created, which is re-sizable just like 
	 * the tabbed one.
	 * <p>
	 * This constructor lets the <code>ResizableContainer</code> use a  
	 * client provided {@link MessageArea} rather than the default one.  
	 * 
	 * @param content
	 * @param tabTitle
	 * @param anchorPrefix
	 * @param extMessageArea
	 */
	public ResizableContainer(Widget content, String tabTitle, 
			String anchorPrefix, MessageArea extMessageArea) {	
		this.content = content;
		this.tabTitle = tabTitle;
		this.anchorPrefix = anchorPrefix;
		this.messageArea = extMessageArea;
		
		if (content == null) {
			throw new RuntimeException("Widget content cannot be null");
		}
		
		if (content instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) content;
			messageEmitter.registerMessageListener(this);
		}

		if (tabTitle != null) {
			hasTabs = true;
		}
		
		mainContainer = new VerticalContainer();
		if (messageArea == null) {
			messageArea = new MessageArea();
		}
		mainContainer.add(messageArea);
		
		mainContainer.addStyleName(STYLE_NAME_MAIN_CONTAINER);
		messageArea.addStyleName(STYLE_NAME_MESSAGE_AREA);
		
		if (hasTabs) {
			tabContainer = new TabContainer();
			mainContainer.add(tabContainer);
			
			tabContainer.addStyleName(STYLE_NAME_TAB_CONTAINER);

			tabContainer.addSelectionHandler(new SelectionHandler<Integer>() {
				@Override
				public void onSelection(SelectionEvent<Integer> event) {
					selectedTabIndex = event.getSelectedItem();
					for (int i = 0; i < resizableTabs.size(); ++i) {
						resizableTabs.get(i).setSelected(
								i == event.getSelectedItem());
					}
				}
			});
			
			addTab(content, tabTitle);
			tabContainer.selectTab(0);
		}
		else {
			nonTabScrollContainer = new ScrollContainer();
			nonTabScrollContainer.setWidget(content);
			mainContainer.add(nonTabScrollContainer);
			
			nonTabButtonsContainer = new HorizontalContainer();
			mainContainer.add(nonTabButtonsContainer);
			
			nonTabButtonsContainer.addStyleName(STYLE_NAME_BUTTONS_CONTAINER);
		}
		
		initWidget(mainContainer);
		
		init();
		
	}
	
	/**
	 * Create a <code>ResizableContainer</code> given a {@link Widget} 
	 * <code>leftContent</code>, a {@link Widget} <code>rightContent</code>, 
	 *  a <code>tabTitle</code>, an 
	 * <code>anchorPrefix</code> and a {@link MessageArea}.
	 * <p>
	 * If <code>null</code> or empty <code>tabTitle</code> is provided, 
	 * a non-tabbed container is created, which is re-sizable just like 
	 * the tabbed one.
	 * <p>
	 * This constructor lets the <code>ResizableContainer</code> use it's 
	 * own internal {@link MessageArea} rather than requiring the client to 
	 * provide it.
	 * 
	 * @param leftContent
	 * @param rightContent
	 * @param tabTitle
	 * @param anchorPrefix
	 */
	public ResizableContainer(Widget leftContent, Widget rightContent, String tabTitle, 
			String anchorPrefix) {
		this(leftContent, rightContent, tabTitle, anchorPrefix, null);
	}
	
	/**
	 * Create a <code>ResizableContainer</code> given a {@link Widget} 
	 * <code>leftContent</code>, a {@link Widget} <code>rightContent</code>, 
	 *  a <code>tabTitle</code>, an 
	 * <code>anchorPrefix</code> and a {@link MessageArea}.
	 * <p>
	 * If <code>null</code> or empty <code>tabTitle</code> is provided, 
	 * a non-tabbed container is created, which is re-sizable just like 
	 * the tabbed one.
	 * <p>
	 * This constructor lets the <code>ResizableContainer</code> use a  
	 * client provided {@link MessageArea} rather than the default one.  
	 * 
	 * @param leftContent
	 * @param rightContent
	 * @param tabTitle
	 * @param anchorPrefix
	 * @param extMessageArea
	 */
	public ResizableContainer(Widget leftContent, Widget rightContent, String tabTitle, 
			String anchorPrefix, MessageArea extMessageArea) {	
		this.tabTitle = tabTitle;
		this.anchorPrefix = anchorPrefix;
		this.messageArea = extMessageArea;
		
		if (leftContent == null) {
			throw new RuntimeException("Widget content cannot be null");
		}
		
		if (rightContent == null) {
			throw new RuntimeException("Widget content cannot be null");
		}
		
		if (tabTitle != null) {
			hasTabs = true;
		}
		
		if (leftContent instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) leftContent;
			messageEmitter.registerMessageListener(this);
		}

		if (rightContent instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) rightContent;
			messageEmitter.registerMessageListener(this);
		}

		mainContainer = new VerticalContainer();
		if (messageArea == null) {
			messageArea = new MessageArea();
		}
		mainContainer.add(messageArea);
		
		mainContainer.addStyleName(STYLE_NAME_MAIN_CONTAINER);
		messageArea.addStyleName(STYLE_NAME_MESSAGE_AREA);
		
		if (hasTabs) {
			tabContainer = new TabContainer();
			mainContainer.add(tabContainer);
			
			tabContainer.addStyleName(STYLE_NAME_TAB_CONTAINER);

			tabContainer.addSelectionHandler(new SelectionHandler<Integer>() {
				@Override
				public void onSelection(SelectionEvent<Integer> event) {
					selectedTabIndex = event.getSelectedItem();
					for (int i = 0; i < resizableTabs.size(); ++i) {
						resizableTabs.get(i).setSelected(
								i == event.getSelectedItem());
					}
				}
			});
			
			addTab(leftContent, rightContent, tabTitle);
			tabContainer.selectTab(0);
		}
		else {
			HorizontalContainer nonTabHorizontalContainer = new HorizontalContainer();
			ScrollContainer nonTabLeftScrollContainer = new ScrollContainer();
			ScrollContainer nonTabRightScrollContainer = new ScrollContainer();
			nonTabLeftScrollContainer.setWidget(leftContent);
			nonTabRightScrollContainer.setWidget(rightContent);
			nonTabHorizontalContainer.add(nonTabLeftScrollContainer);
			nonTabHorizontalContainer.add(nonTabRightScrollContainer);
			mainContainer.add(nonTabHorizontalContainer);
			
			nonTabButtonsContainer = new HorizontalContainer();
			mainContainer.add(nonTabButtonsContainer);
			
			nonTabButtonsContainer.addStyleName(STYLE_NAME_BUTTONS_CONTAINER);
		}
		
		initWidget(mainContainer);
		
		init();
		
	}
	
	/**
	 * If this <code>ResizableContainer</code> has tabs, returns the 
	 * {@link TabContainer} that contains those tabs, else returns 
	 * <code>null</code>.
	 *  
	 * @return the tabContainer
	 */
	public TabContainer getTabContainer() {
		return tabContainer;
	}

	/**
	 * If this <code>ResizableContainer</code> does not have tabs, returns the 
	 * {@link HorizontalContainer} that contains the buttons, else returns 
	 * <code>null</code>.
	 * 
	 * @return the nonTabButtonsContainer
	 */
	public HorizontalContainer getNonTabButtonsContainer() {
		return nonTabButtonsContainer;
	}

	/**
	 * @return the messageArea
	 */
	public MessageArea getMessageArea() {
		return messageArea;
	}

	/**
	 * @return the content
	 */
	public Widget getContent() {
		return content;
	}

	/**
	 * @return the anchorPrefix
	 */
	public String getAnchorPrefix() {
		return anchorPrefix;
	}

	/**
	 * @param anchorPrefix the anchorPrefix to set
	 */
	public void setAnchorPrefix(String anchorPrefix) {
		this.anchorPrefix = anchorPrefix;
	}

	/**
	 * @return the offsetHeight
	 */
	public int getOffsetHeight() {
		return offsetHeight;
	}

	/**
	 * @param offsetHeight the offsetHeight to set
	 */
	public void setOffsetHeight(int offsetHeight) {
		this.offsetHeight = offsetHeight;
	}

	/**
	 * Add a new Tab to the re-sizable container. 
	 * <p>
	 * Both <code>content</code> and <code>tabTitle</code> are required. 
	 * <code>tabTitle</code> cannot in addition be <code>null</code>.
	 * <p>
	 * If the Widget <code>content</code> is already in one of the tabs, 
	 * it will be moved to the right-most index.
	 * 
	 * @param content
	 * @param tabTitle
	 */
	public void addTab(Widget content, String tabTitle) {
		
		if (content == null) {
			throw new RuntimeException("Widget content cannot be null");
		}
		
		if (content instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) content;
			messageEmitter.registerMessageListener(this);
		}

		if (tabTitle == null) {
			throw new RuntimeException("String tabTitle cannot be null");
		}

		if (hasTabs) {
			ResizableTab resizableTab = getResizableTab(tabTitle);
			if (resizableTab == null) {
				resizableTab = new ResizableTab(content, tabTitle);
			}
			addResizableTab(resizableTab);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * Add a new Tab to the re-sizable container with the given 
	 * <code>leftContent</code> and <code>rightContent</code> being
	 * independently scrollable. 
	 * <p>
	 * <code>leftContent</code>, <code>rightContent</code> and 
	 * <code>tabTitle</code> are required. 
	 * <code>tabTitle</code> should in addition have at least one non-space character.
	 * 
	 * @param leftContent
	 * @param rightContent
	 * @param tabTitle
	 */
	public void addTab(Widget leftContent, Widget rightContent, String tabTitle) {
		
		if (leftContent == null) {
			throw new RuntimeException("Left Content Widget cannot be null");
		}
		
		if (rightContent == null) {
			throw new RuntimeException("Right Content Widget cannot be null");
		}
		
		if (leftContent instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) leftContent;
			messageEmitter.registerMessageListener(this);
		}

		if (rightContent instanceof IMessageEmitter) {
			IMessageEmitter messageEmitter = (IMessageEmitter) rightContent;
			messageEmitter.registerMessageListener(this);
		}

		if (tabTitle == null) {
			throw new RuntimeException("String tabTitle cannot be null");
		}
		
		if (hasTabs) {
			ResizableTab resizableTab = getResizableTab(tabTitle);
			if (resizableTab == null) {
				resizableTab = new DualPanelResizableTab(leftContent, rightContent, tabTitle);
			}
			addResizableTab(resizableTab);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * A convenience method that adds a button to the bottom container.
	 * <p>
	 * The added buttons are placed in same order in which they are added.
	 * 
	 * @param tabTitle
	 * @param buttonWidget
	 * 
	 * @see #addBottomWidget(String, Widget)
	 */
	public void addButton(String tabTitle, ButtonWidget buttonWidget) {
		addBottomWidget(tabTitle, buttonWidget);
	}
	
	/**
	 * Adds a widget to the bottom container.
	 * <p>
	 * The added widgets are placed in same order in which they are added.
	 * 
	 * @param tabTitle
	 * @param Widget
	 */
	public void addBottomWidget(String tabTitle, Widget Widget) {
		if (hasTabs) {
			ResizableTab resizableTab = getResizableTab(tabTitle);
			if (resizableTab != null) {
				resizableTab.buttonsContainer.add(Widget);
			} else {
				throw new RuntimeException("No tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			nonTabButtonsContainer.add(Widget);
		}
	}
	
	/**
	 * Select the tab with the given <code>tabIndex</code>.
	 * 
	 * @param tabIndex
	 */
	public void selectTab(int tabIndex) {
		if (hasTabs) {
			if ((tabIndex >= 0) && (tabIndex < resizableTabs.size())) {
				tabContainer.selectTab(tabIndex);
			} else {
				throw new RuntimeException("No tab with tabIndex " + tabIndex
						+ " exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * Select the tab with the given <code>tabTitle</code>.
	 * 
	 * @param tabTitle
	 */
	public void selectTab(String tabTitle) {
		if (hasTabs) {
			int tabIndex = getTabIndex(tabTitle);
			if (tabIndex >= 0) {
				tabContainer.selectTab(tabIndex);
			} else {
				throw new RuntimeException("No tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * If this <code>ResizableContainer</code> has tabs, returns the 
	 * {@link TabContainer} that contains those tabs, else throws a  
	 * {@link RuntimeException}.
	 * 
	 * @param tabTitle
	 * @return
	 */
	public HorizontalContainer getButtonsContainer(String tabTitle) {
		if (hasTabs) {
			ResizableTab resizableTab = getResizableTab(tabTitle);
			if (resizableTab != null) {
				return resizableTab.buttonsContainer;
			} else {
				throw new RuntimeException("No tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	@Override
	public void updateSize() {
		if (hasTabs) {
			resizableTabs.get(selectedTabIndex).updateScrollPanelSize();
		}
		else {
			updateNonTabScrollContainerSize();
		}
	}
	
	/**
	 * Add the given widget's height to offsetHeight while calculating scrollContainer height.
	 * 
	 * @param w widget to be added when calculating offset height
	 */
	public void addHeightOffsettingWidget(Widget w) {
		if (w != null) {
			heightOffsettingWidgets.add(w);
		}
	}

	/**
	 * Add the given widget's height to offsetHeight while calculating scrollContainer height.
	 * 
	 * @param w widget to be added when calculating offset height
	 */
	public void addWidthOffsettingWidget(Widget w) {
		if (w != null) {
			widthOffsettingWidgets.add(w);
		}
	}

	/**
	 * @return the leftContentWidthPercentage
	 */
	public int getLeftContentWidthPercentage(String tabTitle) {
		int leftContentWidthPercentage = 0;
		ResizableTab resizableTab = getResizableTab(tabTitle);
		if (resizableTab != null) {
			if (resizableTab instanceof DualPanelResizableTab) {
				DualPanelResizableTab dualPanelResizableTab = (DualPanelResizableTab) resizableTab;
				leftContentWidthPercentage = dualPanelResizableTab.getLeftContentWidthPercentage();
			}
		}
		else {
			throw new RuntimeException("No tab with tabTitle " + tabTitle + " exists.");
		}
		
		return leftContentWidthPercentage;
	}

	/**
	 * @param leftContentWidthPercentage the leftContentWidthPercentage to set
	 */
	public void setLeftContentWidthPercentage(String tabTitle, int leftContentWidthPercentage) {
		ResizableTab resizableTab = getResizableTab(tabTitle);
		if (resizableTab != null) {
			if (resizableTab instanceof DualPanelResizableTab) {
				DualPanelResizableTab dualPanelResizableTab = (DualPanelResizableTab) resizableTab;
				dualPanelResizableTab.setLeftContentWidthPercentage(leftContentWidthPercentage);
			}
		}
		else {
			throw new RuntimeException("No tab with tabTitle " + tabTitle + " exists.");
		}
		
	}

	@Override
	public void setInfoMessage(String infoMessage) {
		messageArea.setInfoMessage(infoMessage);
		updateSize();
	}

	@Override
	public void setErrorMessage(String errorMessage) {
		messageArea.setErrorMessage(errorMessage);
		updateSize();
	}

	@Override
	public void setErrorHTML(String errorHTML) {
		messageArea.setErrorHTML(errorHTML);
		updateSize();
	}
	
	@Override
	public void reset() {
		messageArea.reset();
		updateSize();
	}

	/**
	 * Add a {@link SelectionHandler} to this tabContainer;
	 * 
	 * @param handler
	 */
	public void addSelectionHandler(SelectionHandler<Integer> handler) {
		if (hasTabs) {
			tabContainer.addSelectionHandler(handler);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}

	/**
	 * Enable/disable the tab identified by the given <code>tabTitle</code>.
	 * 
	 * @param tabTitle
	 * @param enabled
	 */
	public void setTabEnabled(String tabTitle, boolean enabled) {
		if (hasTabs) {
			tabContainer.getTabBar().setTabEnabled(getTabIndex(tabTitle), enabled);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * Removes the tab at the specified index.
	 * <p>
	 * Selects the first tab after successful removal.
	 * 
	 * @param index
	 * @return
	 */
	public boolean removeTab(int index) {
		boolean removeSuccessful = false;
		if (hasTabs) {
			int size = resizableTabs.size();
			if ((index >= 0) && (index < size)) {
				resizableTabs.remove(index);
				removeSuccessful = tabContainer.remove(index);
				if (removeSuccessful) {
					tabContainer.selectTab(0);
				}
			} else {
				throw new RuntimeException("No tab with index " + index 
						+ " exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
		
		return removeSuccessful;
	}
	
	

	/**
	 * Removes the tab with the given <code>tabTitle</code>.
	 * 
	 * @param tabTitle
	 * @return
	 */
	public boolean removeTab(String tabTitle) {
		boolean removeSuccessful = false;
		if (hasTabs) {
			int tabIndex = getTabIndex(tabTitle);
			if (tabIndex >= 0) {
				removeSuccessful = removeTab(tabIndex);
			} else {
				throw new RuntimeException("No tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
		
		return removeSuccessful;
	}
	
	/**
	 * Renames tab at the given <code>tabIndex</code> to the <code>newTabTitle</code>.
	 * 
	 * @param tabIndex
	 * @param newTabTitle
	 */
	public void renameTab(int tabIndex, String newTabTitle) {
		if (hasTabs) {
			int size = resizableTabs.size();
			if ((tabIndex >= 0) && (tabIndex < size)) {
				ResizableTab resizableTab = resizableTabs.remove(tabIndex);
				tabContainer.remove(tabIndex);
				
				resizableTab.setTabTitle(newTabTitle);
				resizableTabs.add(tabIndex, resizableTab);
				tabContainer.insert(resizableTab.getVerticalContainer(),
						resizableTab.getTabTitle(), tabIndex);
			} else {
				throw new RuntimeException("No tab with index " + tabIndex 
						+ " exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	/**
	 * Renames tab with the given <code>oldTabTitle</code> to the <code>newTabTitle</code>.
	 * 
	 * @param oldTabTitle
	 * @param newTabTitle
	 */
	public void renameTab(String oldTabTitle, String newTabTitle) {
		if (hasTabs) {
			int tabIndex = getTabIndex(oldTabTitle);
			if (tabIndex >= 0) {
				renameTab(tabIndex, newTabTitle);
			} else {
				throw new RuntimeException("Cannot rename because no tab with tabTitle '" + tabTitle
						+ "' exists");
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}

	/**
	 * Initialize this <code>ResizableContainer</code>.
	 * <p>
	 * This resets the <code>ResizableContainer</code>'s 
	 * {@link MessageArea} and updates the size of <code>ResizableContainer</code>.
	 * 
	 */
	protected void init() {
		messageArea.reset();
		
		Scheduler.get().scheduleFixedDelay(new RepeatingCommand() {

			@Override
			public boolean execute() {
				updateSize();
				return false;
			}
			
		}, 500);		
	}
	
	/**
	 * If the given <code>resizableTab</code> is already in one of the tabs, 
	 * it will be moved to the right-most index.
	 * <p>
	 * Selects the first tab after adding.
	 * 
	 * @param resizableTab
	 */
	protected void addResizableTab(ResizableTab resizableTab) {
		if (hasTabs) {
			if (!resizableTabs.contains(resizableTab)) {
				resizableTabs.add(resizableTab);
			}
			else {
				int origIndex = resizableTabs.indexOf(resizableTab);
				int size = resizableTabs.size();
				if (origIndex >= 0) {
					int finalIndex = size - 1;
					ResizableTab preExistingResizableTab = resizableTabs.remove(origIndex);
					resizableTabs.add(finalIndex, preExistingResizableTab);
				}
			}
			tabContainer.add(resizableTab.getVerticalContainer(),
					resizableTab.getTabTitle());
			tabContainer.selectTab(0);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
	}
	
	protected ResizableTab getResizableTab(int tabIndex) {
		ResizableTab resizableTab = null;
		
		if (hasTabs) {
			if (tabIndex >= 0) {
				resizableTab = resizableTabs.get(tabIndex);
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}

		return resizableTab;
	}
	
	/**
	 * 
	 * @param tabTitle
	 * @return
	 */
	protected ResizableTab getResizableTab(String tabTitle) {
		ResizableTab resizableTab = null;
		
		if (hasTabs) {
			int tabIndex = getTabIndex(tabTitle);
			resizableTab = getResizableTab(tabIndex);
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}

		return resizableTab;
	}
	
	/**
	 * 
	 * @param tabTitle
	 * @return
	 */
	public int getTabIndex(String tabTitle) {
		int tabIndex = -1;
		
		if (hasTabs) {
			if ((tabTitle != null) && !tabTitle.trim().isEmpty()) {
				for (int i = 0; i < resizableTabs.size(); i++) {
					ResizableTab resizableTab = resizableTabs.get(i);
					String resizableTabTitle = resizableTab.getTabTitle();
					if (tabTitle.equals(resizableTabTitle)) {
						tabIndex = i;
						break;
					}
				}
			}
		}
		else {
			throw new RuntimeException("This ResizableContainer does not have tabs");
		}
		
		return tabIndex;
	}

	/**
	 * 
	 */
	protected void updateNonTabScrollContainerSize() {
		int height = Window.getClientHeight()
				- CMS_HEADER_MENU_SIZE_PX
				- offsetHeight
				- messageArea.getOffsetHeight()
				- nonTabButtonsContainer.getOffsetHeight()
				- BOTTOM_OFFSET_PX;
			
			for (Widget w : heightOffsettingWidgets) {
				height -= w.getOffsetHeight(); 
			}
			
			nonTabScrollContainer.setHeight(height + "px");
	}
	
	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;

		super.onUnload();
	}
	
	/**
	 * Inner class which contains the widgets comprising a 
	 * re-sizable tab.
	 * 
	 * @author esakhat (sanjay.khattar@ericsson.com)
	 *
	 */
	protected class ResizableTab {
		
		protected String tabTitle;
		protected Widget tabContent;
		
		protected VerticalContainer verticalContainer;
		protected ScrollContainer scrollContainer;
		protected HorizontalContainer buttonsContainer;
		
		/**
		 * Create a <code>ResizableTab</code> given the {@link Widget} 
		 * <code>content</code> and <code>String tabTitle</code>.
		 * <p>
		 * If either <code>content</code> is <code>null</code> or 
		 * <code>tabTitle</code> is <code>null</code> or empty, a 
		 * <code>ResizableTab</code> cannot be created, and a 
		 * {@link NullPointerException} is thrown.  
		 * 
		 * @param tabContent
		 * @param tabTitle
		 */
		public ResizableTab(Widget tabContent, String tabTitle) {
			this.tabContent = tabContent;
			this.tabTitle = tabTitle;
			
			if (tabContent == null)  {
				throw new NullPointerException("Widget tabContent cannot be null");
			}
			
			if ((tabTitle == null) || (tabTitle.trim().isEmpty())) {
				throw new NullPointerException("String tabTitle cannot be null or empty");
			}
			
			verticalContainer = new VerticalContainer();
			scrollContainer = new ScrollContainer();
			buttonsContainer = new HorizontalContainer();
			
			verticalContainer.add(scrollContainer);
			verticalContainer.add(buttonsContainer);
			
			scrollContainer.setWidget(tabContent);

			scrollContainer.addStyleName(STYLE_NAME_SCROLL_CONTAINER);
			buttonsContainer.addStyleName(STYLE_NAME_BUTTONS_CONTAINER);
		}

		/**
		 * @return the tabTitle
		 */
		public String getTabTitle() {
			return tabTitle;
		}

		/**
		 * @param tabTitle the tabTitle to set
		 */
		public void setTabTitle(String tabTitle) {
			this.tabTitle = tabTitle;
		}

		/**
		 * @return the tabContent
		 */
		public Widget getTabContent() {
			return tabContent;
		}

		/**
		 * @param tabContent the tabContent to set
		 */
		public void setTabContent(Widget tabContent) {
			this.tabContent = tabContent;
		}		

		/**
		 * @return the verticalContainer
		 */
		public VerticalContainer getVerticalContainer() {
			return verticalContainer;
		}
		
		/**
		 * 
		 * @param selected
		 */
		public void setSelected(boolean selected) {
			if (selected) {
				messageArea.reset();
				updateScrollPanelSize();
			}
		}
		
		/**
		 * 
		 */
		protected void updateScrollPanelSize() {
			int height = Window.getClientHeight()
				- CMS_HEADER_MENU_SIZE_PX
				- offsetHeight
				- messageArea.getOffsetHeight()
				- tabContainer.getTabBar().getOffsetHeight()
				- buttonsContainer.getOffsetHeight()
				- BOTTOM_OFFSET_PX;
			
			for (Widget w : heightOffsettingWidgets) {
				height -= w.getOffsetHeight(); 
			}
			
			scrollContainer.setHeight(height + "px");
		}
		
		/**
		 * 
		 * @param tabTitle
		 * @param anchorPrefix
		 * @return
		 */
		protected String getAnchor(String tabTitle, String anchorPrefix) {
			String anchor = anchorPrefix;
			
			if ((tabTitle != null) && (anchorPrefix != null)) {
				String tabSuffix = ANCHOR_TAB_PORTION_PARAMETER_NAME + "=" + tabTitle;
				if (anchorPrefix.contains("?")) {	
					tabSuffix = "&" + tabSuffix;
				}
				else {
					tabSuffix = "?" + tabSuffix;
				}
				anchor = anchorPrefix + tabSuffix;
			}
			
			return anchor;
		}
		
	}

	/**
	 * Inner class which contains the widgets comprising a 
	 * re-sizable tab containing a left and a right panel.
	 * 
	 * @author esakhat (sanjay.khattar@ericsson.com)
	 *
	 */
	protected class DualPanelResizableTab extends ResizableTab {

		protected static final int MIN_LEFT_CONTENT_WIDTH_PERCENTAGE = 0;
		protected static final int MAX_LEFT_CONTENT_WIDTH_PERCENTAGE = 100;

		protected Widget leftTabContent;
		protected Widget rightTabContent;
		
		protected HorizontalContainer horizontalContainer;
		
		protected ScrollContainer leftScrollContainer;
		protected ScrollContainer rightScrollContainer;
		
		protected int leftContentWidthPercentage = 0;
		protected boolean isLeftContentWidthPercentageSet = false;

		/**
		 * Create a <code>DualPanelResizableTab</code> given the {@link Widget} 
		 * <code>leftContent</code>, <code>rightContent</code> 
		 * and <code>String tabTitle</code>.
		 * <p>
		 * If either <code>leftContent</code> or <code>rightContent</code> is <code>null</code> or 
		 * <code>tabTitle</code> is <code>null</code> or empty, a 
		 * <code>DualPanelResizableTab</code> cannot be created, and a 
		 * {@link NullPointerException} is thrown.  
		 * 
		 * @param tabContent
		 * @param tabTitle
		 */
		public DualPanelResizableTab(Widget leftTabContent, Widget rightTabContent, String tabTitle) {
			super(rightTabContent, tabTitle);
			this.leftTabContent = leftTabContent;
			this.rightTabContent = rightTabContent;
			
			if (leftTabContent == null)  {
				throw new NullPointerException("Widget leftTabContent cannot be null");
			}
			
			if (rightTabContent == null)  {
				throw new NullPointerException("Widget rightTabContent cannot be null");
			}
			
			verticalContainer = new VerticalContainer();
			horizontalContainer = new HorizontalContainer();
			
			leftScrollContainer = new ScrollContainer();
			rightScrollContainer = new ScrollContainer();
			
			buttonsContainer = new HorizontalContainer();
			
			verticalContainer.add(horizontalContainer);
			verticalContainer.add(buttonsContainer);
			
			horizontalContainer.add(leftScrollContainer);
			horizontalContainer.add(rightScrollContainer);
			
			leftScrollContainer.setWidget(leftTabContent);
			rightScrollContainer.setWidget(rightTabContent);

			horizontalContainer.addStyleName(STYLE_NAME_HORIZONTAL_CONTAINER);
			leftScrollContainer.addStyleName(STYLE_NAME_LEFT_SCROLL_CONTAINER);
			rightScrollContainer.addStyleName(STYLE_NAME_RIGHT_SCROLL_CONTAINER);
			buttonsContainer.addStyleName(STYLE_NAME_BUTTONS_CONTAINER);
		}

		/**
		 * @return the leftTabContent
		 */
		public Widget getLeftTabContent() {
			return leftTabContent;
		}

		/**
		 * @param leftTabContent the leftTabContent to set
		 */
		public void setLeftTabContent(Widget leftTabContent) {
			this.leftTabContent = leftTabContent;
		}

		/**
		 * @return the rightTabContent
		 */
		public Widget getRightTabContent() {
			return rightTabContent;
		}

		/**
		 * @param rightTabContent the rightTabContent to set
		 */
		public void setRightTabContent(Widget rightTabContent) {
			this.rightTabContent = rightTabContent;
		}

		/**
		 * @return the leftContentWidthPercentage
		 */
		public int getLeftContentWidthPercentage() {
			return leftContentWidthPercentage;
		}

		/**
		 * @param leftContentWidthPercentage the leftContentWidthPercentage to set
		 */
		public void setLeftContentWidthPercentage(int leftContentWidthPercentage) {
			if (leftContentWidthPercentage < MIN_LEFT_CONTENT_WIDTH_PERCENTAGE) {
				leftContentWidthPercentage = MIN_LEFT_CONTENT_WIDTH_PERCENTAGE;
			}
			if (leftContentWidthPercentage > MAX_LEFT_CONTENT_WIDTH_PERCENTAGE) {
				leftContentWidthPercentage = MAX_LEFT_CONTENT_WIDTH_PERCENTAGE;
			}
			
			this.leftContentWidthPercentage = leftContentWidthPercentage;
			this.isLeftContentWidthPercentageSet = true;
		}

		/**
		 * 
		 */
		protected void updateScrollPanelSize() {
			int height = Window.getClientHeight()
				- CMS_HEADER_MENU_SIZE_PX
				- offsetHeight
				- messageArea.getOffsetHeight()
				- tabContainer.getTabBar().getOffsetHeight()
				- buttonsContainer.getOffsetHeight()
				- BOTTOM_OFFSET_PX;
			
			for (Widget w : heightOffsettingWidgets) {
				height -= w.getOffsetHeight(); 
			}
			
			leftScrollContainer.setHeight(height + "px");
			rightScrollContainer.setHeight(height + "px");

			if (isLeftContentWidthPercentageSet) {
				int width = Window.getClientWidth() - HORIZONTAL_MARGIN_PX
						- offsetWidth;
				for (Widget w : widthOffsettingWidgets) {
					width -= w.getOffsetWidth();
				}
				width = Math.max(width, RESIZABLE_CONTAINER_MIN_WIDTH);
				int leftScrollContainerWidth = width
						* leftContentWidthPercentage / 100;
				int rightScrollContainerWidth = width
						- leftScrollContainerWidth;
				leftScrollContainer.setWidth(leftScrollContainerWidth + "px");
				rightScrollContainer.setWidth(rightScrollContainerWidth + "px");
			}
		}
			
	}

}

