package com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.tandbergtv.neptune.ui.componentmgr.client.i18n.ComponentManagerConstants;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.model.UiComponent;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.service.ComponentUiService;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.service.ComponentUiServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCancelHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCommitHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * The ComponentForm shows the data for a Component
 * 
 */
public class ComponentForm extends Composite implements HasViewCommitHandlers<UiComponent>,
        HasViewCancelHandlers {

	/* Widgets */
	private final VerticalPanel mainPanel = new VerticalPanel();
	private VerticalPanel errorMessagePanel = new VerticalPanel();
	private FormContainer formContainer = new FormContainer(HasHorizontalAlignment.ALIGN_LEFT);
	private TextBoxWidget componentNameBox = new TextBoxWidget();
	private TextBoxWidget baseURLTextBox = new TextBoxWidget();
	private ButtonWidget saveButton, cancelButton;
	private BusyIndicator busyIndicator = new BusyIndicator(this);
	private final int TEXT_FIELD_SIZE = 80;

	/* Services */
	private final ComponentManagerConstants constants = GWT.create(ComponentManagerConstants.class);
	private ComponentUiServiceAsync service;

	/* Constants / Styles */
	private static final String STYLE_NAME = "componentManager-ComponentForm";
	private static final String STYLE_VALIDATION_TABLE = "componentManager-ComponentForm-errorMessageTable";

	/* The state */
	private UiComponent component;
	private boolean readOnly;

	/**
	 * Constructor
	 * 
	 * @param component The component to view
	 */
	public ComponentForm(UiComponent component) {
		this(component, false);
	}

	/**
	 * Constructor
	 * 
	 * @param component The component to view
	 * @param readOnly Show the data in read-only mode or not
	 */
	public ComponentForm(UiComponent component, boolean readOnly) {
		initWidget(this.mainPanel);

		if (component == null)
			component = new UiComponent();

		this.component = component;
		this.readOnly = readOnly;

		initialize();
	}

	// ========================================================================
	// ===================== EVENT HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addViewCommitHandler(ViewCommitHandler<UiComponent> handler) {
		return addHandler(handler, ViewCommitEvent.getType());
	}

	@Override
	public HandlerRegistration addViewCancelHandler(ViewCancelHandler handler) {
		return addHandler(handler, ViewCancelEvent.getType());
	}

	// ========================================================================
	// ===================== FORM API
	// ========================================================================

	/**
	 * Get the component entity being edited by this view. The view may be showing different data based
	 * on the component edits.
	 * 
	 * @return The component entity being edited by the view.
	 */
	public UiComponent getComponent() {
		return this.component;
	}

	/**
	 * Determine if the view is read only or not
	 * 
	 * @return True if read-only, false otherwise
	 */
	public boolean isReadOnly() {
		return readOnly;
	}

	/**
	 * Reset the form to the initial state in the component entity
	 */
	public void reset() {
		clearErrorMessages();
		updateView();
	}

	/**
	 * Show the error messages, using the append flag to indicate if the previously displayed error
	 * messages should be appended to or cleared.
	 * 
	 * @param messages The messages to display
	 * @param append Flag to indicate if the previous messages should be appended to or cleared
	 */
	public void showErrorMessages(List<String> messages, boolean append) {
		if (!append)
			clearErrorMessages();

		/* Show the error messages */
		for (String message : messages)
			showErrorMessage(message, true);
	}

	/**
	 * Show the error message, using the append flag to indicate if the previously displayed error
	 * messages should be appended to or cleared.
	 * 
	 * @param message The message to display
	 * @param append Flag to indicate if the previous messages should be appended to or cleared
	 */
	public void showErrorMessage(String message, boolean append) {
		if (!append)
			clearErrorMessages();

		errorMessagePanel.add(new LabelWidget(message));
	}

	/**
	 * Clear any displayed error messages
	 */
	public void clearErrorMessages() {
		errorMessagePanel.clear();
	}

	// ========================================================================
	// ===================== INTERNAL METHODS
	// ========================================================================

	/*
	 * Initialize the widgets displayed in this form
	 */
	private void initialize() {
		/* Set the style */
		mainPanel.addStyleName(STYLE_NAME);

		/* Build the error messages section */
		errorMessagePanel.setStyleName(STYLE_VALIDATION_TABLE);
		mainPanel.add(errorMessagePanel);

		/* Add the required form fields */
		componentNameBox.setVisibleLength(TEXT_FIELD_SIZE);
		formContainer.addRow(constants.componentName(), componentNameBox, true);
		baseURLTextBox.setVisibleLength(TEXT_FIELD_SIZE);
		formContainer.addRow(constants.componentLocation(), baseURLTextBox, true);

		/* Add the save button */
		if (!readOnly) {
			saveButton = new ButtonWidget(constants.saveButton(), new ClickHandler() {
				public void onClick(ClickEvent event) {
					/* Validate the form, make sure that the basic validation passes */
					handleSaveButtonClick();
				}
			});
			saveButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
			formContainer.addButton(saveButton);
		}

		/* Add the cancel button */
		this.cancelButton = new ButtonWidget(constants.cancelButton(), new ClickHandler() {
			public void onClick(ClickEvent event) {
				handleCancelButtonClick();
			}
		});
		cancelButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		formContainer.addButton(cancelButton);
		mainPanel.add(formContainer);

		/* Update the view */
		updateView();
	}

	/*
	 * Update the view with the provided state
	 */
	private void updateView() {
		/* Update the view with the provided component data */
		componentNameBox.setText(component.getName());
		baseURLTextBox.setText(component.getBaseURL());
	}


	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Cancel button clicked, fire event indicating this
	 */
	private void handleCancelButtonClick() {
		/* Fire event indicating cancel was clicked */
		fireEvent(new ViewCancelEvent());
	}

	/*
	 * Save button clicked, save the component data
	 */
	private void handleSaveButtonClick() {
		/* clear any previous errors */
		clearErrorMessages();

		/* validate */
		if (!isFormValid())
			return;

		/* Build the component to save */
		UiComponent componentToSave = new UiComponent();

		/* Get the fields */
		String name = componentNameBox.getText();
		if (name != null)
			name = name.trim();
		componentToSave.setName(name);
		componentToSave.setBaseURL(baseURLTextBox.getText());

		/* Set the 'invisible' fields */
		componentToSave.setId(component.getId());

		/* Save the component */
		busyIndicator.center();
		getService().saveComponent(componentToSave, new NeptuneAsyncCallback<UiComponent>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				handleSaveComponentFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiComponent result) {
				busyIndicator.hide();
				handleSaveComponentSuccess(result);
			}
		});
	}

	/*
	 * Check if the form is valid. If invalid, the error messages are added to the error panel
	 */
	private boolean isFormValid() {
		return true;
	}

	/*
	 * Get the component service
	 */
	private ComponentUiServiceAsync getService() {
		if (service == null) {
			service = GWT.create(ComponentUiService.class);
		}

		return service;
	}

	/*
	 * Handle the failure callback when saving the component
	 */
	private void handleSaveComponentFailure(Throwable error) {
		/* Do not change the view, just update the error message panel */
		if (error instanceof ValidationException) {
			ValidationException ve = (ValidationException) error;
			for (String message : ve.getValidationMessages()) {
				LabelWidget label = new LabelWidget(message);
				errorMessagePanel.add(label);
			}
		} else {
			LabelWidget label = new LabelWidget(error.getLocalizedMessage());
			errorMessagePanel.add(label);
		}
	}

	/*
	 * Handle the success callback when saving the component
	 */
	private void handleSaveComponentSuccess(UiComponent savedComponent) {
		this.component = savedComponent;
		updateView();

		/* Fire an event indicating that the component is saved */
		this.fireEvent(new ViewCommitEvent<UiComponent>(this.component));
	}
}
