package com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view;

import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_CREATE;
import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_DELETE;
import static com.tandbergtv.neptune.ui.componentmgr.client.ComponentManagerComponentPermissions.COMPONENT_VIEW;
import static com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentDataProvider.NAME_COLUMN;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.componentmgr.client.i18n.ComponentManagerConstants;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.model.UiComponentKey;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentDataProvider;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentRecord;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentTableViewEvent;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentTableViewEventHandler;
import com.tandbergtv.neptune.ui.componentmgr.client.tab.component.view.ComponentTableViewEvent.EventType;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.CookieStoreBasedPageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasRecordClickHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * View for the component table and search / filter criteria
 * 
 */
class ComponentTableView extends LazyView implements HasRecordClickHandlers<ComponentRecord> {

	private final BusyIndicator busyIndicator;
	private VerticalContainer tableContainer;
	private ComponentDataProvider dataProvider;
	private Table<UiComponentKey, ComponentRecord> table;
	private ComponentManagerConstants constants;

	/* Styles / Constants */
	private static final String STYLE_NAME = "componentManager-ComponentTableView";
	private static final String STYLE_TABLE = STYLE_NAME + "-table";
	private static final String COOKIE_PREFIX = "componentList";

	/**
	 * Constructor
	 */
	public ComponentTableView(BusyIndicator busyIndicator) {
		this.busyIndicator = busyIndicator;
		addStyleName(STYLE_NAME);
	}

	/**
	 * Reset the view, clearing the criteria and reseting the table.
	 */
	public void reset() {
		if (getWidget() == null)
			return;

		if (table.isInitialized()) {
			table.reset();
			table.showListView();
		}
	}

	/**
	 * Refresh the view with new state
	 */
	public void refresh(boolean showFirstPage) {
		ensureWidget();

		if (showFirstPage)
			table.getPageFeature().setPageNumber(1);

		/* Initialize and refresh the table */
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should not happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				table.reset(false);
				table.showListView();
				table.refresh();
			}
		});
	}

	/**
	 * Get the selected records in the table
	 * 
	 * @return The selected records in the table
	 */
	public List<ComponentRecord> getSelectedRecords() {
		return table.getSelectedRecords();
	}

	/**
	 * Check if the table is showing the list view of table records
	 * 
	 * @return true if showing the list view, false otherwise
	 */
	public boolean isShowingListView() {
		return (table != null && table.isShowingListView());
	}

	/**
	 * Add the user table view event handler
	 * 
	 * @param handler The event handler
	 * @return The handler registration
	 */
	public HandlerRegistration addComponentTableViewEventHandler(ComponentTableViewEventHandler handler) {
		return addHandler(handler, ComponentTableViewEvent.getType());
	}

	/*
	 * Add handler for the record click event
	 */
	@Override
	public HandlerRegistration addRecordClickHandler(RecordClickHandler<ComponentRecord> handler) {
		return addHandler(handler, RecordClickEvent.getType());
	}

	/*
	 * Get the anchor for this widget
	 */
	@Override
	public String getAnchor() {
		/* Lazy view, widget is not ready. The anchor is empty */
		if (getWidget() == null)
			return "";

		return table.getBookmarkFeature().buildAnchor(table.getTableAnchor());
	}

	/*
	 * Set the anchor on this widget
	 */
	@Override
	protected void setWidgetAnchor(final String anchor) {
		/* Reset the view */
		reset();

		/* Update the table */
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should not happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				/* The table is refreshed, set the table anchor */
				table.setAnchor(anchor);
			}
		});
	}

	/*
	 * Create the lazy widget
	 */
	@Override
	protected Widget createWidget() {
		ComponentManagerConstants constants = getComponentConstants();
		tableContainer = new VerticalContainer();

		/* Build the data provider and the table widget */
		dataProvider = new ComponentDataProvider(this);
		table = new Table<UiComponentKey, ComponentRecord>(dataProvider);
		table.addStyleName(STYLE_TABLE);
		table.addAnchorChangeHandler(new TableAnchorChangeHandler() {
			@Override
			public void onAnchorChange(TableAnchorChangeEvent event) {
				handleTableAnchorChange(event.getAnchor());
			}
		});
		tableContainer.add(table);

		/* Add paging feature */
		PageFeature pageFeature = new CookieStoreBasedPageFeatureImpl(COOKIE_PREFIX);
		pageFeature.setShowTotalCount(true);
		table.addPageFeature(pageFeature);

		/* Add sorting feature */
		SortFeature<UiComponentKey, ComponentRecord> sortFeature = new SortFeatureImpl<UiComponentKey, ComponentRecord>(
		        dataProvider.getColumn(NAME_COLUMN), SortOrder.ASCENDING);
		sortFeature.getSortableColumns().addAll(dataProvider.getColumns());
		table.addSortFeature(sortFeature);

		/* Add the bookmark feature */
		BookmarkFeature bookmarkFeature = new BookmarkFeatureImpl();
		table.addBookmarkFeature(bookmarkFeature);

		/* Add create button */
		if (canCreateRecords()) {
			Button createButton = new ButtonWidget(constants.createButton());
			createButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					handleCreateButtonClick();
				}
			});
			createButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
			table.registerWidgetOnActionContainer(createButton);
		}

		/* Add delete button */
		if (canDeleteRecords()) {
			Button deleteButton = new ButtonWidget(constants.deleteButton());
			deleteButton.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					handleDeleteButtonClick();
				}
			});
			deleteButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
			table.registerWidgetOnActionContainer(deleteButton);
		}

		/* Do not set the table in the container, leave it empty / blank */
		return tableContainer;
	}

	// ========================================================================
	// ===================== INTERNAL STUFF
	// ========================================================================

	/**
	 * Get the constants
	 * 
	 * @return The constants
	 */
	ComponentManagerConstants getComponentConstants() {
		if (constants == null)
			constants = GWT.create(ComponentManagerConstants.class);

		return constants;
	}

	/**
	 * Get the table
	 * 
	 * @return The table
	 */
	Table<UiComponentKey, ComponentRecord> getTable() {
		return table;
	}

	/**
	 * Get the busy indicator
	 */
	BusyIndicator getBusyIndicator() {
		return busyIndicator;
	}

	/**
	 * Determine if the user can create records
	 * 
	 * @return true if user can create records, false otherwise
	 */
	boolean canCreateRecords() {
		return ClientAuthorizationManager.isAuthorized(COMPONENT_VIEW, COMPONENT_CREATE);
	}

	/**
	 * Determine if the user can delete records
	 * 
	 * @return true if user can delete records, false otherwise
	 */
	boolean canDeleteRecords() {
		return ClientAuthorizationManager.isAuthorized(COMPONENT_VIEW, COMPONENT_DELETE);
	}

	/*
	 * Refresh the table contents
	 */
	private void initializeTable(final AsyncCallback<Void> callback) {
		if (table.isInitialized()) {
			callback.onSuccess(null);
			return;
		}

		table.initialize(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Will never happen */
				if (callback != null)
					callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				if (callback != null)
					callback.onSuccess(result);
			}
		});
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the create button click
	 */
	private void handleCreateButtonClick() {
		/* Fire a view event */
		fireEvent(new ComponentTableViewEvent(EventType.CREATE));
	}

	/*
	 * Handle the delete button click
	 */
	private void handleDeleteButtonClick() {
		/* Fire a view event */
		fireEvent(new ComponentTableViewEvent(EventType.DELETE));
	}

	/*
	 * Handle the table anchor change
	 */
	private void handleTableAnchorChange(TableAnchor tableAnchor) {
		String anchor = table.getBookmarkFeature().buildAnchor(tableAnchor);
		fireEvent(new ViewAnchorChangeEvent(anchor));
	}

	/*
	 * Handle the record click
	 */
	void handleTableRecordClick(ComponentRecord record) {
		fireEvent(new RecordClickEvent<ComponentRecord>(record));
	}
}
