package com.tandbergtv.neptune.ui.framework.client;

import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_CENTER;
import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_LEFT;
import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_TOP;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.framework.client.application.ApplicationMessages;
import com.tandbergtv.neptune.ui.framework.client.application.HeaderWidget;
import com.tandbergtv.neptune.ui.framework.client.application.HomeTabPanelFactory;
import com.tandbergtv.neptune.ui.framework.client.application.LoginEvent;
import com.tandbergtv.neptune.ui.framework.client.application.LoginForm;
import com.tandbergtv.neptune.ui.framework.client.application.LoginHandler;
import com.tandbergtv.neptune.ui.framework.client.application.MenuItemSelectionEvent;
import com.tandbergtv.neptune.ui.framework.client.application.MenuItemSelectionHandler;
import com.tandbergtv.neptune.ui.framework.client.application.TreeTabPanel;
import com.tandbergtv.neptune.ui.framework.client.application.TreeTabPanel.MenuNode;
import com.tandbergtv.neptune.ui.framework.client.application.TreeTabPanel.TabNode;
import com.tandbergtv.neptune.ui.framework.client.impl.EventManager;
import com.tandbergtv.neptune.ui.framework.client.impl.NeptuneSecurityImpl;
import com.tandbergtv.neptune.ui.framework.client.impl.ServiceLoaderImpl;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.application.PortletFactory;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.BeforeContentChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.ShowAboutEvent;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.ShowUserGuideEvent;
import com.tandbergtv.neptune.widgettoolkit.client.component.Component;
import com.tandbergtv.neptune.widgettoolkit.client.menu.ActionMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.menu.GroupMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.menu.MenuItemBase;
import com.tandbergtv.neptune.widgettoolkit.client.menu.MenuItemProxy;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.InsufficientRolesException;
import com.tandbergtv.neptune.widgettoolkit.client.security.UserNotLoggedInException;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public class NeptuneApplicationImpl extends NeptuneApplication {

	/* Constants / Style Names */
	private static final String DEFAULT_TOKEN = "Home";
	private static final String STYLE_NAME = "nfw-Application";
	private static final String STYLE_CONTENT_CONTAINER = "nfw-Application-content";

	/* Services and Event Handlers */
	private LoginUiServiceAsync loginService;
	private ServerInfoProviderServiceAsync serverInfoProviderService;
	private ProductInfoProviderServiceAsync productInfoProviderService;

	private ApplicationMessages messages;
	private ApplicationEventHandler eventHandler;
	private Command resizeCommand;
	private HandlerRegistration windowResizeRegistration;
	private HandlerRegistration historyChangeRegistration;
	private boolean ignoreHistoryEvent = false;

	/* Widgets */
	private VerticalContainer mainContainer;
	private SimpleContainer contentContainer;
	private HeaderWidget headerWidget;
	private LoginForm loginForm;
	private TreeTabPanel contentWidget;

	// ========================================================================
	// ===================== CONSTRUCTOR
	// ========================================================================

	/**
	 * Constructor.
	 */
	public NeptuneApplicationImpl() {
		initializeServices();
		initializeErrorHandlers();
		initializeWidgets();
	}

	/*
	 * Initialize the services managed by the application
	 */
	private void initializeServices() {
		this.loginService = GWT.create(LoginUiService.class);
		this.serverInfoProviderService = GWT.create(ServerInfoProviderService.class);
		this.productInfoProviderService = GWT.create(ProductInfoProviderService.class);
		this.messages = GWT.create(ApplicationMessages.class);
		this.resizeCommand = new Command() {
			@Override
			public void execute() {
				handleResizeContentContainer();
			}
		};
	}

	/*
	 * Initialize the basic error handlers for the RPC calls made by the application
	 */
	private void initializeErrorHandlers() {
		/* Error Handler for when the user is not logged into the application */
		this.registerErrorHandler(UserNotLoggedInException.class, new Runnable() {
			public void run() {
				handleUserNotAuthenticatedError();
			}
		});

		/* Error Handler for when the user does not have permissions to perform operation */
		this.registerErrorHandler(InsufficientRolesException.class, new Runnable() {
			public void run() {
				handleUserNotAuthorizedError();
			}
		});
	}

	/*
	 * Initialize the widgets that make up the application
	 */
	private void initializeWidgets() {
		/* The event handler for the application widgets */
		this.eventHandler = new ApplicationEventHandler();

		/* Build the main container */
		mainContainer = new VerticalContainer();
		mainContainer.setVerticalAlignment(ALIGN_TOP);
		mainContainer.setHorizontalAlignment(ALIGN_CENTER);
		mainContainer.setStylePrimaryName(STYLE_NAME);
		getApplicationContainer().setWidget(mainContainer);

		/* Build the content section */
		contentContainer = new SimpleContainer();
		contentContainer.setStylePrimaryName(STYLE_CONTENT_CONTAINER);
		mainContainer.add(contentContainer);

		/* Build the login form */
		loginForm = new LoginForm();
		loginForm.addLoginHandler(eventHandler);

		/* Build the header widget */
		headerWidget = new HeaderWidget();
		headerWidget.addEventHandler(eventHandler);
		headerWidget.setLogoutButtonVisible(false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void hideMenu() {
		if (contentWidget != null)
			contentWidget.hideMenu();
	}

	// ========================================================================
	// ===================== WIDGET - METHOD OVERRIDES
	// ========================================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		/* Add the window resize and the history change event handlers */
		this.windowResizeRegistration = Window.addResizeHandler(eventHandler);
		this.historyChangeRegistration = History.addValueChangeHandler(eventHandler);

		/* Resize the application after the widget is loaded into the DOM */
		resizeContentContainer();
	}

	@Override
	protected void onUnload() {
		/* Remove the window resize and history change event handlers */
		this.windowResizeRegistration.removeHandler();
		this.windowResizeRegistration = null;
		this.historyChangeRegistration.removeHandler();
		this.historyChangeRegistration = null;

		super.onUnload();
	}

	// ========================================================================
	// ===================== APPLICATION INITIALIZATION / DESTROY
	// ========================================================================

	/*
	 * Initialize the application asynchronously. Initialize only when product information is
	 * retrieved
	 */
	@Override
	protected void initializeApplication(final AsyncCallback<Void> callback) {
		productInfoProviderService.getProductInfo(new NeptuneAsyncCallback<UIProductInfo>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);

			}

			@Override
			public void onNeptuneSuccess(UIProductInfo result) {
				buildWidget(result);
				callback.onSuccess(null);
			}
		});
	}

	/*
	 * build UI widgets for application title, subtitle and version
	 */
	private void buildWidget(UIProductInfo info) {
		setApplicationSubtitle(info.getProductSubTitle());
		setApplicationTitle(info.getProductTitle());
		setApplicationVersion(info.getProductVersion());
		String serverTimeDisplayPattern = info.getServerTimeDisplayPattern();

		/* Set the application title, sub-title and version in child widgets */
		loginForm.setApplicationTitle(getApplicationTitle());
		loginForm.setApplicationVersion(getApplicationVersion());
		headerWidget.setApplicationTitle(getApplicationTitle());
		headerWidget.setApplicationSubtitle(getApplicationSubtitle());
		headerWidget.setServerTimeDisplayPattern(serverTimeDisplayPattern);

		/* Determine the current user, and initialize the application view */
		getCurrentUserCredentials();
	}

	/*
	 * Destroy the application
	 */
	@Override
	protected void destroyApplication() {
		/* Clear any application state maintained for current user */
		clearApplicationState();

		super.destroyApplication();
	}

	/*
	 * Verify if user is currently logged in, and get the current user's roles if logged in
	 */
	private void getCurrentUserCredentials() {
		loginService.getCredentials(new AsyncCallback<Credentials>() {
			@Override
			public void onFailure(Throwable caught) {
				handleGetCredentialsFailure(caught);
			}

			public void onSuccess(Credentials result) {
				handleGetCredentialsSuccess(result);
			}
		});
	}

	/*
	 * Handle the failure call when getting user credentials
	 */
	private void handleGetCredentialsFailure(Throwable error) {
		showLoginPage();
	}

	/*
	 * Handle the success call when getting user credentials
	 */
	private void handleGetCredentialsSuccess(Credentials credentials) {
		buildApplicationForUser(credentials.getUsername(), credentials.getRoles());
	}

	// ========================================================================
	// ===================== SHOWING LOGIN PAGE
	// ========================================================================

	/*
	 * Show the login page and clear any maintained security state
	 */
	private void showLoginPage() {
		/* Clear application state */
		clearApplicationState();

		/* Reset the header widget and remove from the parent container */
		headerWidget.clearCurrentUser();
		headerWidget.setLogoutButtonVisible(false);
		headerWidget.removeFromParent();

		/* Reset the login form and set as application contents */
		loginForm.reset();
		contentContainer.setWidget(loginForm);

		/* Resize the application content container */
		resizeContentContainer();
	}

	/*
	 * Clear any application state loaded for the current user
	 */
	private void clearApplicationState() {
		/* Clear previously maintained security and other services */
		this.setSecurity(null);
		this.setComponentProperties(null);
		this.setEventRegistry(null);
		this.setEventSink(null);
		this.setServiceLoader(null);
		this.contentWidget = null;

		/* Destroy all components and release the content widget */
		for (Component component : this.getComponents()) {
			try {
				component.destroy();
			} catch (Exception e) {
				/* Client side error */
				GWT.log("Failure when destroying component: " + component.getClass().getName(), e);
			}
		}
	}

	/*
	 * Determine if the login page is the current content page of the application
	 */
	private boolean isLoginPageShowing() {
		return contentContainer.equals(loginForm.getParent());
	}

	// ========================================================================
	// ===================== LOADING APPLICATION FOR AUTHENTICATED USER
	// ========================================================================

	/*
	 * Load the application for the given user name and roles
	 */
	private void buildApplicationForUser(String userName, List<String> roles) {
		/* Set the new security for the application */
		this.setSecurity(new NeptuneSecurityImpl(userName, roles));

		/* Get the server information for the UI Components */
		serverInfoProviderService.getInfo(new AsyncCallback<ServerInfo>() {
			@Override
			public void onFailure(Throwable caught) {
				handleGetInfoFailure(caught);
			}

			@Override
			public void onSuccess(ServerInfo result) {
				handleGetInfoSuccess(result);
			}
		});
	}

	/*
	 * Handle the failure call when getting the server info for the UI components
	 */
	private void handleGetInfoFailure(Throwable error) {
		showLoginPage();
		Window.alert(error.getLocalizedMessage());
		loginForm.setFocus();
	}

	/*
	 * Handle the server call when getting the server info for the UI components
	 */
	private void handleGetInfoSuccess(ServerInfo serverInfo) {
		/* if there were any errors while preparing for component server properties, show errors */
		if (serverInfo.getErrorMsgs() != null && !serverInfo.getErrorMsgs().isEmpty()) {
			StringBuilder sb = new StringBuilder();
			for (String e : serverInfo.getErrorMsgs()) {
				if (sb.length() > 0) {
					sb.append(",");
				}
				sb.append(e);
			}
			Window.alert(sb.toString());
		}

		buildApplicationContents(serverInfo);
	}

	/*
	 * Build the application contents with the current security, and the fetched server info
	 */
	private void buildApplicationContents(ServerInfo serverInfo) {
		/* Initialize the services / application state */
		this.setComponentProperties(serverInfo.getInfoMap());
		EventManager eventManager = new EventManager();
		this.setEventRegistry(eventManager);
		this.setEventSink(eventManager);
		this.setServiceLoader(new ServiceLoaderImpl(getComponents()));

		/* Initialize all the components */
		for (Component component : getComponents()) {
			component.init(getEventRegistry(), getEventSink(), getServiceLoader(),
			        getComponentProperties());
		}

		/* Build the Tree Tab Panel widget to contain menu items and content widgets */
		this.contentWidget = new TreeTabPanel(eventHandler);
		contentWidget.addListener(eventHandler);
		contentWidget.addMenuItemSelectionHandler(eventHandler);

		/* Initialize the application history */
		initializeHistory();

		/* Build the menu structure for the tree tab panel from the components 'tabs' */
		List<MenuItemBase> tabs = listComponentTabs();
		MenuNode rootMenuNode = contentWidget.getRootMenu();
		TreeTabPanelNodeProducer producer = new TreeTabPanelNodeProducer(rootMenuNode);
		for (MenuItemBase tab : tabs)
			tab.accept(producer);
		contentWidget.doLayout();

		/* Activate the appropriate component widget based on the current history token */
		contentWidget.setActiveTab(History.getToken());

		/* Update the header widget and display the content widget */
		headerWidget.setCurrentUser(getSecurity().getUsername());
		headerWidget.setLogoutButtonVisible(true);
		mainContainer.insert(headerWidget, 0);
		mainContainer.setCellHorizontalAlignment(headerWidget, ALIGN_LEFT);
		contentContainer.setWidget(contentWidget);
		resizeContentContainer();
	}

	/*
	 * Initialize the application history, listening for history change events, and ensuring that
	 * the history events are correctly triggered
	 */
	private void initializeHistory() {
		/* Determine the current anchor or use default if none present */
		String currentToken = History.getToken();
		if (currentToken == null || currentToken.length() == 0)
			currentToken = DEFAULT_TOKEN;

		/*
		 * GWT BUG WORKAROUND: IE does not fire history events until first history item is set.
		 * Forcefully setting first History token.
		 */
		History.newItem(currentToken, false);
	}

	/*
	 * Build the widget menu items that need to be rendered in the application
	 */
	private List<MenuItemBase> listComponentTabs() {
		List<MenuItemBase> tabs = new ArrayList<MenuItemBase>();

		/* Build the menu item for the 'Home' widget */
		WidgetMenuItem homeMenuItem = new WidgetMenuItem("Home", buildHomeWidgetFactory());
		tabs.add(homeMenuItem);

		/* Build the list of menu items for each component */
		for (Component component : this.getComponents()) {
			tabs.addAll(component.listTabs(this.getSecurity()));
		}

		/* Sort the menu items to ensure that all proxy items go to the end of the list */
		Collections.sort(tabs, new Comparator<MenuItemBase>() {
			public int compare(MenuItemBase o1, MenuItemBase o2) {
				if (o1 instanceof MenuItemProxy && !(o2 instanceof MenuItemProxy))
					return 1;
				return 0;
			}
		});

		return tabs;
	}

	/*
	 * Build the Widget Factory used to create the 'Home' widget
	 */
	private HomeTabPanelFactory buildHomeWidgetFactory() {
		/* Build the portlet widget factories */
		List<PortletFactory> portletFactories = new ArrayList<PortletFactory>();
		for (Component component : this.getComponents()) {
			List<PortletFactory> componentPortletFactories = component.listPortlets(getSecurity());
			if (componentPortletFactories != null)
				portletFactories.addAll(componentPortletFactories);
		}

		return new HomeTabPanelFactory(portletFactories);
	}

	// ========================================================================
	// ===================== ERROR HANDLERS
	// ========================================================================

	/*
	 * Handle the error triggered when a server call is made and no user is authenticated.
	 */
	private void handleUserNotAuthenticatedError() {
		/* If the login form is not the current content widget, show the login form */
		if (!isLoginPageShowing()) {
			showLoginPage();
		}
	}

	/*
	 * Handle the error triggered when a server call is made and the user fails authorization.
	 */
	private void handleUserNotAuthorizedError() {
		Window.alert(messages.notAuthorizedErrorMessage());
	}

	// ========================================================================
	// ===================== EVENT HANDLING - LOGIN
	// ========================================================================

	/*
	 * Attempt to log the user in
	 */
	private void performLoginAction(final LoginEvent event) {
		String userName = event.getUserName();
		String password = event.getPassword();
		loginService.login(userName, password, new AsyncCallback<List<String>>() {
			@Override
			public void onFailure(Throwable caught) {
				handleLoginFailure(event, caught);
			}

			@Override
			public void onSuccess(List<String> result) {
				handleLoginSuccess(event, result);
			}
		});
	}

	/*
	 * Handle the success call when attempting user login
	 */
	private void handleLoginSuccess(LoginEvent event, List<String> roles) {
		buildApplicationForUser(event.getUserName(), roles);
	}

	/*
	 * Handle the failure call when attempting user login
	 */
	private void handleLoginFailure(LoginEvent event, Throwable error) {
		List<String> errorMessages = new ArrayList<String>();
		errorMessages.add("Invalid username/password");
		loginForm.showErrorMessages(errorMessages);
		loginForm.setFocus();
	}

	// ========================================================================
	// ===================== EVENT HANDLING - LOGOUT
	// ========================================================================

	/*
	 * Log the current user out
	 */
	private void performLogoutAction(HeaderWidget.Event event) {
		/* First verify that the current content widget can be 'changed' */
		if (cancelContentChange()) {
			return;
		}

		loginService.logout(new AsyncCallback<Void>() {
			@Override
			public void onFailure(Throwable caught) {
				handleLogoutFailure(caught);
			}

			@Override
			public void onSuccess(Void result) {
				handleLogoutSuccess();
			}
		});
	}

	/*
	 * Handle the success call when logging out from the server
	 */
	private void handleLogoutSuccess() {
		/* Attempt to close the window after warning user */
		Window.alert("Logged out successfully.\n For security reasons, "
		        + "please close this window if it did not close automatically.");
		closeWindow();

		/* If window does not close, reset anchor and return the login page */
		History.newItem(null, false);
		showLoginPage();
	}

	/*
	 * Handle the failure call when logging out from the server
	 */
	private void handleLogoutFailure(Throwable error) {
		Window.alert("Logout failed. Reason: " + error.getMessage());
	}

	/*
	 * Native handler to close the window after logout action
	 */
	native public static void closeWindow()
	/*-{
		$wnd.close();
	}-*/;

	// ========================================================================
	// ===================== EVENT HANDLING - HEADER WIDGET BUTTONS
	// ========================================================================

	/*
	 * Show the 'about' screen
	 */
	private void performShowAboutAction(HeaderWidget.Event event) {
		this.fireEvent(new ShowAboutEvent(this));
	}

	/*
	 * Show the 'user guide' screen
	 */
	private void performShowGuideAction(HeaderWidget.Event event) {
		this.fireEvent(new ShowUserGuideEvent(this));
	}

	// ========================================================================
	// ===================== EVENT HANDLING - TREE TAB PANEL & HISTORY
	// ========================================================================

	/*
	 * Handle anchor change notification sent by component widget or by selection of a menu item
	 */
	private void handleComponentAnchorChange(String anchor) {
		History.newItem(anchor, false);
	}

	/*
	 * Handle the anchor change triggered by a new History item. Update application view to show
	 * appropriate component widget
	 */
	private void handleHistoryValueChange(ValueChangeEvent<String> event) {
		if (this.ignoreHistoryEvent) {
			this.ignoreHistoryEvent = false;
			return;
		}

		/* First verify that the current content widget can be 'changed' */
		if (cancelContentChange()) {
			this.ignoreHistoryEvent = true;
			History.back();
			return;
		}

		String anchor = event.getValue();

		/* Replace blank anchor with default anchor */
		if (anchor == null || anchor.trim().length() == 0) {
			anchor = DEFAULT_TOKEN;
		}

		/* Ignore new anchor if the content widget is not initialized */
		if (contentWidget != null)
			contentWidget.setActiveTab(anchor);
	}

	/*
	 * Handle the menu item selection
	 */
	private void handleMenuItemSelected(MenuItemSelectionEvent event) {
		if (cancelContentChange()) {
			event.cancel();
		}
	}

	/*
	 * Determine if the content change event needs to be canceled
	 */
	private boolean cancelContentChange() {
		Widget content = (contentWidget != null) ? contentWidget.getContentWidget() : null;
		if (content != null) {
			BeforeContentChangeEvent event = new BeforeContentChangeEvent(content);
			fireEvent(event);
			return event.isCanceled();
		}

		return false;
	}

	// ========================================================================
	// ===================== APPLICATION RESIZING
	// ========================================================================

	/*
	 * Resize the content container to ensure that the entire visible area is visible
	 */
	private void resizeContentContainer() {
		/* Reset the size constraints added to style attribute */
		contentContainer.setSize("", "");
		Scheduler.get().scheduleDeferred(resizeCommand);
	}

	/* Handle the resize of the content container */
	private void handleResizeContentContainer() {
		/* Only if attached to the document, attempt to resize the container */
		if (contentContainer.isAttached()) {
			int clientHeight = Window.getClientHeight();
			int bodyHeight = RootPanel.get().getOffsetHeight();
			int difference = clientHeight - bodyHeight;
			if (difference > 0) {
				int containerHeight = contentContainer.getOffsetHeight() + difference;
				contentContainer.setHeight(containerHeight + "px");
			}
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLER FOR DIFFERENT WIDGETS
	// ========================================================================

	/*
	 * Event handler for the HeaderWidget events
	 */
	private class ApplicationEventHandler implements HeaderWidget.EventHandler, LoginHandler,
	        TreeTabPanel.AnchorChangeListener, TreeTabPanel.Listener, ValueChangeHandler<String>,
	        ResizeHandler, MenuItemSelectionHandler {

		/*
		 * Header Widget - 'about' button click
		 */
		@Override
		public void onAboutButtonClick(HeaderWidget.Event event) {
			performShowAboutAction(event);
		}

		/*
		 * Header Widget - 'user guide' button click
		 */
		@Override
		public void onUserGuideButtonClick(HeaderWidget.Event event) {
			performShowGuideAction(event);
		}

		/*
		 * Header Widget - 'logout' button click
		 */
		@Override
		public void onLogoutButtonClick(HeaderWidget.Event event) {
			performLogoutAction(event);
		}

		/*
		 * Login Form - 'login' button click
		 */
		@Override
		public void onLogin(LoginEvent event) {
			performLoginAction(event);
		}

		/*
		 * Tree Tab Panel - event to update the current history anchor
		 */
		@Override
		public void anchorChanged(String anchor) {
			handleComponentAnchorChange(anchor);
		}

		/*
		 * Tree Tab Panel - event on selection of a new menu item
		 */
		@Override
		public void tabChanged(TabNode tabNode) {
			handleComponentAnchorChange(tabNode.getToken());
		}

		/*
		 * History - value change event
		 */
		@Override
		public void onValueChange(ValueChangeEvent<String> event) {
			handleHistoryValueChange(event);
		}

		@Override
		public void onResize(ResizeEvent event) {
			resizeContentContainer();
		}

		/* 
		 * TreeTabPanel - menu selection 
		 */
		@Override
		public void onMenuItemSelected(MenuItemSelectionEvent event) {
			handleMenuItemSelected(event);
		}
	}

	// ========================================================================
	// ===================== TREE TAB PANEL WIDGET PRODUCER
	// ========================================================================

	private final class TreeTabPanelNodeProducer implements MenuItemBase.Visitor {
		private final class ForwardingAnchorChangeListener implements
		        TreeTabPanel.AnchorChangeListener {
			private final WidgetMenuItem menuitem;

			private ForwardingAnchorChangeListener(WidgetMenuItem panel) {
				this.menuitem = panel;
			}

			public void anchorChanged(String anchor) {
				menuitem.getAnchorChangeListener().anchorChanged(anchor);
			}
		}

		private final TreeTabPanel.MenuNode menuNode;

		private TreeTabPanelNodeProducer(TreeTabPanel.MenuNode menuNode) {
			this.menuNode = menuNode;
		}

		public void visit(WidgetMenuItem menuitem) {
			final TabNode tabNode = menuNode.addTabNode(menuitem.getName(),
			        menuitem.getDisplayName(), menuitem.getWidgetFactory(),
			        new ForwardingAnchorChangeListener(menuitem));
			menuitem.setReverseAnchorChangeListener(new WidgetMenuItem.AnchorChangeListener() {
				public void anchorChanged(String anchor) {
					tabNode.getReverseAnchorChangeListener().anchorChanged(anchor);
				}
			});
		}

		public void visit(GroupMenuItem menu) {
			TreeTabPanel.MenuNode subMenuNode = menuNode.addMenuNode(menu.getName(),
			        menu.getDisplayName());

			for (MenuItemBase menuitem : menu.getSubPanels())
				menuitem.accept(new TreeTabPanelNodeProducer(subMenuNode));
		}

		public void visit(ActionMenuItem action) {
			menuNode.addActionNode(action.getName(), action.getDisplayName(), action.getRunnable());
		}

		@Override
		public void visit(MenuItemProxy proxy) {
			/* This menuNode is guaranteed to be the root */
			MenuNode node = menuNode.getMenuNode(proxy.getName());

			/* Mighty odd, someone is trying to attach to a non-existent parent */
			if (node == null)
				node = menuNode.addMenuNode(proxy.getName(), proxy.getDisplayName());

			for (MenuItemBase item : proxy.getItems())
				item.accept(new TreeTabPanelNodeProducer(node));
		}
	}
}
