package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl;

import java.util.Date;

import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.datepicker.client.CalendarUtil;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;

/**
 * Stores sort column and sort order of the table in a cookie in addition to memory.
 * 
 * @author Vijay Silva
 */
public class CookieStoreBasedSortFeatureImpl<K, R extends Record<K>> extends SortFeatureImpl<K, R> {

	private final String cookieNamePrefix;
	private Date expiryDate;
	private boolean writeToCookie = true;

	private static final String SORT_ORDER_SUFFIX = "_sortOrder";
	private static final String SORT_NAME_SUFFIX = "_sortColumnName";

	/**
	 * Constructor
	 * 
	 * @param cookieNamePrefix The cookie name prefix
	 */
	public CookieStoreBasedSortFeatureImpl(String cookieNamePrefix, Column<?, R> defaultSortColumn,
	        SortOrder defaultSortOrder) {
		super(defaultSortColumn, defaultSortOrder);

		this.cookieNamePrefix = cookieNamePrefix;
		this.expiryDate = new Date();
		CalendarUtil.addMonthsToDate(this.expiryDate, 1200);
	}

	@Override
	public SortOrder getSortOrder() {
		/* Read the value from the cookie first, then default to value in memory */
		SortOrder sortOrder = getSortOrderFromCookie();
		return (sortOrder != null) ? sortOrder : super.getSortOrder();
	}

	@Override
	public void setSortOrder(SortOrder sortOrder) {
		/* Set the value in 'memory' */
		super.setSortOrder(sortOrder);

		/* Get the value set, and write it to the cookie */
		if (writeToCookie) {
			sortOrder = super.getSortOrder();
			String value = (sortOrder != null) ? sortOrder.name() : "";
			Cookies.setCookie(getSortOrderCookieName(), value, getCookieExpiryDate());
		}
	}

	@Override
	public Column<?, R> getSortColumn() {
		/* Read the value from the cookie first, then default to value in memory */
		Column<?, R> sortColumn = getSortColumnFromCookie();
		return (sortColumn != null) ? sortColumn : super.getSortColumn();
	}

	@Override
	public void setSortColumn(Column<?, R> column) {
		/* Set the value in 'memory' */
		super.setSortColumn(column);

		/* Get the value set, and write it to the cookie */
		if (writeToCookie) {
			column = super.getSortColumn();
			String value = (column != null) ? column.getName() : null;
			if (value == null)
				value = "";
			Cookies.setCookie(getSortNameCookieName(), value, getCookieExpiryDate());
		}
	}

	@Override
	public void reset() {
		/* Reset is called by base constructor, the cookie name is not always available */
		this.writeToCookie = false;
		super.reset();
		this.writeToCookie = true;

		/* Cannot sync cookie value with memory here because cookie name is not available */
	}

	/**
	 * Get the expiry date to use when writing the cookies
	 * 
	 * @return The cookie expiry date
	 */
	public Date getCookieExpiryDate() {
		return this.expiryDate;
	}

	/**
	 * Set the cookie expiry date, or set to null for the cookie to never expire. The date is used
	 * for future writes to the cookie. The current cookie is not changed.
	 * 
	 * @param expiryDate The cookie expiry date
	 */
	public void setCookieExpiryDate(Date expiryDate) {
		this.expiryDate = expiryDate;
	}

	/*
	 * Get the name of the cookie for the sort order
	 */
	private String getSortOrderCookieName() {
		return this.cookieNamePrefix + SORT_ORDER_SUFFIX;
	}

	/*
	 * Get the sort order value from the cookie
	 */
	private SortOrder getSortOrderFromCookie() {
		try {
			String value = Cookies.getCookie(getSortOrderCookieName());
			if (value != null)
				return SortOrder.valueOf(value);
		} catch (Exception e) {
		}

		return null;
	}

	/*
	 * Get the name of the cookie for the sort name
	 */
	private String getSortNameCookieName() {
		return this.cookieNamePrefix + SORT_NAME_SUFFIX;
	}

	/*
	 * Get the sort column matched from the cookie
	 */
	private Column<?, R> getSortColumnFromCookie() {
		try {
			String value = Cookies.getCookie(getSortNameCookieName());
			if (value != null) {
				for (Column<?, R> column : getSortableColumns()) {
					if (column != null && value.equals(column.getName()))
						return column;
				}
			}
		} catch (Exception e) {
		}

		return null;
	}
}
