/*
 * Created on Jan 3, 2012
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view;

import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_CREATE;
import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_DELETE;
import static com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions.USER_VIEW;
import static com.tandbergtv.neptune.ui.realm.client.tab.user.view.UserDataProvider.USERNAME_COLUMN;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserConfiguration;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserKey;
import com.tandbergtv.neptune.ui.realm.client.tab.user.view.UserTableViewEvent.EventType;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.CookieStoreBasedPageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasRecordClickHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RecordClickHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * View for the external users table and search / filter criteria
 * 
 * @author Vijay Silva
 */
class UserTableView extends LazyView implements HasRecordClickHandlers<UserRecord> {

	private final UiUserConfiguration configuration;
	private final BusyIndicator busyIndicator;
	private VerticalContainer tableContainer;
	private UserDataProvider dataProvider;
	private Table<UiUserKey, UserRecord> table;
	private RealmConstants constants;

	/* Styles / Constants */
	private static final String STYLE_NAME = "realm-UserTableView";
	private static final String STYLE_TABLE = STYLE_NAME + "-table";
	private static final String COOKIE_PREFIX = "userList";
	
	private ResizableContainer container;
	private ButtonWidget createButton;
	private ButtonWidget createExternalButton;
	private ButtonWidget deleteButton;

	/**
	 * Constructor
	 */
	public UserTableView(UiUserConfiguration configuration, BusyIndicator busyIndicator, ResizableContainer container) {
		this.configuration = configuration;
		this.busyIndicator = busyIndicator;
		this.container = container;
		addStyleName(STYLE_NAME);
	}

	/**
	 * Reset the view, clearing the criteria and reseting the table.
	 */
	public void reset() {
		if (getWidget() == null)
			return;

		if (table.isInitialized()) {
			table.reset();
			table.showListView();
		}
	}

	/**
	 * Refresh the view with new state
	 */
	public void refresh(boolean showFirstPage) {
		ensureWidget();

		if (showFirstPage)
			table.getPageFeature().setPageNumber(1);

		/* Initialize and refresh the table */
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should not happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				table.reset(false);
				table.showListView();
				table.refresh();
			}
		});
	}

	/**
	 * Get the selected records in the table
	 * 
	 * @return The selected records in the table
	 */
	public List<UserRecord> getSelectedRecords() {
		return table.getSelectedRecords();
	}

	/**
	 * Check if the table is showing the list view of table records
	 * 
	 * @return true if showing the list view, false otherwise
	 */
	public boolean isShowingListView() {
		return (table != null && table.isShowingListView());
	}

	/**
	 * Add the user table view event handler
	 * 
	 * @param handler The event handler
	 * @return The handler registration
	 */
	public HandlerRegistration addUserTableViewEventHandler(UserTableViewEventHandler handler) {
		return addHandler(handler, UserTableViewEvent.getType());
	}

	/*
	 * Add handler for the record click event
	 */
	@Override
	public HandlerRegistration addRecordClickHandler(RecordClickHandler<UserRecord> handler) {
		return addHandler(handler, RecordClickEvent.getType());
	}

	/*
	 * Get the anchor for this widget
	 */
	@Override
	public String getAnchor() {
		/* Lazy view, widget is not ready. The anchor is empty */
		if (getWidget() == null)
			return "";

		return table.getBookmarkFeature().buildAnchor(table.getTableAnchor());
	}

	/*
	 * Set the anchor on this widget
	 */
	@Override
	protected void setWidgetAnchor(final String anchor) {
		/* Reset the view */
		reset();

		/* Update the table */
		initializeTable(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Should not happen */
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				/* The table is refreshed, set the table anchor */
				table.setAnchor(anchor);
			}
		});
	}

	/*
	 * Create the lazy widget
	 */
	@Override
	protected Widget createWidget() {
		RealmConstants constants = getRealmConstants();
		tableContainer = new VerticalContainer();

		/* Build the data provider and the table widget */
		dataProvider = new UserDataProvider(this);
		table = new Table<UiUserKey, UserRecord>(dataProvider);
		table.addStyleName(STYLE_TABLE);
		table.addAnchorChangeHandler(new TableAnchorChangeHandler() {
			@Override
			public void onAnchorChange(TableAnchorChangeEvent event) {
				handleTableAnchorChange(event.getAnchor());
			}
		});
		tableContainer.add(table);

		/* Add paging feature */
		PageFeature pageFeature = new CookieStoreBasedPageFeatureImpl(COOKIE_PREFIX);
		pageFeature.setShowTotalCount(true);
		table.addPageFeature(pageFeature);

		/* Add sorting feature */
		SortFeature<UiUserKey, UserRecord> sortFeature = new SortFeatureImpl<UiUserKey, UserRecord>(
		        dataProvider.getColumn(USERNAME_COLUMN), SortOrder.ASCENDING);
		sortFeature.getSortableColumns().addAll(dataProvider.getColumns());
		table.addSortFeature(sortFeature);

		/* Add the bookmark feature */
		BookmarkFeature bookmarkFeature = new BookmarkFeatureImpl();
		table.addBookmarkFeature(bookmarkFeature);

		/* Add the create external button */
		createExternalButton = new ButtonWidget(constants.createExternalUserButton());
		createExternalButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleCreateExternalButtonClick();
			}
		});
		
		createExternalButton.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		createExternalButton.addStyleName(StyleNames.STYLE_EB_BTN);
		createExternalButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
		createExternalButton.addStyleName(StyleNames.STYLE_CREATE_BUTTON_ICON);
		container.addButton(constants.userTab(), createExternalButton);		
				
		showCreateExternalButton();
		

		/* Add create button */
		createButton = new ButtonWidget(constants.createButton());
		createButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleCreateButtonClick();
			}
		});
		
		createButton.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		createButton.addStyleName(StyleNames.STYLE_EB_BTN);
		createButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
		createButton.addStyleName(StyleNames.STYLE_CREATE_BUTTON_ICON);
		container.addButton(constants.userTab(), createButton);
		showCreateButton();

		/* Add delete button */
		deleteButton = new ButtonWidget(constants.deleteButton());
		deleteButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleDeleteButtonClick();
			}
		});
		
		deleteButton.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		deleteButton.addStyleName(StyleNames.STYLE_EB_BTN);
		deleteButton.addStyleName(StyleNames.STYLE_DELETE_BUTTON_ICON);
		container.addButton(constants.userTab(), deleteButton);
		showDeleteButton();

		/* Do not set the table in the container, leave it empty / blank */
		return tableContainer;
	}

	// ========================================================================
	// ===================== INTERNAL STUFF
	// ========================================================================

	/**
	 * Get the constants
	 * 
	 * @return The constants
	 */
	RealmConstants getRealmConstants() {
		if (constants == null)
			constants = GWT.create(RealmConstants.class);

		return constants;
	}

	/**
	 * Get the table
	 * 
	 * @return The table
	 */
	Table<UiUserKey, UserRecord> getTable() {
		return table;
	}

	/**
	 * Get the busy indicator
	 */
	BusyIndicator getBusyIndicator() {
		return busyIndicator;
	}

	/**
	 * Determine if the user can delete records
	 * 
	 * @return true if user can delete records, false otherwise
	 */
	boolean canDeleteRecords() {
		return ClientAuthorizationManager.isAuthorized(USER_VIEW, USER_DELETE);
	}

	/*
	 * Refresh the table contents
	 */
	private void initializeTable(final AsyncCallback<Void> callback) {
		if (table.isInitialized()) {
			callback.onSuccess(null);
			return;
		}

		table.initialize(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				/* Will never happen */
				if (callback != null)
					callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				if (callback != null)
					callback.onSuccess(result);
			}
		});
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the create external button click
	 */
	private void handleCreateExternalButtonClick() {
		/* Fire a view event */
		fireEvent(new UserTableViewEvent(EventType.CREATE_EXTERNAL));
		createButton.setVisible(false);
		createExternalButton.setVisible(false);
		deleteButton.setVisible(false);
	}

	/*
	 * Handle the create button click
	 */
	private void handleCreateButtonClick() {
		/* Fire a view event */
		fireEvent(new UserTableViewEvent(EventType.CREATE));
		createButton.setVisible(false);
		deleteButton.setVisible(false);
		createExternalButton.setVisible(false);
	}

	/*
	 * Handle the delete button click
	 */
	private void handleDeleteButtonClick() {
		/* Fire a view event */
		fireEvent(new UserTableViewEvent(EventType.DELETE));
		
	}

	/*
	 * Handle the table anchor change
	 */
	private void handleTableAnchorChange(TableAnchor tableAnchor) {
		String anchor = table.getBookmarkFeature().buildAnchor(tableAnchor);
		fireEvent(new ViewAnchorChangeEvent(anchor));
	}

	/*
	 * Handle the record click
	 */
	void handleTableRecordClick(UserRecord record) {
		fireEvent(new RecordClickEvent<UserRecord>(record));
		createButton.setVisible(false);
		deleteButton.setVisible(false);
		createExternalButton.setVisible(false);
	}
	
	public void showCreateButton(){
		if(createButton == null)
			return;
		
		boolean canCreate = ClientAuthorizationManager.isAuthorized(USER_VIEW, USER_CREATE);
		boolean externalMode = configuration.isExternalMode();
		if (canCreate && (!externalMode || configuration.isCreateInternalUserAllowed())) {
				createButton.setVisible(true);
		}
		else
			createButton.setVisible(false);
	}
	
	public void showCreateExternalButton(){
		if(createExternalButton == null)
			return;
		
		boolean canCreate = ClientAuthorizationManager.isAuthorized(USER_VIEW, USER_CREATE);
		boolean externalMode = configuration.isExternalMode();
		if(canCreate && externalMode)
			createExternalButton.setVisible(true);
		else
			createExternalButton.setVisible(false);
	}
	
	public void showDeleteButton(){
		if(deleteButton == null)
			return;
		
		if(canDeleteRecords())
			deleteButton.setVisible(true);
		else
			deleteButton.setVisible(false);
	}
	
	public void hideCreateButton(){
		if(createButton != null)
			createButton.setVisible(false);
	}
	
	public void hideCreateExternalButton(){
		if(createExternalButton !=null)
			createExternalButton.setVisible(false);
	}
	
	public void hideDeleteButton(){
		if(deleteButton != null)
			deleteButton.setVisible(false);
	}
}
