package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.INeptuneWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableListViewPanel.Listener;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.ExpandingFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.BookmarkFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.CheckBox;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.DetailFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.ExpandingFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.FeatureEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.FeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.PageFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.SortFeatureHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

/**
 * <p>
 * Table that supports a simple view by default, but that can be extended with the addition on
 * features. Supported Features: SortFeature DetailFeature PageFeature BookmarkFeature
 * </p>
 * <p>
 * TODO, continue re-factor to properly encapsulate the built-in table functionality. The
 * InternalTable interface should not allow access to the internal column structure to support the
 * additional features.
 * </p>
 */
public class Table<K, R extends Record<K>> extends Composite implements INeptuneWidget {

	/* Widgets */
	private final SimplePanel mainPanel;
	private final TableListViewPanel<K, R> listViewPanel;

	/* State */
	private final DataProvider<K, R> dataProvider;
	private final InternalTable<K, R> internalTable;
	private String viewName = null;
	private String listViewName = DEFAULT_LIST_VIEW_NAME;
	private AnchorChangeListener anchorChangeNotifier;
	private final List<TableAnchorChangeHandler> anchorChangeHandlers = new ArrayList<TableAnchorChangeHandler>();
	private final List<TableViewChangeHandler<K, R>> viewHandlers = new ArrayList<TableViewChangeHandler<K, R>>();

	/* Feature handlers */
	private SortFeatureHandler<K, R> sortFeatureHandler;
	private PageFeatureHandler<K, R> pageFeatureHandler;
	private DetailFeatureHandler<K, R> detailFeatureHandler;
	private BookmarkFeatureHandler<K, R> bookmarkFeatureHandler;
	private ExpandingFeatureHandler<K, R> expandingFeatureHandler;
	private final List<FeatureHandler<?>> featureHandlers;

	/* Initialization properties */
	private boolean initialized = false;
	private boolean initializationStarted = false;
	private boolean providerInitializing = false;
	private List<AsyncCallback<Void>> initializationCallbacks = new ArrayList<AsyncCallback<Void>>();

	/* Default view name for the table list view */
	private static final String DEFAULT_LIST_VIEW_NAME = "";

	/**
	 * Constructor
	 * 
	 * @param dataProvider the data provider
	 */
	public Table(DataProvider<K, R> dataProvider) {
		this(dataProvider, null);
	}

	/**
	 * Constructor
	 * 
	 * @param dataProvider the data provider
	 * @param expandingFeature The expanding feature
	 */
	public Table(DataProvider<K, R> dataProvider, ExpandingFeature<K, R> expandingFeature) {
		super();

		this.dataProvider = dataProvider;
		this.internalTable = new InternalTable<K, R>(this);
		this.featureHandlers = new ArrayList<FeatureHandler<?>>();
		if (expandingFeature != null) {
			addExpandingFeature(expandingFeature);
		}

		/* The main widget panel */
		this.mainPanel = new SimpleContainer();
		this.mainPanel.setStylePrimaryName(TableConstants.STYLE_TABLE_WIDGET);

		/* The panel for the list view */
		this.listViewPanel = new TableListViewPanel<K, R>(dataProvider, getExpandingFeature());
		this.listViewPanel.addListener(new TableListViewPanel.Listener() {
			public void updated() {
				handleListViewUpdated();
			}
		});

		/* Leave the panel contents blank */
		mainPanel.setWidget(new LabelWidget(""));
		this.initWidget(mainPanel);
	}

	// ========================================================================
	// ===================== INITIALIZATION
	// ========================================================================

	/**
	 * Initialize the table. Causes the table to initialize the data provider and the table
	 * features, and ensures that the table list view is displayed after successful initialization.
	 * Ensure that all features have been installed, and the view names have been correctly set
	 * before calling initialization.
	 * 
	 * @param callback The asynchronous callback to notify after initialization completes.
	 */
	public synchronized void initialize(AsyncCallback<Void> callback) {
		if (this.initialized) {
			if (callback != null)
				callback.onSuccess(null);
			return;
		}

		this.initializationStarted = true;
		if (callback != null)
			initializationCallbacks.add(callback);
		if (!this.providerInitializing) {
			providerInitializing = true;

			/* Initialize the data provider asynchronously */
			dataProvider.initialize(new NeptuneAsyncCallback<Void>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					handleProviderInitializationFailure(caught);
				}

				@Override
				public void onNeptuneSuccess(Void result) {
					handleProviderInitializationSuccess(result);
				}
			});
		}
	}

	/**
	 * Determine if the table has completed initialization successfully.
	 * 
	 * @return true if initialization is complete, false otherwise
	 */
	public synchronized boolean isInitialized() {
		return initialized;
	}

	/*
	 * Determine if the table is in the process of initializing or has completed initialization
	 */
	private synchronized void validateNotInitialized(String operation) {
		if (initializationStarted) {
			String msg = "The table initialization has already started, cannot perform operation: "
			        + operation + " after initialization.";
			throw new RuntimeException(msg);
		}
	}

	/*
	 * Handle the failure event triggered by the data provider initialization
	 */
	private synchronized void handleProviderInitializationFailure(Throwable error) {
		this.providerInitializing = false;

		/* Notify all registered callbacks and clear list */
		for (AsyncCallback<Void> callback : this.initializationCallbacks) {
			try {
				callback.onFailure(error);
			} catch (Exception e) {
			}
		}
		this.initializationCallbacks.clear();
	}

	/*
	 * Handle the success event triggered by the data provider initialization
	 */
	private synchronized void handleProviderInitializationSuccess(Void result) {
		this.providerInitializing = false;

		/* Initialize all feature handlers */
		for (FeatureHandler<?> handler : featureHandlers)
			handler.initialize();

		/* Mark the table as initialized */
		this.initialized = true;

		/* Show the list view after initialization is marked complete */
		showListView();

		/* Notify all registered callbacks and clear list */
		for (AsyncCallback<Void> callback : this.initializationCallbacks) {
			try {
				callback.onSuccess(result);
			} catch (Exception e) {
			}
		}
		this.initializationCallbacks.clear();
	}

	// ========================================================================
	// ===================== TABLE VIEW MANAGEMENT
	// ========================================================================

	/**
	 * Get the name of the current view displayed as the table view
	 * 
	 * @return the table view name
	 */
	public String getViewName() {
		return this.viewName;
	}

	/**
	 * Get the view name used for the list view.
	 * 
	 * @return The list view name
	 */
	public String getListViewName() {
		return this.listViewName;
	}

	/**
	 * Set the name to use for the list view. This name should be set before the table is
	 * initialized.
	 * 
	 * @param listViewName The list view name (cannot be null)
	 */
	public void setListViewName(String listViewName) {
		this.validateNotInitialized("setListViewName");
		if (listViewName == null)
			throw new IllegalArgumentException("The list view name cannot be null.");

		this.listViewName = listViewName;
	}

	/**
	 * Allows the feature handler to change the view displayed by the table
	 * 
	 * @param view The new view
	 * @param viewName The view name used by the table
	 */
	void showView(Widget view, String viewName) {
		if (!this.isInitialized()) {
			String msg = "Cannot set the table view before the table is initialized.";
			throw new RuntimeException(msg);
		}

		mainPanel.setWidget(view);

		/* Check if changing to the list view and fire event if required */
		String oldViewName = this.viewName;
		this.viewName = viewName;
		if (listViewName.equals(viewName) && !listViewName.equals(oldViewName))
			fireTableListViewEvent(new TableViewEvent());
	}

	/**
	 * Determine if the view is showing as the table view
	 * 
	 * @param view The view
	 * @return true if the view is the table view, false otherwise
	 */
	boolean isShowingView(Widget view) {
		return (mainPanel.getWidget() == view);
	}

	/**
	 * Changes the view displayed by the table to the list view
	 */
	public void showListView() {
		showView(listViewPanel, listViewName);
	}

	/**
	 * Determine if the table is showing the list view
	 * 
	 * @return true if the list view is current table view, false otherwise
	 */
	public boolean isShowingListView() {
		return isShowingView(listViewPanel);
	}

	/**
	 * Add an event handler for the table view change event
	 * 
	 * @param handler the table view change event handler
	 */
	public void addTableViewChangeHandler(TableViewChangeHandler<K, R> handler) {
		if (handler != null && !viewHandlers.contains(handler)) {
			viewHandlers.add(handler);
		}
	}

	/**
	 * Remove a registered event handler for the table view change event
	 * 
	 * @param handler the table view change event handler
	 */
	public void removeTableViewChangeHandler(TableViewChangeHandler<K, R> handler) {
		viewHandlers.remove(handler);
	}

	/*
	 * Fire a event indicating that the list view is being displayed
	 */
	private void fireTableListViewEvent(TableViewEvent event) {
		for (TableViewChangeHandler<K, R> handler : viewHandlers)
			handler.onShowListView(event);
	}

	// ========================================================================
	// ===================== TABLE ANCHOR MANAGEMENT
	// ========================================================================

	/**
	 * Builds a new table anchor using the current state of the table and its features
	 * 
	 * @return The table anchor
	 */
	public TableAnchor getTableAnchor() {
		TableAnchor anchor = new TableAnchor();
		anchor.setViewName(this.viewName);
		for (FeatureHandler<?> handler : this.featureHandlers) {
			Map<String, String> featureTokens = handler.getAnchorTokens();
			if (featureTokens != null)
				anchor.getTokens().putAll(featureTokens);
		}

		return anchor;
	}

	/**
	 * Build the table widget using the provided table anchor. This will reset the table view and
	 * state, and re-initialize based on the state in the anchor. Ensure that the table is
	 * initialized before calling this method.
	 * 
	 * @param anchor The table anchor
	 */
	public void setTableAnchor(TableAnchor anchor) {
		/* Reset the table */
		reset();

		/* Set the list view as the default view */
		showListView();

		/* Go through each feature handler and set the anchor tokens */
		for (FeatureHandler<?> featureHandler : featureHandlers) {
			featureHandler.setAnchor(anchor);
		}

		/* Refresh the table list view if the list view is still showing */
		if (isShowingListView())
			refresh();
	}

	/**
	 * Add a listener for anchor change events triggered by the table
	 * 
	 * @param handler the event handler
	 */
	public void addAnchorChangeHandler(TableAnchorChangeHandler handler) {
		if (handler != null && !this.anchorChangeHandlers.contains(handler))
			anchorChangeHandlers.add(handler);
	}

	/**
	 * Remove a listener for anchor change events triggered by the table
	 * 
	 * @param handler the event handler
	 */
	public void removeAnchorChangeHandler(TableAnchorChangeHandler handler) {
		anchorChangeHandlers.remove(handler);
	}

	/**
	 * Get the anchor change notifier to use for updating the browser history
	 * 
	 * @return The anchor change notifier
	 */
	public AnchorChangeListener getAnchorChangeNotifier() {
		return this.anchorChangeNotifier;
	}

	/**
	 * Set the anchor change notifier to use for updating the history. If using the bookmark
	 * feature, ensure that this notifier is set before the user can perform any actions on the
	 * table.
	 * 
	 * @param anchorChangeNotifier The anchor change notifier
	 */
	public void setAnchorChangeNotifier(AnchorChangeListener anchorChangeNotifier) {
		this.anchorChangeNotifier = anchorChangeNotifier;
	}

	// ========================================================================
	// ===================== SELECTION CHANGE MANAGEMENT
	// ========================================================================

	/**
	 * Add a listener for selection change events triggered by the table
	 * 
	 * @param handler the event handler
	 */
	public void addSelectionChangeHandler(final SelectionChangeHandler<K, R> handler) {
		// Creating another SelectionChangeHandler to return this Table widget
		listViewPanel.addSelectionChangeHandler(new SelectionChangeHandler<K, R>() {

			@Override
			public void onSelectionChange(SelectionChangeEvent<K, R> event) {
				handler.onSelectionChange(new SelectionChangeEvent<K, R>(Table.this, event
				        .getSelectedRecords()));
			}
		});
	}

	/**
	 * Remove a listener for selection change events triggered by the table
	 * 
	 * @param handler the event handler
	 */
	public void removeSelectionChangeHandler(SelectionChangeHandler<K, R> handler) {
		listViewPanel.removeSelectionChangeHandler(handler);
	}

	// ========================================================================
	// ===================== PAGE FEATURE
	// ========================================================================

	/**
	 * Installs the Page Feature. Ensure that the table is not initialized and that the feature is
	 * not previously installed before calling this method.
	 * 
	 * @param pageFeature the page feature
	 */
	public void addPageFeature(PageFeature pageFeature) {
		/* Ensure that initialization has not started */
		validateNotInitialized("addPageFeature");

		/* Validate that the feature is not already installed */
		if (this.hasPageFeature()) {
			String msg = "The Page Feature is already installed, cannot add this feature again.";
			throw new RuntimeException(msg);
		}

		this.pageFeatureHandler = new PageFeatureHandler<K, R>(internalTable, pageFeature);
		featureHandlers.add(pageFeatureHandler);
	}

	/**
	 * Determine if the page feature is installed
	 * 
	 * @return true if the page feature is installed, false otherwise
	 */
	public boolean hasPageFeature() {
		return (this.pageFeatureHandler != null);
	}

	/**
	 * Get the page feature, if previously installed
	 * 
	 * @return the page feature if installed, or null
	 */
	public PageFeature getPageFeature() {
		return (hasPageFeature()) ? pageFeatureHandler.getFeature() : null;
	}

	// ========================================================================
	// ===================== SORT FEATURE
	// ========================================================================

	/**
	 * Installs the Sort Feature. Ensure that the table is not initialized and that the feature is
	 * not previously installed before calling this method.
	 * 
	 * @param sortFeature the sort feature
	 */
	public void addSortFeature(SortFeature<K, R> sortFeature) {
		/* Ensure that initialization has not started */
		validateNotInitialized("addSortFeature");

		/* Validate that the feature is not already installed */
		if (this.hasSortFeature()) {
			String msg = "The Sort Feature is already installed, cannot add this feature again.";
			throw new RuntimeException(msg);
		}

		this.sortFeatureHandler = new SortFeatureHandler<K, R>(internalTable, sortFeature);
		featureHandlers.add(sortFeatureHandler);
	}

	/**
	 * Determine if the sort feature is installed
	 * 
	 * @return true if the sort feature is installed, false otherwise
	 */
	public boolean hasSortFeature() {
		return (this.sortFeatureHandler != null);
	}

	/**
	 * Get the sort feature, if previously installed
	 * 
	 * @return the sort feature if installed, or null
	 */
	public SortFeature<K, R> getSortFeature() {
		return (hasSortFeature()) ? sortFeatureHandler.getFeature() : null;
	}

	// ========================================================================
	// ===================== DETAIL FEATURE
	// ========================================================================

	/**
	 * Installs the Detail Feature. Ensure that the table is not initialized and that the feature is
	 * not previously installed before calling this method.
	 * 
	 * @param detailFeature the detail feature
	 */
	public void addDetailFeature(DetailFeature<K, R> detailFeature) {
		/* Ensure that initialization has not started */
		validateNotInitialized("addDetailFeature");

		/* Validate that the feature is not already installed */
		if (this.hasDetailFeature()) {
			String msg = "The Detail Feature is already installed, cannot add this feature again.";
			throw new RuntimeException(msg);
		}

		this.detailFeatureHandler = new DetailFeatureHandler<K, R>(internalTable, detailFeature);
		featureHandlers.add(detailFeatureHandler);
	}

	/**
	 * Determine if the detail feature is installed
	 * 
	 * @return true if the detail feature is installed, false otherwise
	 */
	public boolean hasDetailFeature() {
		return (this.detailFeatureHandler != null);
	}

	/**
	 * Get the detail feature, if previously installed
	 * 
	 * @return the detail feature if installed, or null
	 */
	public DetailFeature<K, R> getDetailFeature() {
		return (hasDetailFeature()) ? detailFeatureHandler.getFeature() : null;
	}

	/**
	 * Show the create view generated by the detail feature. The detail feature must be installed
	 * before calling this method.
	 */
	public void showCreateView() {
		detailFeatureHandler.showCreateView();
	}

	/**
	 * Show the detail view generated by the detail feature. The detail feature must be installed
	 * before calling this method.
	 * 
	 * @param The record for which the detail view must be shown
	 */
	public void showDetailView(R record) {
		detailFeatureHandler.showDetailView(record);
	}

	// ========================================================================
	// ===================== BOOKMARK FEATURE
	// ========================================================================

	/**
	 * Install the Bookmark Feature. Ensure that the table is not initialized and that the feature
	 * is not previously installed before calling this method.
	 * 
	 * @param bookmarkFeature the bookmark feature
	 */
	public void addBookmarkFeature(BookmarkFeature feature) {
		/* Ensure that initialization has not started */
		validateNotInitialized("addBookmarkFeature");

		/* Validate that the feature is not already installed */
		if (this.hasBookmarkFeature()) {
			String msg = "The Bookmark Feature is already installed, cannot add this feature again.";
			throw new RuntimeException(msg);
		}

		this.bookmarkFeatureHandler = new BookmarkFeatureHandler<K, R>(internalTable, feature);
		featureHandlers.add(bookmarkFeatureHandler);
	}

	/**
	 * Determine if the bookmark feature is installed
	 * 
	 * @return true if the bookmark feature is installed, false otherwise
	 */
	public boolean hasBookmarkFeature() {
		return (this.bookmarkFeatureHandler != null);
	}

	/**
	 * Get the bookmark feature, if previously installed
	 * 
	 * @return the bookmark feature if installed, or null
	 */
	public BookmarkFeature getBookmarkFeature() {
		return (hasBookmarkFeature()) ? bookmarkFeatureHandler.getFeature() : null;
	}

	/**
	 * Provide the table with the portion of the anchor that is specifically for the table widget.
	 * The method can be called only after the bookmark feature is installed and the table is
	 * initialized.
	 * 
	 * @param anchor The anchor for the table widget
	 */
	public void setAnchor(String anchor) {
		bookmarkFeatureHandler.handleBrowserAnchorChange(anchor);
	}

	/**
	 * Update the anchor for the table by notifying all listeners of the table anchor change events
	 * that the table anchor is changed. If the bookmark feature is installed, the anchor will show
	 * in the browser history.
	 */
	public void updateAnchor() {
		fireAnchorChangeEvent();
	}

	// ========================================================================
	// ===================== EXPANDING FEATURE
	// ========================================================================

	/**
	 * Installs the Expanding Feature. Ensure that the table is not initialized and that the feature
	 * is not previously installed before calling this method.
	 * 
	 * @param expandingFeature the expanding feature
	 */
	private void addExpandingFeature(ExpandingFeature<K, R> expandingFeature) {
		/* Ensure that initialization has not started */
		validateNotInitialized("addExpandingFeature");

		/* Validate that the feature is not already installed */
		if (this.hasExpandingFeature()) {
			String msg = "The Expanding Feature is already installed, cannot add this feature again.";
			throw new RuntimeException(msg);
		}

		this.expandingFeatureHandler = new ExpandingFeatureHandler<K, R>(internalTable,
		        expandingFeature);
		featureHandlers.add(expandingFeatureHandler);
	}

	/**
	 * Determine if the expanding feature is installed
	 * 
	 * @return true if the expanding feature is installed, false otherwise
	 */
	public boolean hasExpandingFeature() {
		return (this.expandingFeatureHandler != null);
	}

	/**
	 * Get the expanding feature, if previously installed
	 * 
	 * @return the expanding feature if installed, or null
	 */
	public ExpandingFeature<K, R> getExpandingFeature() {
		return (hasExpandingFeature()) ? expandingFeatureHandler.getFeature() : null;
	}

	// ========================================================================
	// ===================== TABLE METHODS
	// ========================================================================

	/**
	 * Loads the records using the data provider. Ensure that the table is initialized before
	 * calling this method. Does not change the current view.
	 */
	public void refresh() {
		if (!isInitialized()) {
			throw new RuntimeException("Cannot refresh the table until initialization completes.");
		}

		this.listViewPanel.refresh();
	}

	/**
	 * Reset the table and its features to the initial state. The table must already be initialized
	 * before calling this method. The reset table clear the view displayed by the table to be a
	 * blank view. Ensure that an appropriate view is set after calling reset. Reset will not
	 * re-fetch records from the server. An additional call to refresh the table is required to
	 * update the records.
	 */
	public void reset() {
		reset(true);
	}

	/**
	 * Reset the table to the initial state. The feature data remains unchanged. The table must
	 * already be initialized before calling this method. The reset table clear the view displayed
	 * by the table to be a blank view. Ensure that an appropriate view is set after calling reset.
	 * Reset will not re-fetch records from the server. An additional call to refresh the table is
	 * required to update the records.
	 * 
	 * @param resetFeatures true to reset the feature data as well, false otherwise
	 */
	public void reset(boolean resetFeatures) {
		/* The table must be initialized */
		if (!isInitialized()) {
			String msg = "The table must be initialized before being reset.";
			throw new RuntimeException(msg);
		}

		/* Clear all displayed records in the table, and clear the view widget */
		listViewPanel.clearRecords();
		showView(new LabelWidget(""), null);

		/* Reset the feature handlers */
		if (resetFeatures) {
			for (FeatureHandler<?> handler : featureHandlers)
				handler.reset();
		}
	}

	/**
	 * Determine if the table widget is part of the DOM and is currently visible
	 * 
	 * @return true if the table is part of the HTML DOM and is visible, false otherwise
	 */
	public boolean isDisplayed() {
		if (!this.isAttached())
			return false;

		Widget current = this;
		while (current != null && current.isVisible())
			current = current.getParent();
		return (current == null);
	}

	/**
	 * Get the list of records (rows) displayed in the table
	 * 
	 * @return The displayed data records
	 */
	public List<R> getRecords() {
		return listViewPanel.getRecords();
	}

	/**
	 * Get the list of records that are selected via the check box
	 * 
	 * @return The list of selected records
	 */
	public List<R> getSelectedRecords() {
		return listViewPanel.getSelectedRecords();
	}

	/**
	 * Get the number of data rows in the table
	 * 
	 * @return The data row count
	 */
	public int getDataRowCount() {
		return listViewPanel.getDataRowCount();
	}

	/**
	 * Allows the user to register a widget on the table's action container section. This can
	 * contain buttons that trigger actions on the table (example: create, delete, etc).
	 * 
	 * @param widget The widget to register
	 */
	public void registerWidgetOnActionContainer(Widget widget) {
		listViewPanel.getActionContainer().add(widget);
	}
	
	/**
	 * Get the table action container
	 * 
	 * @return The table action container
	 */ 
	public HorizontalContainer getActionContainer() {
		return listViewPanel.getActionContainer();
	}

	// ========================================================================
	// ===================== INTERNAL TABLE METHODS
	// ========================================================================

	/**
	 * Get the data columns in the table. Does not get the check-box column if present.
	 * 
	 * @return The list of columns in the table
	 */
	List<Column<?, R>> getColumns() {
		return listViewPanel.listVisibleColumns();
	}

	/**
	 * Gets all the widgets in each row for the provided column. Each cell (row, column) can have at
	 * most one widget. Does not get the header row widget.
	 * 
	 * @param column The column to get the widgets for
	 * @return The list of widgets for each of the rows for the given column
	 */
	List<Widget> getWidgetsForColumn(Column<?, R> column) {
		return listViewPanel.listWidgetsForVisibleColumn(column);
	}

	/**
	 * Get the table record that is rendered (in any data column) with the input widget
	 * 
	 * @param widget The widget
	 * @return The table record rendered with that widget
	 */
	R getRecordForWidget(Widget widget) {
		return listViewPanel.getRecordForWidget(widget);
	}

	/**
	 * Get the record that matches the given key
	 * 
	 * @param key The key
	 * @param callback The callback to notify with the matching record is found
	 */
	void getRecordForKey(K key, AsyncCallback<R> callback) {
		dataProvider.getRecord(key, callback);
	}

	/**
	 * Get the widget that represents the column header for the given column
	 * 
	 * @param column The column
	 * @return The header widget
	 */
	
	//HorizontalPanel getHeaderWidgetForColumn(Column<?, R> column) {
	public HorizontalPanel getHeaderWidgetForColumn(Column<?, R> column) {
		return listViewPanel.getHeaderColPanelForColumn(column);
	}

	/**
	 * Get the label widget that contains the heading text for the given column. This label is part
	 * of the header widget for the column
	 * 
	 * @param column The column
	 * @return The heading label
	 */
	
	//Label getHeaderWidgetLabelForColumn(Column<?, R> column) {
	public Label getHeaderWidgetLabelForColumn(Column<?, R> column) {
		return listViewPanel.getHeaderLabelForColumn(column);
	}

	/**
	 * Get the panel that contains the widgets for controlling paging for the table
	 * 
	 * @return The paging main container
	 */
    public SimpleContainer getPaginationPanel() {
		return listViewPanel.getPaginationPanel();
	}

	/**
	 * Get the header row checkbox
	 * 
	 * @return The header row checkbox
	 */
	public CheckBox<R> getHeaderRowCheckBox() {
		return listViewPanel.getHeaderRowCheckBox();
	}

	/**
	 * Get the data rows checkboxes
	 * 
	 * @return The data rows checkboxes
	 */
	public List<CheckBox<R>> getDataRowsCheckBoxes() {
		return listViewPanel.getDataRowsCheckBoxes();
	}

	/**
	 * Set the total record count on the table.
	 * 
	 * @param recordCount The record count
	 */
	void setTotalRecordCount(Integer recordCount) {
		listViewPanel.setTotalRecordCount(recordCount);
	}

	/**
	 * Handle the feature event triggered by a change to the feature state caused by the feature
	 * handler view event
	 * 
	 * @param featureEvent The feature event
	 */
	void handleFeatureEvent(FeatureEvent featureEvent) {
		/* If the view is the list view, refresh the table */
		if (isShowingListView()) {
			refresh();
		}

		/* If an anchor change is required, notify listeners */
		if (featureEvent.isAnchorChangeRequired()) {
			fireAnchorChangeEvent();
		}
	}

	/**
	 * Fire an event indicating that the detail view is being displayed
	 * 
	 * @param event The detail view event
	 */
	void fireDetailViewEvent(TableDetailViewEvent<K, R> event) {
		for (TableViewChangeHandler<K, R> handler : viewHandlers) {
			handler.onShowDetailView(event);
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Fire the records updated event to all registered internal listeners
	 */
	private void handleListViewUpdated() {
		if (hasPageFeature()) {
			/* Ensure that records are present on the page */
			int recordCount = getPageFeature().getRecordCount();
			int pageSize = getPageFeature().getPageSize();
			int dataRows = getDataRowCount();

			/* If showing the total count, make sure that selected page is not after last page */
			if (getPageFeature().showTotalCount()) {
				if (dataRows == 0 && recordCount > 0) {
					/* Try and go to the last page */
					int lastPage = recordCount / pageSize;
					if (recordCount % pageSize > 0)
						++lastPage;
					getPageFeature().setPageNumber(lastPage);
					listViewPanel.setTotalRecordCount(null);
					refresh();
					return;
				} else if (dataRows == 0 && recordCount == 0) {
					getPageFeature().setPageNumber(1);
				}
			}
		}

		/* Each of the feature handlers need to be refreshed after the list view update */
		for (FeatureHandler<?> handler : this.featureHandlers) {
			handler.refresh();
		}
	}

	/*
	 * Fire the anchor change event
	 */
	private void fireAnchorChangeEvent() {
		TableAnchor anchor = this.getTableAnchor();
		TableAnchorChangeEvent event = new TableAnchorChangeEvent(anchor);
		for (TableAnchorChangeHandler handler : anchorChangeHandlers) {
			handler.onAnchorChange(event);
		}
	}

	public void setSelectedForRecord(R record, boolean selected) {
		for (CheckBox<R> checkbox : getDataRowsCheckBoxes()) {
			if (checkbox.getRecord().equals(record)) {
				checkbox.setValue(selected);
			}
		}
	}

	public void setHeaderRowSelected(boolean selected) {
		getHeaderRowCheckBox().setValue(selected);
	}

	/**
	 * Add a listener to be notified when the table is refreshed
	 * 
	 * @param listener the listener
	 */
	public void addRefreshListener(final RefreshListener listener) {
		listViewPanel.addListener(new Listener() {
			@Override
			public void updated() {
				listener.updated();
			}
		});
	}

	public interface RefreshListener {
		void updated();
	}

	TableListViewPanel<K, R> getTableListViewPanel() {
		return this.listViewPanel;
	}

    public Integer getTotalRecordCount() {
        return listViewPanel.getTotalRecordCount();
    }

    public HorizontalContainer getTopPanel() {
        return listViewPanel.getTopPanel();
    }

    public HorizontalContainer getBottomPanel() {
        return listViewPanel.getBottomPanel();
    }
}
