package com.tandbergtv.neptune.widgettoolkit.client.remote;

import java.util.logging.Level;
import java.util.logging.Logger;

import com.google.gwt.http.client.Request;
import com.google.gwt.http.client.RequestBuilder;
import com.google.gwt.http.client.RequestCallback;
import com.google.gwt.http.client.RequestException;
import com.google.gwt.http.client.Response;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.rpc.InvocationException;
import com.google.gwt.user.client.rpc.StatusCodeException;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneException;
import com.tandbergtv.neptune.widgettoolkit.client.util.StringUtils;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ToastWindow;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ToastWindowBuilder;

/**
 * Base class for all {@link AsyncCallback} implementations that are used in Neptune remoting. This base class takes
 * care of processing all thrown {@link NeptuneException} instances that are specific to the Neptune runtime. All other
 * thrown exceptions are passed along. This class has to be used in every single GWT RPC call to the Neptune server.
 * 
 *
 * @param <T>
 */
public abstract class NeptuneAsyncCallback<T> implements AsyncCallback<T> {

	public static final String SESSION_TIMEOUT_FLAG = "session_timeout_error=true";

	private static final int MAX_RETRY_TIME = 1000 * 60 * 30;

	private Logger logger = Logger.getLogger("NeptuneAsyncCallback");

	/**
	 * check it per 5 seconds.
	 */
	private int serviceCheckSleepTime = 5000;

	private long totalRetryTime = serviceCheckSleepTime;

	private Timer serviceCheckTimer;

	private ToastWindow toastWindow;
	
	
	private static int serviceUnhandleRPCCounter = 0;

	/**
	 * Overwrite the onFailure method from AsyncCall
	 */
	@Override
	public final void onFailure(Throwable caught) {
		NeptuneApplication app = NeptuneApplication.getApplication();
		
		if (caught instanceof StatusCodeException) {
			StatusCodeException statusCodeException = (StatusCodeException) caught;
			int statusCode = statusCodeException.getStatusCode();

			/**
			 * redirect request from oauth provider.
			 */
			if (statusCode == 302) {
				logger.warning("code 302, Sessin timeout, redirect to login page");
				reloadBrowser();
				return;
			}

			/**
			 * Service is not ready or not accessed now. it might be in the restart process . Wait 5 seconds, reload it
			 * again. but we can use request build to check server status firstly before do a reload.
			 * 
			 * 0 happened only in IE When call it's many times, it's a bug of ajax call in IE.
			 */
			if (statusCode == 503 || statusCode == 404 || statusCode == 0) {
				
				/**
				 * To avoid too many RPCs to handle the holding and retrying.
				 * and keep 2 RPCs call to avoid the ajax error in IE in sometime due to an known GWT json issue.
				 */
				if (serviceUnhandleRPCCounter < 2) {
					serviceUnhandleRPCCounter++;
					
					handleUnavailableService();
				}
				return;
			
			}

			if (statusCode == 500 && isInternalServerErrorPage(caught)) {
				logger.log(Level.WARNING, "500 error caused by restart, ignore it");
				return;
			}
		}

		/**
		 * In case the other situation we don't send 302 well.
		 */
		if (caught instanceof InvocationException) {
			String msg = "Fail to execute RPC call with InvocationException: \n" + caught.getMessage();
			if (isLoginPage(caught)) {
				logger.log(Level.WARNING, "Session timeout, redirect to login page");
				reloadBrowser();
				return;
			} else {
				logger.log(Level.SEVERE, msg);
			}
		}

		if (!app.handleThrowable(caught)) {
			onNeptuneFailure(caught);
		}

	}

	private void reloadBrowser() {

		String url = Window.Location.getHref();
		if (url.contains(SESSION_TIMEOUT_FLAG)) {
			Window.Location.reload();
			return;
		}

		String sourceURL = url;
		String hashBookmark = "";
		if (url.contains("#")) {
			sourceURL = url.substring(0, url.indexOf("#"));
			hashBookmark = url.substring(url.indexOf("#"));
		} 

		url = sourceURL + getConcatSymbol(sourceURL) + SESSION_TIMEOUT_FLAG + hashBookmark;
		Window.Location.replace(url);
	}

	/**
	 * The 500 error page is for traditional jsp page not for RPC, so have to ignore it.
	 * 
	 * @param caught
	 * @return
	 */
	private boolean isInternalServerErrorPage(Throwable caught) {
		String msg = StringUtils.trimToEmpty(caught.getMessage());
		return msg.contains("<html") && msg.contains("500 Internal Server Error");
	}

	/**
	 * RPC will truncate the content in IE, only can get the part of content if the page is big. so have to only catch
	 * the top content to check.
	 * 
	 * @param caught
	 * @return
	 */
	private boolean isLoginPage(Throwable caught) {
		String msg = StringUtils.trimToEmpty(caught.getMessage());
		return msg.contains("DTD XHTML") && msg.contains("#loginForm") && msg.contains("j_username");
	}

	/**
	 * Mock A long pulling until the service is available, then re-redirect to login page.
	 */
	private void handleUnavailableService() {

		/**
		 * avoid create multiple instances;
		 */
		closeAlertWidget();

		serviceCheckTimer = new Timer() {

			

			public void run() {
				totalRetryTime += serviceCheckSleepTime;
				
				if (totalRetryTime > MAX_RETRY_TIME) {
					logger.info("Beyond the max try time: totalRetryTime>" + totalRetryTime + ":MAX_RETRY_TIME"); 
					
					closeAlertWidget();
					/**
					 * Generate 503 page from Haproxy.
					 */
					Window.Location.reload();
					return;
				}
				
				/**
				 * we need to add a random key to avoid the browser cache, it happened in IE 11.
				 */
				String testURL = Window.Location.getHref() + getConcatSymbol(Window.Location.getHref())
						+ "rsk=" + System.currentTimeMillis();
				
				RequestBuilder rb = new RequestBuilder(RequestBuilder.GET, testURL);

				try {
					rb.sendRequest(null, addResponseCallback());
				} catch (RequestException e) { // NOSONAR
					logger.info("RequestException : " + e.getMessage());
				}
			}
			
		};

		showServiceUnavailableAlert();
		serviceCheckTimer.scheduleRepeating(serviceCheckSleepTime);
	}
	
	private String getConcatSymbol(String href) {
		return StringUtils.trimToEmpty(href).contains("?") ?  "&" : "?";
	}

	private void closeAlertWidget() {
		if (serviceCheckTimer != null) {
			serviceCheckTimer.cancel();
		}
		if (toastWindow != null) {
			toastWindow.close();
		}
	}

	private RequestCallback addResponseCallback() {
		return new RequestCallback() {

			@Override
			public void onResponseReceived(Request request, Response response) {
				logger.info("Get service check response:" + response.getStatusCode());

				if (response.getStatusCode() == 200) {
					closeAlertWidget();
					reloadBrowser();
					return;
				}
				showServiceUnavailableAlert();
			}

			@Override
			public void onError(Request request, Throwable exception) {
				logger.severe("rbq erro:" + exception.getMessage());
				showServiceUnavailableAlert();
			}
		};
	}

	private void showServiceUnavailableAlert() {
		if (toastWindow == null) {
			String msg = "Retrying request, because service is not available, please wait a while.";
			toastWindow = new ToastWindowBuilder().error(msg, serviceCheckSleepTime + 1000).middle().show();
		} else {
			toastWindow.close();
			toastWindow.show();
		}
	}

	/**
	 * Overwrite the onFailure method from AsyncCall
	 */
	@Override
	public final void onSuccess(T result) {
		onNeptuneSuccess(result);
	}

	public abstract void onNeptuneFailure(Throwable caught);

	public abstract void onNeptuneSuccess(T result);
}
