/*
 * Created on Dec 28, 2011
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.ui.realm.client.tab.role.UiRole;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUser;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserConfiguration;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiService;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.PasswordTextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCancelHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCommitHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * The user form shows the data for a user
 * 
 * @author Vijay Silva
 */
public class UserForm extends Composite implements HasViewCommitHandlers<UiUser>,
		HasViewCancelHandlers {

	/* Widgets */
	private final VerticalPanel mainPanel = new VerticalPanel();
	private FormContainer formContainer = new FormContainer(HasHorizontalAlignment.ALIGN_LEFT);
	private TextBoxWidget userNameBox = new TextBoxWidget();
	private CheckBoxWidget externalAuthCheckBox = null;
	private PasswordTextBoxWidget passwordTextBox1 = null;
	private PasswordTextBoxWidget passwordTextBox2 = null;
	private ListBoxWidget<UiRole> rolesListBox = new ListBoxWidget<UiRole>();
	private ListBoxWidget<UiRole> selectedRolesListBox = new ListBoxWidget<UiRole>();
	private HorizontalPanel rolesBox;
	private ButtonWidget exclude = new ButtonWidget("> ");
	private ButtonWidget include = new ButtonWidget("< ");
	private ButtonWidget excludeAll = new ButtonWidget(">>");
	private ButtonWidget includeAll = new ButtonWidget("<<");
	private TextBoxWidget firstNameTextBox = new TextBoxWidget();
	private TextBoxWidget lastNameTextBox = new TextBoxWidget();
	private CheckBoxWidget isActiveCheckBox = new CheckBoxWidget("", false);
	private TextBoxWidget emailTextBox = new TextBoxWidget();
	private TextBoxWidget employeeIdTextBox = new TextBoxWidget();
	private TextBoxWidget departmentTextBox = new TextBoxWidget();
	private TextBoxWidget locationTextBox = new TextBoxWidget();
	private TextBoxWidget phoneTextBox = new TextBoxWidget();
	private TextBoxWidget extensionTextBox = new TextBoxWidget();
	private ButtonWidget saveButton, cancelButton;
	private BusyIndicator busyIndicator = new BusyIndicator(this);

	/* Services */
	private final RealmConstants constants = GWT.create(RealmConstants.class);
	private UserUiServiceAsync userService;

	/* Constants / Styles */
	private static final String STYLE_NAME = "realm-UserForm";

	/* The state */
	private final UiUserConfiguration configuration;
	private UiUser user;
	private boolean readOnly;
	private boolean rolesLoaded = false;

	/* view */
	ResizableContainer container;

	/**
	 * Constructor
	 * 
	 * @param configuration
	 *            The user configuration
	 * @param user
	 *            The user to view
	 * @param roles
	 *            The supported roles
	 */
	public UserForm(UiUserConfiguration configuration, UiUser user, ResizableContainer container) {
		this(configuration, user, false, container);
	}

	/**
	 * Constructor
	 * 
	 * @param configuration
	 *            The user configuration
	 * @param user
	 *            The user to view
	 * @param readOnly
	 *            Show the user data in read-only mode or not
	 */
	public UserForm(UiUserConfiguration configuration, UiUser user, boolean readOnly,
			ResizableContainer container) {
		this.configuration = configuration;
		this.container = container;
		initWidget(this.mainPanel);

		if (user == null)
			user = new UiUser();

		this.user = user;
		this.readOnly = readOnly;

		initialize();
	}

	// ========================================================================
	// ===================== EVENT HANDLERS
	// ========================================================================

	@Override
	public HandlerRegistration addViewCommitHandler(ViewCommitHandler<UiUser> handler) {
		return addHandler(handler, ViewCommitEvent.getType());
	}

	@Override
	public HandlerRegistration addViewCancelHandler(ViewCancelHandler handler) {
		return addHandler(handler, ViewCancelEvent.getType());
	}

	// ========================================================================
	// ===================== FORM API
	// ========================================================================

	/**
	 * Get the user entity being edited by this view. The view may be showing
	 * different data based on the user edits.
	 * 
	 * @return The user entity being edited by the view.
	 */
	public UiUser getUser() {
		return this.user;
	}

	/**
	 * Determine if the view is read only or not
	 * 
	 * @return True if read-only, false otherwise
	 */
	public boolean isReadOnly() {
		return readOnly;
	}

	/**
	 * Reset the form to the initial state in the user entity
	 */
	public void reset() {
		clearErrorMessages();
		updateView();
	}

	/**
	 * Show the error messages, using the append flag to indicate if the
	 * previously displayed error messages should be appended to or cleared.
	 * 
	 * @param messages
	 *            The messages to display
	 * @param append
	 *            Flag to indicate if the previous messages should be appended
	 *            to or cleared
	 */
	public void showErrorMessages(List<String> messages, boolean append) {
		if (!append)
			clearErrorMessages();

		/* Show the error messages */
		for (String message : messages)
			showErrorMessage(message, true);
	}

	/**
	 * Show the error message, using the append flag to indicate if the
	 * previously displayed error messages should be appended to or cleared.
	 * 
	 * @param message
	 *            The message to display
	 * @param append
	 *            Flag to indicate if the previous messages should be appended
	 *            to or cleared
	 */
	public void showErrorMessage(String message, boolean append) {
		if (!append)
			clearErrorMessages();

		container.setErrorMessage(message);
	}

	/**
	 * Clear any displayed error messages
	 */
	public void clearErrorMessages() {
		container.reset();
	}

	// ========================================================================
	// ===================== INTERNAL METHODS
	// ========================================================================

	/*
	 * Initialize the widgets displayed in this form
	 */
	private void initialize() {
		/* Set the style */
		mainPanel.addStyleName(STYLE_NAME);
		// userNameBox.setStyleName(TableConstants.STYLE_FORM_TEXTBOX);

		/* Add the user name form field */
		boolean userNameRequired = !readOnly && !isExternalUser() && isNewUser();
		formContainer.addRow(constants.userName(), userNameBox, userNameRequired);

		/*
		 * In hybrid mode, the user can choose to create internal / external
		 * user
		 */
		boolean showExternalAuthentication = canShowExternalAuthenticationCheckbox();
		if (showExternalAuthentication) {
			externalAuthCheckBox = new CheckBoxWidget();
			externalAuthCheckBox.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
				@Override
				public void onValueChange(ValueChangeEvent<Boolean> event) {
					handleExternalAuthCheckBoxValueChange(event.getValue());
				}
			});
			formContainer.addRow(constants.externalAuthentication(), externalAuthCheckBox);
		}

		/* Add the password fields only if they can be shown */
		if ((showPasswordFields() || showExternalAuthentication) && !isExternalLdapUser()) {
			/* If this user is being created, password is required */
			boolean passwordRequired = isNewUser();
			passwordTextBox1 = new PasswordTextBoxWidget();
			passwordTextBox1.setStyleName(TableConstants.STYLE_FORM_TEXTBOX);
			formContainer.addRow(constants.password(), passwordTextBox1, passwordRequired);

			passwordTextBox2 = new PasswordTextBoxWidget();
			passwordTextBox2.setStyleName(TableConstants.STYLE_FORM_TEXTBOX);
			formContainer.addRow(constants.reenterPassword(), passwordTextBox2, passwordRequired);
		}

		/* Add the role form field */
		boolean roleRequired = !readOnly && !user.isAdminUser();
		rolesBox = createDialogBox();
		formContainer.addRow("Roles", rolesBox, roleRequired);

		/* Add the required form fields */
		boolean required = !readOnly && !isExternalUser();
		formContainer.addRow(constants.firstName(), firstNameTextBox, required);
		formContainer.addRow(constants.lastName(), lastNameTextBox, required);
		formContainer.addRow(constants.email(), emailTextBox, required);

		/* Add the optional form fields */

		formContainer.addRow(constants.isActive(), isActiveCheckBox);
		formContainer.addRow(constants.employeeId(), employeeIdTextBox);
		formContainer.addRow(constants.department(), departmentTextBox);
		formContainer.addRow(constants.location(), locationTextBox);
		formContainer.addRow(constants.phone(), phoneTextBox);
		formContainer.addRow(constants.extension(), extensionTextBox);

		/* Add the save button */
		if (!readOnly) {
			saveButton = new ButtonWidget(constants.saveButton(), new ClickHandler() {
				public void onClick(ClickEvent event) {
					/*
					 * Validate the form, make sure that the basic validation
					 * passes
					 */
					handleSaveButtonClick();
				}
			});
			saveButton.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
			saveButton.addStyleName(StyleNames.STYLE_EB_BTN);
			saveButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
			saveButton.addStyleName(StyleNames.STYLE_SAVE_BUTTON_ICON);
			container.addButton(constants.userTab(), saveButton);
		}

		/* Add the cancel button */
		this.cancelButton = new ButtonWidget(constants.cancelButton(), new ClickHandler() {
			public void onClick(ClickEvent event) {
				handleCancelButtonClick();
			}
		});

		cancelButton.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		cancelButton.addStyleName(StyleNames.STYLE_EB_BTN);
		container.addButton(constants.userTab(), cancelButton);
		mainPanel.add(formContainer);

		/* Add the roles */
		loadRoles();

		/* Update the view */
		updateView();

		/* resize */
		updateSize();
	}

	/*
	 * Check if the user being viewed is an external user
	 */
	private boolean isExternalUser() {
		return (user.getExternalKey() != null && !user.getExternalKey().isEmpty());
	}

	/*
	 * Check if this user is a new user being created
	 */
	private boolean isNewUser() {
		return !user.getKey().isValueAssigned();
	}

	/*
	 * Check if the external authentication check box can be shown
	 */
	private boolean canShowExternalAuthenticationCheckbox() {
		/*
		 * Show check box only in hybrid mode when create internal is allowed
		 * and not read only
		 */
		if (readOnly || !configuration.isHybridMode()
				|| !configuration.isCreateInternalUserAllowed())
			return false;

		/* Can show check box for new users that are not external users */
		return (isNewUser() && !isExternalUser());
	}

	/*
	 * Determine if the password field(s) can be displayed
	 */
	private boolean showPasswordFields() {
		/* Do not show password when the view is read-only */
		if (this.readOnly)
			return false;

		/* There is an external key for the user, do not show password field */
		if (isExternalUser())
			return false;
		
		/* Do not show password for saved externally authenticated users */
		if (user.isExternallyAuthenticated())
			return false;

		return true;
	}

	/*
	 * Load the roles from the server
	 */
	private void loadRoles() {
	/*	if (readOnly || user.isAdminUser()) {
			List<UiRole> list = new ArrayList<UiRole>();
			list.addAll(user.getRoles());
			handleRolesLoaded(null, rolesListBox);
			handleRolesLoaded(list, selectedRolesListBox);
		} else {
			 Get roles from server 
		*/	UserUiServiceAsync service = GWT.create(UserUiService.class);
			service.getRoles(new NeptuneAsyncCallback<List<UiRole>>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					Window.alert(caught.getLocalizedMessage());
					handleRolesLoaded(null, rolesListBox);
					updateView();
				}

				@Override
				public void onNeptuneSuccess(List<UiRole> result) {
					rolesLoaded = true;
					List<UiRole> list = new ArrayList<UiRole>();
					list.addAll(user.getRoles());
					result.removeAll(list);
					handleRolesLoaded(result, rolesListBox);
					handleRolesLoaded(list, selectedRolesListBox);
					updateView();
				}
			});
		//}
	}

	/*
	 * Handle the roles loaded from the server
	 */
	private void handleRolesLoaded(List<UiRole> roles, ListBoxWidget<UiRole> lbox) {
		if (roles != null && !roles.isEmpty()) {

			/* Clear out the roles and add them again */
			lbox.clear();
			for (UiRole role : roles)
				lbox.addItem(role.getName(), role);
		}
		lbox.setEnabled(canEnableRolesListBox());
	}

	/*
	 * Update the view with the provided state
	 */
	private void updateView() {
		/* Update the view with the provided user data */
		userNameBox.setText(user.getUserName());
		if (externalAuthCheckBox != null) {
			externalAuthCheckBox.setValue(user.isExternallyAuthenticated());
		}
		if (passwordTextBox1 != null) {
			passwordTextBox1.setText("");
			passwordTextBox2.setText("");
		}
		firstNameTextBox.setText(user.getFirstName());
		lastNameTextBox.setText(user.getLastName());
		isActiveCheckBox.setValue(user.isActive());
		emailTextBox.setValue(user.getEmail());
		employeeIdTextBox.setValue(user.getEmployeeId());
		departmentTextBox.setValue(user.getDepartment());
		locationTextBox.setValue(user.getLocation());
		phoneTextBox.setValue(user.getPhone());
		extensionTextBox.setValue(user.getExtension());

		/* Enable / disable the fields based on state */
		userNameBox.setEnabled(!isExternalUser() && isNewUser());
		if (externalAuthCheckBox != null) {
			handleExternalAuthCheckBoxValueChange(externalAuthCheckBox.getValue());
		}

		rolesListBox.setEnabled(canEnableRolesListBox());
		selectedRolesListBox.setEnabled(canEnableRolesListBox());
		excludeAll.setEnabled(canEnableRolesListBox());
		includeAll.setEnabled(canEnableRolesListBox());
		exclude.setEnabled(canEnableRolesListBox());
		include.setEnabled(canEnableRolesListBox());
		boolean enabled = !readOnly && !isExternalUser() && !isExternalLdapUser();
		firstNameTextBox.setEnabled(enabled);
		lastNameTextBox.setEnabled(enabled);
		emailTextBox.setEnabled(enabled);
		isActiveCheckBox.setEnabled(enabled);
		employeeIdTextBox.setEnabled(enabled);
		departmentTextBox.setEnabled(enabled);
		locationTextBox.setEnabled(enabled);
		phoneTextBox.setEnabled(enabled);
		extensionTextBox.setEnabled(enabled);
		saveButton.setEnabled(enabled);
	}

	/*
	 * Determine if the roles list box can be enabled
	 */
	private boolean canEnableRolesListBox() {
		return (!readOnly && !user.isAdminUser() && rolesLoaded && !isExternalLdapUser());
	}

	/*
	 * Get the roles selected in the widget
	 */
	private List<UiRole> getSelectedRoles(ListBoxWidget<UiRole> lbox) {
		List<UiRole> selectedRoles = new ArrayList<UiRole>();
		for (int index = 0; index < lbox.getItemCount(); index++) {
			if (lbox.isItemSelected(index)) {
				selectedRoles.add(lbox.getItem(index));
			}
		}

		return selectedRoles;
	}

	/*
	 * Update the selected roles
	 */
	private void addSelectedRoles(List<UiRole> selectedRoles, ListBoxWidget<UiRole> lbox) {
		/* Update the selected roles */
		if (selectedRoles != null && !selectedRoles.isEmpty()) {
			for (UiRole role : selectedRoles) {
				lbox.addItem(role.getName(), role);
			}
		}
	}

	private void removeSelectedRoles(List<UiRole> selectedRoles, ListBoxWidget<UiRole> lbox) {
		if (selectedRoles != null && !selectedRoles.isEmpty()) {
			for (UiRole role : selectedRoles) {
				lbox.removeItem(role);
			}
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/*
	 * Cancel button clicked, fire event indicating this
	 */
	private void handleCancelButtonClick() {
		/* Fire event indicating cancel was clicked */
		fireEvent(new ViewCancelEvent());
		container.reset();
	}

	/*
	 * Save button clicked, save the user data
	 */
	private void handleSaveButtonClick() {
		/* clear any previous errors */
		clearErrorMessages();

		/* validate */
		if (!isFormValid())
			return;

		/* Build the user to save */
		UiUser userToSave = new UiUser();

		/* Get the user name */
		String userName = userNameBox.getText();
		if (userName != null)
			userName = userName.trim();
		userToSave.setUserName(userName);

		/* Set the external authentication flag */
		if (externalAuthCheckBox != null) {
			userToSave.setExternallyAuthenticated(externalAuthCheckBox.getValue());
		} else {
			userToSave.setExternallyAuthenticated(user.isExternallyAuthenticated());
		}

		/* Get the password */
		String password = null;
		if (passwordTextBox1 != null && passwordTextBox1.isEnabled())
			password = passwordTextBox1.getText();
		if (!isBlankPassword(password)) {
			userToSave.setPassword(password);
			userToSave.setChangePassword(true);
		}

		/* Get the selected role(s) */
		userToSave.getRoles().addAll(selectedRolesListBox.getAllItems());

		/* Get the other fields */
		userToSave.setFirstName(firstNameTextBox.getText());
		userToSave.setLastName(lastNameTextBox.getText());
		userToSave.setEmail(emailTextBox.getText());
		userToSave.setActive(isActiveCheckBox.getValue());
		userToSave.setEmployeeId(employeeIdTextBox.getText());
		userToSave.setDepartment(departmentTextBox.getText());
		userToSave.setLocation(locationTextBox.getText());
		userToSave.setPhone(phoneTextBox.getText());
		userToSave.setExtension(extensionTextBox.getText());

		/* Set the 'invisible' fields */
		userToSave.setId(user.getId());
		userToSave.setExternalKey(user.getExternalKey());
		userToSave.setAdminUser(user.isAdminUser());

		/* Save the user */
		busyIndicator.center();
		getUserService().saveUser(userToSave, new NeptuneAsyncCallback<UiUser>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				handleSaveUserFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiUser result) {
				busyIndicator.hide();
				handleSaveUserSuccess(result);
			}
		});
	}

	private HorizontalPanel createDialogBox() {

		// Create a table to layout the content
		HorizontalPanel dialogContents = new HorizontalPanel();
		dialogContents.setWidth("50em");
		dialogContents.setHeight("15em");
		dialogContents.setSpacing(10);

		// Add some text to the top of the dialog
		selectedRolesListBox.setStyleName(TableConstants.STYLE_ROLES_LISTBOX);
		selectedRolesListBox.setVisibleItemCount(10);

		VerticalPanel multiBoxPanel = new VerticalPanel();
		multiBoxPanel.setSpacing(10);
		multiBoxPanel.add(new HTML("Applied Roles"));
		multiBoxPanel.add(selectedRolesListBox);
		dialogContents.add(multiBoxPanel);
		
		include.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		include.addStyleName(TableConstants.STYLE_ARROW_BUTTON);
		include.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				List<UiRole> selectedRoles = getSelectedRoles(rolesListBox);
				removeSelectedRoles(selectedRoles, rolesListBox);
				addSelectedRoles(selectedRoles, selectedRolesListBox);

			}
		});
		
		includeAll.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		includeAll.addStyleName(TableConstants.STYLE_ARROW_BUTTON);
		includeAll.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				List<UiRole> allItems = new ArrayList<UiRole>(rolesListBox.getAllItems());
				removeSelectedRoles(allItems, rolesListBox);
				addSelectedRoles(allItems, selectedRolesListBox);

			}
		});

		exclude.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		exclude.addStyleName(TableConstants.STYLE_ARROW_BUTTON);
		exclude.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				List<UiRole> selectedRoles = getSelectedRoles(selectedRolesListBox);
				removeSelectedRoles(selectedRoles, selectedRolesListBox);
				addSelectedRoles(selectedRoles, rolesListBox);

			}
		});
		excludeAll.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
		excludeAll.addStyleName(TableConstants.STYLE_ARROW_BUTTON);
		excludeAll.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				List<UiRole> allItems = new ArrayList<UiRole>(selectedRolesListBox.getAllItems());
				removeSelectedRoles(allItems,selectedRolesListBox );
				addSelectedRoles(allItems, rolesListBox);
			}
		});

		VerticalPanel buttonsPanel = new VerticalPanel();
		buttonsPanel.setSpacing(10);
		buttonsPanel.add(new HTML(""));
		buttonsPanel.add(include);
		buttonsPanel.add(includeAll);
		buttonsPanel.add(exclude);
		buttonsPanel.add(excludeAll);
		dialogContents.add(buttonsPanel);

		rolesListBox.setStyleName(TableConstants.STYLE_ROLES_LISTBOX);
		rolesListBox.setVisibleItemCount(10);

		VerticalPanel multiBoxPanel1 = new VerticalPanel();
		multiBoxPanel1.setSpacing(10);
		multiBoxPanel1.add(new HTML("Excluded Roles"));
		multiBoxPanel1.add(rolesListBox);

		dialogContents.add(multiBoxPanel1);
		
		return dialogContents;
	}

	/*
	 * Check if the form is valid. If invalid, the error messages are added to
	 * the error panel
	 */
	private boolean isFormValid() {
		if (passwordTextBox1 != null && passwordTextBox1.isEnabled()) {
			String password1 = passwordTextBox1.getText();
			String password2 = passwordTextBox2.getText();
			if (!same(password1, password2)) {
				showErrorMessage(constants.passwordsDontMatchError(), false);
				passwordTextBox1.setText("");
				passwordTextBox2.setText("");
				return false;
			}
		}

		return true;
	}

	/*
	 * Determine if two string values are the same
	 */
	private boolean same(String s1, String s2) {
		if (s1 == null)
			return (s2 == null);
		else if (s2 == null)
			return false;
		else
			return s1.equals(s2);
	}

	/*
	 * Check if the password is blank
	 */
	private boolean isBlankPassword(String password) {
		return (password == null || password.length() == 0);
	}

	/*
	 * Get the user service
	 */
	private UserUiServiceAsync getUserService() {
		if (userService == null) {
			userService = GWT.create(UserUiService.class);
		}

		return userService;
	}
	
	private boolean isExternalLdapUser() {
		return (!isNewUser() && user.isExternalLdapUser());
	}

	/*
	 * Handle the value change in the external authentication check box
	 */
	private void handleExternalAuthCheckBoxValueChange(Boolean value) {
		boolean passwordAllowed = (value == null || !value.booleanValue());

		if (passwordTextBox1 != null) {
			passwordTextBox1.setText("");
			passwordTextBox1.setEnabled(passwordAllowed);
			int index = formContainer.getFieldRowIndex(passwordTextBox1);
			formContainer.setFieldRowVisible(index, passwordAllowed);

			passwordTextBox2.setText("");
			passwordTextBox2.setEnabled(passwordAllowed);
			index = formContainer.getFieldRowIndex(passwordTextBox2);
			formContainer.setFieldRowVisible(index, passwordAllowed);
		}

		/* Externally authenticated users must always be active */
		if (!passwordAllowed)
			isActiveCheckBox.setValue(true);
		isActiveCheckBox.setEnabled(passwordAllowed);
	}

	/*
	 * Handle the failure callback when saving the user
	 */
	private void handleSaveUserFailure(Throwable error) {
		/* Do not change the view, just update the error message panel */
		if (error instanceof ValidationException) {
			StringBuffer sb = new StringBuffer();
			ValidationException ve = (ValidationException) error;
			for (String message : ve.getValidationMessages()) {
				sb.append(message);
				sb.append("<br>");
			}
			container.setErrorHTML(sb.toString());
		} else {
			container.setErrorMessage(error.getLocalizedMessage());
		}
	}

	/*
	 * Handle the success callback when saving the user
	 */
	private void handleSaveUserSuccess(UiUser savedUser) {
		this.user = savedUser;
		updateView();

		/* Update the userName field */
		int index = formContainer.getFieldRowIndex(userNameBox);
		formContainer.setFieldRequired(index, false);

		/*
		 * Remove the external authentication check box field and password
		 * fields if required
		 */
		if (externalAuthCheckBox != null && !canShowExternalAuthenticationCheckbox()) {
			formContainer.removeRow(formContainer.getFieldRowIndex(externalAuthCheckBox));
			externalAuthCheckBox = null;

			/* Remove the password fields if required */
			if (passwordTextBox1 != null && !showPasswordFields()) {
				formContainer.removeRow(formContainer.getFieldRowIndex(passwordTextBox1));
				formContainer.removeRow(formContainer.getFieldRowIndex(passwordTextBox2));
				passwordTextBox1 = null;
				passwordTextBox2 = null;
			}
		}

		if (passwordTextBox1 != null) {
			index = formContainer.getFieldRowIndex(passwordTextBox1);
			formContainer.setFieldRequired(index, false);
			index = formContainer.getFieldRowIndex(passwordTextBox2);
			formContainer.setFieldRequired(index, false);
		}

		/* Fire an event indicating that the user is saved */
		this.fireEvent(new ViewCommitEvent<UiUser>(this.user));
	}

	public ButtonWidget getCancelButton() {
		return cancelButton;
	}

	public ButtonWidget getSaveButton() {
		return saveButton;
	}

	public void updateSize() {
		if (container != null) {
			Scheduler.get().scheduleDeferred(new Scheduler.ScheduledCommand() {

				@Override
				public void execute() {
					container.updateSize();

				}
			});
		}
	}
}