package com.tandbergtv.neptune.ui.realm.client.tab.role;

import static com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants.STYLE_DATACELL_NUMERIC;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.ui.realm.client.RealmComponentPermissions;
import com.tandbergtv.neptune.ui.realm.client.i18n.RealmConstants;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.BookmarkFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AbstractDetailFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.BookmarkFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.CookieStoreBasedPageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelEllipsedView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelIntegerView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class RoleTableProvider {
    private final RoleUiServiceAsync roleService = GWT.create(RoleUiService.class);

    private final RoleDataProvider dataProvider;
    private final PageFeature pageFeature;
    private final RoleDetailFeature detailFeature;
    private final SortFeatureImpl<UiRoleKey, RoleRecord> sortFeature;
    private final BookmarkFeature bookmarkFeature;

    public static final int NAME_COLUMN_MAX_WIDTH = 35;
    public static final int DESCRIPTION_COLUMN_MAX_WIDTH = 50;

    private final Table<UiRoleKey, RoleRecord> table;
    private final AnchorChangeListener anchorChangeListener;

    /**
     * Constructor
     */
    public RoleTableProvider() {
        dataProvider = new RoleDataProvider();
        table = new Table<UiRoleKey, RoleRecord>(dataProvider);
        table.setWidth("100%");

        pageFeature = new CookieStoreBasedPageFeatureImpl("roleList");
        detailFeature = new RoleDetailFeature();
        sortFeature = new SortFeatureImpl<UiRoleKey, RoleRecord>(dataProvider.nameColumn,
                SortOrder.ASCENDING);
        sortFeature.getSortableColumns().add(dataProvider.nameColumn);
        bookmarkFeature = new BookmarkFeatureImpl();

        table.addDetailFeature(detailFeature);
        table.addSortFeature(sortFeature);
        table.addPageFeature(pageFeature);
        table.addBookmarkFeature(bookmarkFeature);

        anchorChangeListener = new RoleMenuAnchorChangeListener(this);
    }

    /**
     * Set the anchor change notifier for the table
     *
     * @param anchorChangeNotifier The anchor change notifier
     */
    public void setAnchorChangeNotifier(AnchorChangeListener anchorChangeNotifier) {
        table.setAnchorChangeNotifier(anchorChangeNotifier);
    }

    /**
     * Get the table
     *
     * @return The table
     */
    public Table<UiRoleKey, RoleRecord> getTable() {
        return table;
    }

    public final class RoleDataProvider implements DataProvider<UiRoleKey, RoleRecord> {
        private final List<Column<?, RoleRecord>> columns;
        private ColumnBase<String, RoleRecord> nameColumn;
        private ColumnBase<Integer, RoleRecord> userCountColumn;
        private ColumnBase<String, RoleRecord> descriptionColumn;
        private ColumnBase<Integer, RoleRecord> permissionsColumn;

        public RoleDataProvider() {
            columns = new ArrayList<Column<?, RoleRecord>>();
            nameColumn = new ColumnBase<String, RoleRecord>("name", "Role Name") {
                @Override
                public View<String> getView(RoleRecord record) {
                    return new LabelEllipsedView(record.getRole().getName(), NAME_COLUMN_MAX_WIDTH);
                }

                @Override
                public String getColumnWidth() {
                    return "35%";
                }
            };

            columns.add(nameColumn);


            descriptionColumn = new ColumnBase<String, RoleRecord>("description", "Description") {
                @Override
                public View<String> getView(RoleRecord record) {
                    return new LabelEllipsedView(record.getRole().getDescription(), DESCRIPTION_COLUMN_MAX_WIDTH);
                }
                @Override
                public String getColumnWidth() {
                    return "35%";
                }
            };
            columns.add(descriptionColumn);
            userCountColumn = new ColumnBase<Integer, RoleRecord>("userCount", "Number of Users") {
                @Override
                public View<Integer> getView(RoleRecord record) {
                    return new LabelIntegerView(record.getRole().getUserCount());
                }
                @Override
                public String getColumnWidth() {
                    return "15%";
                }
            };
            userCountColumn.setCellStyle(STYLE_DATACELL_NUMERIC);
            columns.add(userCountColumn);

            permissionsColumn = new ColumnBase<Integer, RoleRecord>("permissions",
                    "Number of Permissions") {
                @Override
                public View<Integer> getView(RoleRecord record) {
                    return new LabelIntegerView(record.getRole().getPermissionIds().size());
                }
                @Override
                public String getColumnWidth() {
                    return "15%";
                }
            };
            permissionsColumn.setCellStyle(STYLE_DATACELL_NUMERIC);
            columns.add(permissionsColumn);
        }

        @Override
        public List<Column<?, RoleRecord>> getColumns() {
            return columns;
        }

        @Override
        public void getRecords(final AsyncCallback<List<RoleRecord>> callback) {
            int length = pageFeature.getPageSize();
            int start = (pageFeature.getPageNumber() - 1) * length;
            if (start < 0) {
                start = 0;
            }
            roleService.listRoles(start, length, sortFeature.getSortColumnName(),
                    sortFeature.isAscending(), new NeptuneAsyncCallback<UiRoleList>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    pageFeature.setRecordCount(-1);
                    callback.onFailure(caught);
                }

                @Override
                public void onNeptuneSuccess(UiRoleList result) {
                    List<RoleRecord> records = new ArrayList<RoleRecord>();
                    for (UiRole role : result.getRoles()) {
                        records.add(new RoleRecord(role));
                    }
                    pageFeature.setRecordCount(result.getTotal());
                    callback.onSuccess(records);
                }
            });
        }

        @Override
        public void initialize(final AsyncCallback<Void> callback) {
            roleService.getModules(new NeptuneAsyncCallback<List<UiModule>>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    callback.onFailure(caught);
                }

                @Override
                public void onNeptuneSuccess(List<UiModule> modules) {
                    detailFeature.modules = modules;

                    callback.onSuccess(null);
                }
            });
        }

        @Override
        public boolean isCheckboxEnabled() {
            return ClientAuthorizationManager.isAuthorized(RealmComponentPermissions.ROLE_DELETE);
        }

        @Override
        public boolean isRecordCountEnabled() {
            return true;
        }

        @Override
        public void getRecord(UiRoleKey key, final AsyncCallback<RoleRecord> callback) {
            roleService.getRole(key, new NeptuneAsyncCallback<UiRole>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    callback.onFailure(caught);
                }

                @Override
                public void onNeptuneSuccess(UiRole result) {
                    callback.onSuccess(new RoleRecord(result));
                }

            });
        }

    }

    public final class RoleDetailFeature extends AbstractDetailFeature<UiRoleKey, RoleRecord> {
        protected List<UiModule> modules;
        private RealmConstants constants = GWT.create(RealmConstants.class);
        private RoleRecordDetailView roleRecordDetailView;
        private boolean isSaveAllowed;

        protected final class RoleRecordDetailView implements DetailView<UiRoleKey, RoleRecord> {
            private final RoleRecord record;

            private ButtonWidget backButton;

            private final LabelWidget nameLabel;
            private final TextBoxWidget nameBox;

            private final LabelWidget descriptionLabel;
            private final TextBoxWidget descriptionBox;

            private ModuleTable moduleTable;
            private ResizableContainer listContainer;

            //private final FormContainer formContainer = new FormContainer( HasHorizontalAlignment.ALIGN_LEFT);
            private final VerticalContainer contentContainer = new VerticalContainer();
            private final VerticalContainer mainContainer = new VerticalContainer();

            public void setListViewContainer(ResizableContainer container){
                listContainer = container;
            }
            private DetailViewCallback<UiRoleKey, RoleRecord> callback;

            public RoleRecordDetailView(final RoleRecord record, final DetailViewCallback<UiRoleKey, RoleRecord> callback) {
                this.record = record;
                this.callback = callback;

                backButton = new ButtonWidget("Back to Role Administration");
                backButton.removeStyleName(StyleNames.STYLE_GWT_BUTTON);
                backButton.addStyleName(StyleNames.STYLE_EB_BTN);
                backButton.addStyleName(StyleNames.STYLE_BACK_TO_ROLES_BUTTON_ICON);

                backButton.addClickHandler( new ClickHandler() {

                    @Override
                    public void onClick(ClickEvent event) {
                        cancel();

                    }
                });


                nameLabel = new LabelWidget("Role Name");
                nameLabel.addStyleName("required-post");

                nameBox = new TextBoxWidget();
                // database column size
                nameBox.setMaxLength(100);
                nameBox.addStyleName("role-name");

                HorizontalContainer hcRole = new HorizontalContainer();
                hcRole.setVerticalAlignment(HorizontalContainer.ALIGN_MIDDLE);
                hcRole.add(nameLabel);
                hcRole.add(nameBox);
                hcRole.addStyleName("role-name-cont");

                descriptionLabel = new LabelWidget("Description");
                descriptionLabel.addStyleName("role-descr-label");

                descriptionBox = new TextBoxWidget();
                // database column size
                descriptionBox.setMaxLength(1000);
                descriptionBox.addStyleName("role-descr");

                HorizontalContainer hcDescr = new HorizontalContainer();
                hcDescr.setVerticalAlignment(HorizontalContainer.ALIGN_MIDDLE);
                hcDescr.add(descriptionLabel);
                hcDescr.add(descriptionBox);
                hcDescr.addStyleName("role-descr-cont");

                //modulePanel = new ModulePanel(modules);
                moduleTable = new ModuleTable(modules);


                HorizontalContainer row1 = new HorizontalContainer();
                //row1.setWidth("100%");

                row1.setVerticalAlignment(HorizontalContainer.ALIGN_MIDDLE);

                row1.add(backButton);
                row1.setCellWidth(backButton, "33%");

                row1.add(hcRole);
                row1.setCellHorizontalAlignment(hcRole, HorizontalContainer.ALIGN_CENTER);
                row1.setCellWidth(hcRole, "33%");

                row1.add(hcDescr);
                row1.setCellHorizontalAlignment(hcDescr, HorizontalContainer.ALIGN_RIGHT);
                row1.setCellWidth(hcDescr, "33%");


                row1.addStyleName("role-detail-header");
                //row1.setw

                contentContainer.add(row1);
                contentContainer.add(moduleTable);
                contentContainer.addStyleName("content-container");
                //contentContainer.setWidth("100%");



                //formContainer.addRow("Name:", nameBox, true);
                //formContainer.addRow("Description:", descriptionBox);
                //formContainer.addRow("Permissions:", modulePanel);
                //formContainer.addRow("Permissions:", moduleTable);
                revert();


            }

            public void setSaveAllowed(boolean allowed) {
                isSaveAllowed = allowed;
            }

            public boolean isSaveAllowed(){
                return isSaveAllowed;
            }

            @Override
            public void commit() {
                record.getRole().setName(nameBox.getText().trim());
                record.getRole().setDescription(descriptionBox.getText().trim());
                //record.getRole().setPermissionIds(modulePanel.getData());
                record.getRole().setPermissionIds(moduleTable.getData());
            }

            @Override
            public RoleRecord getRecord() {
                return record;
            }

            @Override
            public void revert() {
                nameBox.setText(record.getRole().getName());
                descriptionBox.setText(record.getRole().getDescription());
                //modulePanel.setData(record.getRole().getPermissionIds());
                moduleTable.setData(record.getRole().getPermissionIds());
            }

            public void save(){
                callback.save(RoleRecordDetailView.this);
            }

            public void cancel(){
                callback.cancel(RoleRecordDetailView.this);
            }

            @Override
            public Widget getWidget() {
                String roleName = record.getRole().getName();
                if (roleName == null) {
                    roleName = "";
                }
                //mainContainer.add(formContainer);
                mainContainer.add(contentContainer);
                mainContainer.setHorizontalAlignment(HasHorizontalAlignment.ALIGN_LEFT);

                return mainContainer;
            }

            @Override
            public void release() {
            }

            @Override
            public String getStyleName() {
                return null;
            }

            @Override
            public void saveFailed(Throwable throwable) {
                if (throwable instanceof ValidationException) {
                    ValidationException ve = (ValidationException) throwable;
                    StringBuffer messagesSB = new StringBuffer();
                    for (String message : ve.getValidationMessages()) {
                        messagesSB.append(message).append("\n");
                    }
                    listContainer.getMessageArea().setVisible(true);
                    listContainer.setErrorHTML(messagesSB.toString());
                    //errorMessageLabel.setText(messagesSB.toString());
                } else {

                    listContainer.setErrorHTML(throwable.getLocalizedMessage());
                    //errorMessageLabel.setText(throwable.getLocalizedMessage());
                }
            }

        }

        /**
         * Constructor
         */
        public RoleDetailFeature() {
            super(new UiRoleKeySerializer());
        }

        public  RoleRecordDetailView getRecordDetailView(){
            return roleRecordDetailView;
        }

        @Override
        public void delete(final List<RoleRecord> records, final AsyncCallback<Void> callback) {
            if (records.size() == 0) {
                callback.onFailure(new RoleDeletionException(constants.noRoleCheckBoxSelected()));
                return;
            }

            /* Request confirmation from the user */
            if (!Window.confirm(constants.roleDeleteConfirmation())) {
                callback.onSuccess(null);
                return;
            }

            List<UiRoleKey> roleKeys = new ArrayList<UiRoleKey>();
            for (RoleRecord record : records) {
                roleKeys.add(record.getRole().getKey());
            }
            roleService.deleteRoles(roleKeys, new NeptuneAsyncCallback<Void>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    callback.onFailure(caught);
                }

                @Override
                public void onNeptuneSuccess(Void result) {
                    callback.onSuccess(null);
                }
            });

        }

        @Override
        public void getNew(AsyncCallback<RoleRecord> callback) {
            callback.onSuccess(new RoleRecord(new UiRole()));
        }


        @Override
        public DetailView<UiRoleKey, RoleRecord> getView(RoleRecord record,
                DetailViewCallback<UiRoleKey, RoleRecord> callback) {
            roleRecordDetailView = new RoleRecordDetailView(record,
                    callback);
            boolean readOnly = record.getRole().isAdministratorRole();
            boolean hasKey = record.getKey().isValueAssigned();
            String permission = (hasKey) ? RealmComponentPermissions.ROLE_MODIFY
                    : RealmComponentPermissions.ROLE_CREATE;
            boolean saveAllowed = !readOnly && ClientAuthorizationManager.isAuthorized(permission);
            roleRecordDetailView.setSaveAllowed(saveAllowed);
            return roleRecordDetailView;
        }

        @Override
        public boolean hasDetailLink(Column<?, RoleRecord> column) {
            if (ClientAuthorizationManager.isAuthorized(RealmComponentPermissions.ROLE_VIEW)) {
                return column.getName().equals("name");
            } else {
                return false;
            }
        }

        @Override
        public void save(final RoleRecord record, final AsyncCallback<Void> callback) {
            final UiRole role = record.getRole();
            roleService.save(role, new NeptuneAsyncCallback<UiRole>() {

                @Override
                public void onNeptuneFailure(Throwable caught) {
                    callback.onFailure(caught);
                }

                @Override
                public void onNeptuneSuccess(UiRole savedRole) {
                    record.updateRole(savedRole);
                    callback.onSuccess(null);
                }
            });
        }

        @Override
        public boolean showCreateButton() {
            return false;
        }

        @Override
        public boolean showDeleteButton() {
            return false;
        }

        public RoleRecordDetailView getRoleRecordDetailView(){
            return roleRecordDetailView;
        }
    }

    public final static class RoleRecord implements Record<UiRoleKey> {
        private UiRole role;

        public RoleRecord(UiRole role) {
            this.role = role;
        }

        public UiRole getRole() {
            return role;
        }

        public void updateRole(UiRole updatedRole) {
            this.role = updatedRole;
        }

        @Override
        public UiRoleKey getKey() {
            return role.getKey();
        }
    }

    public AnchorChangeListener getAnchorChangeListener() {
        return this.anchorChangeListener;
    }

    protected RoleDetailFeature getRoleDetailFeature(){
        return detailFeature;
    }
}
