package com.tandbergtv.neptune.widgettoolkit.client.file;

import java.util.Date;

import com.google.gwt.core.shared.GWT;
import com.google.gwt.json.client.JSONParser;
import com.google.gwt.json.client.JSONValue;
import com.google.gwt.user.client.Timer;
import com.google.gwt.xhr.client.ReadyStateChangeHandler;
import com.google.gwt.xhr.client.XMLHttpRequest;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetConstants;
import com.tandbergtv.neptune.widgettoolkit.client.util.Base64Util;


/**
 * File uploader that provides start, pause, resume and cancel feature. The uploading is based on AJAX calls.
 * <p>
 * Default upload URL: /fileuploader/upload
 * </p>
 *
 * @author evan
 */
public class FileUploader {

    private static NeptuneWidgetConstants CONSTANTS = GWT.create(NeptuneWidgetConstants.class);

    private String providerId;

    private String titleId;

    private String assetId;

    private String accessToken;

    private boolean isAlwaysSendToken;

    private File file;

    private UploadStatusCallback callback;

    private String uploadUrl = "/fileuploader/upload";

    private String uploadLocation;

    private double fileSize;

    private double start;

    private double end;

    private double lastTrunk = 0;

    private double lastLoaded = 0;

    private double lastProgressTime = 0;

    /** 2M */
    private double chunkSize = 2097152;

    private boolean isCancel = false;

    private boolean isCancelled = false;

    private boolean isFinished = false;

    private boolean isPause = false;

    private boolean isPaused = false;

    private boolean isRetrying = false;

    private boolean isStarted = false;

    private XMLHttpRequest xhr;

    private int retries = 3;

    private int interval = 5000;

    /**
     * Recording the current retry times.
     */
    private int retryTimes = 0;

    private Timer retryTimer;

    public FileUploader(File file, UploadStatusCallback callback) {
        this.file = file;
        this.callback = callback;
    }

    public FileUploader(FileInput fileInput, UploadStatusCallback callback) {
        if (fileInput != null) {
            this.file = (File) fileInput.getFiles().get(0);
        }
        this.callback = callback;
    }

    public FileUploader setChunkSize(long chunkSize) {
        this.chunkSize = chunkSize;
        return this;
    }

    public FileUploader setUploadUrl(String uploadUrl) {
        this.uploadUrl = uploadUrl;
        return this;
    }

    public FileUploader setProviderId(String providerId) {
        this.providerId = providerId;
        return this;
    }

    public FileUploader setTitleId(String titleId) {
        this.titleId = titleId;
        return this;
    }

    public FileUploader setAssetId(String assetId) {
        this.assetId = assetId;
        return this;
    }

    public FileUploader setAccessToken(String accessToken) {
        this.accessToken = accessToken;
        return this;
    }

    /**
     * Set the number of retries the uploader should make when uploading an file trunk fail.
     *
     * @param retries
     *            default to 3
     * @return
     */
    public FileUploader setRetries(int retries) {
        this.retries = retries;
        return this;
    }

    /**
     * Get the number of retries the uploader should make when uploading an file trunk fail.
     *
     * @return
     */
    public int getRetries() {
        return retries;
    }

    /**
     * Time in milliseconds between 2 uploading retries.
     *
     * @param interval
     *            default to 5 seconds
     * @return
     */
    public FileUploader setInterval(int interval) {
        this.interval = interval;
        return this;
    }

    /**
     * If set to true, the file uploader will always send the access token in the request even for each file trunk.
     * Otherwise only the start and cancel request will have the token.
     *
     * @param isAlwaysSendToken
     * @return
     */
    public FileUploader setAlwaysSendToken(boolean isAlwaysSendToken) {
        this.isAlwaysSendToken = isAlwaysSendToken;
        return this;
    }

    /**
     * Start the file uploader.
     */
    public void start() {
        isCancelled = false;
        isFinished = false;

        XMLHttpRequest xhr = XMLHttpRequest.create();
        String params = (uploadUrl.indexOf('?') > -1 ? "&" : "?") + "providerId=" + providerId + "&titleId=" + titleId
                + "&assetId=" + assetId;
        xhr.open("POST", uploadUrl + params);
        xhr.setRequestHeader("Authorization", "Bearer " + accessToken);
        xhr.setRequestHeader("X-File-Name", Base64Util.toBase64(file.getName().getBytes()));
        xhr.setRequestHeader("X-File-Size", file.getSize());
        xhr.setRequestHeader("X-File-Type", file.getType());

        xhr.setOnReadyStateChange(new ReadyStateChangeHandler() {

            @Override
            public void onReadyStateChange(XMLHttpRequest xhr) {
                if (xhr.getReadyState() == 4) {
                    if (xhr.getStatus() == 200) {
                        isStarted = true;
                        uploadLocation = xhr.getResponseHeader("Location");
                        sendFirstChunk();
                        return;
                    }

                    if ((xhr.getResponseHeader("Content-Type") != null)
                            && xhr.getResponseHeader("Content-Type").contains("application/json")) {
                        try {
                            JSONValue json = JSONParser.parseStrict(xhr.getResponseText());
                            onError(CONSTANTS.getString(json.isObject().get("errorKey").isString().stringValue()));
                        } catch (Exception e) {
                            onError(xhr.getStatusText());
                        }
                    } else {
                        onError(xhr.getStatusText());
                    }
                }
            }
        });
        addErrorHandler(xhr);
        xhr.send();
    }

    /**
     * Cancel the file uploading. The cancel action will wait till the current trunk transfer to be finished if there's
     * any, then make the cancel request.
     *
     * @param isImmediately
     *            if set to true the cancel action will abort the ongoing uploading if there's any and send the cancel
     *            request.
     */
    public void cancel(boolean isImmediately) {
        if (isFinished || isCancelled) {
            return;
        }
        if (isImmediately && (xhr != null)) {
            xhr.abort();
        }
        isCancel = true;
        callback.onCancel();
        getRetryTimer().cancel();
        if (isImmediately || isPaused || isRetrying) {
            sendCancelRequest();
        }
        resetRetry();
    }

    /**
     * Pause the file uploading.
     *
     * @param isTriggerEvent
     *            The pause event callback will be invoked if set to true.
     */
    public void pause(boolean isTriggerEvent) {
        if (isFinished || isCancelled || isPause) {
            return;
        }
        isPause = true;
        if (isTriggerEvent && (callback != null)) {
            callback.onPause();
        }
        if (isRetrying) {
            getRetryTimer().cancel();
            isPaused = true;
            if (callback != null) {
                callback.onRetryExhausted((lastTrunk) / fileSize);
            }
        }
    }

    /**
     * Resume the paused file upload process.
     */
    public void resume() {
        if (isFinished || isCancelled || !isPaused) {
            return;
        }

        isPause = false;
        isPaused = false;

        // in retry state, just execute the previous retry
        if (isRetrying) {
            callback.onResume((lastTrunk) / fileSize);
            if (retryTimes > 0) {
                retryTimes--;
            }
            onRetry("");
            return;
        }

        // Resume from a pause
        if (callback != null) {
            callback.onResume((end) / fileSize);
        }
        sendNextChunk();
    }

    /**
     * Return true if the file uploader has been started, which means an upload session has been created already.
     *
     * @return
     */
    public boolean isStarted() {
        return isStarted;
    }

    /**
     * Send a cancel HTTP request to the file server.
     */
    private void sendCancelRequest() {
        XMLHttpRequest xhr = XMLHttpRequest.create();
        xhr.open("DELETE", uploadLocation);
        xhr.setRequestHeader("Authorization", "Bearer " + accessToken);
        xhr.setOnReadyStateChange(new ReadyStateChangeHandler() {

            @Override
            public void onReadyStateChange(XMLHttpRequest xhr) {
                if (xhr.getStatus() == 200) {
                    onCancelled();
                } else {
                    onError(xhr.getStatusText());
                }
            }
        });
        addErrorHandler(xhr);
        xhr.send();
        isStarted = false;
        if (callback != null) {
            callback.onCancelSent();
        }
    }

    private void sendFirstChunk() {
        fileSize = Long.parseLong(file.getSize());
        start = 0L;
        end = chunkSize;
        if (end >= fileSize) {
            end = fileSize;
        }
        sendFileChunk();
    }

    /**
     * Set the trunk position forward and send the next trunk.
     */
    private void sendNextChunk() {
        start += chunkSize;
        end += chunkSize;
        if (end >= fileSize) {
            end = fileSize;
        }
        sendFileChunk();
    }

    private void onTransferComplete(String response) {
        updateLastUploaded();
        resetRetry();

        if (isCancel) {
            sendCancelRequest();
            return;
        }

        if (isPause) {
            isPaused = true;
            if (callback != null) {
                callback.onPaused((end) / fileSize);
            }
            return;
        }

        if (end >= fileSize) {
            // Finished
            isFinished = true;
            if (callback != null) {
                callback.onDone(response);
            }
        } else {
            onProgress((end) / fileSize);
            sendNextChunk();
        }
    }

    private void onProgress(double progress) {
        if (callback != null) {
            callback.onProgress(progress);
        }
    }

    private void updateLastUploaded() {
        lastTrunk = end;
    }

    private void onError(String error) {
        callback.onError(error);
    }

    private void onCancelled() {
        isCancelled = true;
        callback.onCancelled();
    }

    private void onRetry(String error) {
        if (isPaused || isCancelled) {
            // if other user operation has been requested then do not retry
            return;
        }

        if (isCancel) {
            sendCancelRequest();
            return;
        }

        if (isPause) {
            isPaused = true;
            if (callback != null) {
                callback.onPaused((end) / fileSize);
            }
            return;
        }

        isRetrying = true;

        if (++retryTimes > retries) {
            retryTimes = 0;
            pause(false);
            return;
        }
        callback.onRetry(retryTimes, error);
        getRetryTimer().schedule(interval);
    }

    private Timer getRetryTimer() {
        if (retryTimer == null) {
            retryTimer = new Timer() {

                @Override
                public void run() {
                    sendFileChunk();
                }
            };
        }
        return retryTimer;
    }

    private void resetRetry() {
        retryTimes = 0;
        isRetrying = false;
    }

    private void onSpeed(double loaded, double timeInMilli) {
        if (loaded < 0) {
            callback.onSpeed(-1);
            return;
        }
        if ((timeInMilli - lastProgressTime) >= 1000) {
            callback.onSpeed(((loaded - lastLoaded) * 1000) / (timeInMilli - lastProgressTime));
            lastLoaded = loaded;
            lastProgressTime = timeInMilli;
        }
    }

    private void resetSpeedData() {
        lastLoaded = 0;
        lastProgressTime = new Date().getTime();
    }

    private native void sendFileChunk() /*-{
        var self = this;
        self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::resetSpeedData()();

        var xhr = self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::xhr = @com.google.gwt.xhr.client.XMLHttpRequest::create()();
        xhr.onload = function() {
            if (xhr.status == 200) {
                self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onTransferComplete(Ljava/lang/String;)(xhr.getResponseHeader('File-URI'));
                return;
            }

            if (xhr.getResponseHeader('Content-Type') && xhr.getResponseHeader('Content-Type').indexOf('application/json') > -1) {
                try {
                    self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onRetry(Ljava/lang/String;)(
                        @com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::CONSTANTS.@com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetConstants::getString(Ljava/lang/String;)(JSON.parse(xhr.responseText).errorKey)
                    );
                } catch (e) {
                    self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onRetry(Ljava/lang/String;)(xhr.statusText);
                }
            } else {
                self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onRetry(Ljava/lang/String;)(xhr.statusText);
            }
        };

        xhr.onerror = function() {
            self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onRetry(Ljava/lang/String;)('Network error');
        };
        xhr.ontimeout = function(e) {
            self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onRetry(Ljava/lang/String;)('Connection timeout');
        };

        xhr.upload.onprogress = function(event) {
            if (!event.lengthComputable) {
                self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onSpeed(DD)(-1, new Date().getTime());
                return;
            }
            self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onSpeed(DD)(event.loaded, new Date().getTime());
        };

        xhr.open('PUT', this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::uploadLocation, true);
        if (this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::isAlwaysSendToken) {
            xhr.setRequestHeader("Authorization", "Bearer " + this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::accessToken);
        }
        xhr.setRequestHeader("Content-Type", "application/octet-stream");
        xhr.setRequestHeader("Content-Range", this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::start + "-" + (this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::end - 1));

        var chunk = this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::file.slice(this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::start, this.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::end);
        xhr.send(chunk);
    }-*/;

    private native void addErrorHandler(XMLHttpRequest xhr) /*-{
        var self = this;
        xhr.onerror = function() {
            self.@com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader::onError(Ljava/lang/String;)('Network error');
        };
    }-*/;

    public static interface UploadStatusCallback {

        /**
         * File uploading completed.
         *
         * @param fileUrl
         */
        void onDone(String fileUrl);

        /**
         * File uploading progress updated.
         *
         * @param progress
         */
        void onProgress(double progress);

        /**
         * File uploading paused.
         *
         * @param progress
         */
        void onPaused(double progress);

        /**
         * File uploading pause request sent.
         */
        void onPause();

        /**
         * File uploading resumed.
         *
         * @param progress
         */
        void onResume(double progress);

        /**
         * File uploading error.
         *
         * @param error
         *            Error information.
         */
        void onError(String error);

        /**
         * File upload cancel request sent.
         */
        void onCancel();

        /**
         * File upload cancel request has been sent.
         */
        void onCancelSent();

        /**
         * File uploading is cancelled.
         */
        void onCancelled();

        /**
         * File uploading retries.
         *
         * @param retryNum
         *            how many times this retry is.
         * @param error
         *            Error message if any
         */
        void onRetry(int retryNum, String error);

        /**
         * All retry fail.
         *
         * @param progress
         */
        void onRetryExhausted(double progress);

        /**
         * Callback for File upload speed, Byte / Second.
         *
         * @param speed
         */
        void onSpeed(double speed);
    }
}
