package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import java.util.Map;

import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.InternalTable;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageNavigationEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageNavigationHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageResizeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.event.PageResizeHandler;

public class PageFeatureHandler<K, R extends Record<K>> extends
AbstractFeatureHandler<K, R, PageFeature> {

    /* State */
    private PageWidget pageWidget = null;

    /* The anchor tokens for paging */
    private static final String PAGE_NUMBER_ANCHOR_TOKEN = "PageNo";
    private static final String PAGE_SIZE_ANCHOR_TOKEN = "PageSize";

    /**
     * Constructor
     *
     * @param table The table
     * @param dataProvider the data provider
     * @param feature The feature
     */
    public PageFeatureHandler(InternalTable<K, R> table, final PageFeature pageFeature) {
        super(table, pageFeature);
    }

    /*
     * Get the paging widget to use based on the state of the feature
     */
    private void updatePageWidget() {
        if (isNewPageWidgetRequired()) {
            /* Build the widget based on the feature state */
            if (!getFeature().showTotalCount()) {
                this.pageWidget = new SimplePageFeatureWidget();
            } else {
                this.pageWidget = new PageFeatureWidget();
            }

            /* Update the widget state */
            this.pageWidget.setInput(getFeature());

            /* Add event handler for the paging events */
            EventHandler handler = new EventHandler();
            this.pageWidget.addPageNavigationHandler(handler);
            this.pageWidget.addPageResizeHandler(handler);

            /* Set the widget in the table */
            getInternalTable().getPaginationPanel().setWidget(this.pageWidget);
        }

        /* Refresh the widget view based on the feature state */
        this.pageWidget.refresh();
    }

    /*
     * Determine if a new page widget is required
     */
    private boolean isNewPageWidgetRequired() {
        if (pageWidget == null) {
            return true;
        }

        boolean showTotalCount = getFeature().showTotalCount();
        if (showTotalCount && (pageWidget.getClass() != PageFeatureWidget.class)) {
            return true;
        }

        if (!showTotalCount && (pageWidget.getClass() != SimplePageFeatureWidget.class)) {
            return true;
        }

        return false;
    }

    @Override
    protected void resetView() {
        refresh();
    }

    @Override
    public void refresh() {
        getInternalTable().setTotalRecordCount(getFeature().getRecordCount());
        updatePageWidget();
    }

    /**
     * @see FeatureHandler#getAnchorTokens()
     */
    @Override
    public Map<String, String> getAnchorTokens() {
        Map<String, String> tokenMap = super.getAnchorTokens();

        /* Only if the table is showing the list view, add these tokens */
        if (getInternalTable().isShowingListView()) {
            /* Put the start page and the page size in the anchor */
            tokenMap.put(PAGE_NUMBER_ANCHOR_TOKEN, Integer.toString(getFeature().getPageNumber()));
            tokenMap.put(PAGE_SIZE_ANCHOR_TOKEN, Integer.toString(getFeature().getPageSize()));
        }

        return tokenMap;
    }

    /**
     * @see FeatureHandler#setAnchor(TableAnchor)
     */
    @Override
    public void setAnchor(TableAnchor anchor) {
        String pageNumberValue = anchor.getTokens().get(PAGE_NUMBER_ANCHOR_TOKEN);
        try {
            int pageNumber = Integer.parseInt(pageNumberValue);
            if (pageNumber > 0) {
                getFeature().setPageNumber(pageNumber);
            }
        } catch (Exception e) {
            // use default page number 1
            getFeature().setPageNumber(1);
        }

        String pageSizeValue = anchor.getTokens().get(PAGE_SIZE_ANCHOR_TOKEN);
        try {
            int pageSize = Integer.parseInt(pageSizeValue);
            if (pageSize > 0) {
                getFeature().setPageSize(pageSize);
            }
        } catch (Exception e) {
            // use the first page size that is available in the options
            getFeature().setPageSize(getFeature().getPageSizes()[0]);
        }
    }

    /**
     * Set the page number text visible in the paging widget
     *
     * @param pageNumber The number to set as text, null to clear
     */
    public void setPageNumberText(Integer pageNumber) {
        if (this.pageWidget != null) {
            this.pageWidget.setPageNumberText(pageNumber);
        }
    }

    /*
     * Update the page feature model. This is triggered by a user event on the widget
     */
    protected void updatePage(int pageNumber, int size) {
        getFeature().setPageNumber(pageNumber);
        getFeature().setPageSize(size);
        this.fireFeatureEvent(new FeatureEvent());
    }

    /*
     * Event Handler for the paging widget
     */
    private class EventHandler implements PageNavigationHandler, PageResizeHandler {

        @Override
        public void onPageNavigation(PageNavigationEvent event) {
            updatePage(event.getNewPageNumber(), event.getPageFeature().getPageSize());
        }

        @Override
        public void onPageResize(PageResizeEvent event) {
            updatePage(1, event.getNewPageSize());
        }
    }
}
