/*
 * Created on Sep 14, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl;

import java.util.HashMap;
import java.util.Map;

import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.InternalTable;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.anchor.TableAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.Feature;

/**
 * The abstract feature handler implementation
 * 
 * @author Vijay Silva
 */
public abstract class AbstractFeatureHandler<K, R extends Record<K>, F extends Feature> implements
        FeatureHandler<F> {

	/* The table */
	private final InternalTable<K, R> internalTable;
	private final F feature;

	/**
	 * Constructor
	 * 
	 * @param table The table that this feature handler is used by
	 */
	public AbstractFeatureHandler(InternalTable<K, R> table, F feature) {
		this.internalTable = table;
		this.feature = feature;
	}

	/**
	 * Get the internal table allowing the feature handler to access the package protected methods
	 * of the table.
	 * 
	 * @return the internal table
	 */
	protected InternalTable<K, R> getInternalTable() {
		return this.internalTable;
	}

	/**
	 * Get the table
	 * 
	 * @return the table
	 */
	protected Table<K, R> getTable() {
		return this.internalTable.getTable();
	}

	/**
	 * @see FeatureHandler#getFeature()
	 */
	@Override
	public F getFeature() {
		return feature;
	}

	/**
	 * @see FeatureHandler#initialize()
	 */
	@Override
	public void initialize() {
	}

	/**
	 * @see FeatureHandler#reset()
	 */
	@Override
	public void reset() {
		feature.reset();
		resetView();
	}

	/**
	 * Reset the view controlled by the feature handler
	 */
	protected void resetView() {
	}

	/**
	 * @see FeatureHandler#refresh()
	 */
	@Override
	public void refresh() {
	}

	/**
	 * Default implementation returns an empty map
	 * 
	 * @see FeatureHandler#getAnchorTokens()
	 */
	@Override
	public Map<String, String> getAnchorTokens() {
		return new HashMap<String, String>();
	}

	/**
	 * @see FeatureHandler#setAnchor(TableAnchor)
	 */
	@Override
	public void setAnchor(TableAnchor anchor) {
	}

	/**
	 * Notify the table that a a new feature event has occurred as a result of changes to the
	 * feature state / model. The change may or may not result in a change of the table anchor, but
	 * will result in the table view being updated.
	 * 
	 * @param anchorUpdateRequired
	 */
	protected void fireFeatureEvent(FeatureEvent event) {
		getInternalTable().handleFeatureEvent(event);
	}

	/**
	 * Determine if the string value is blank
	 * 
	 * @param value The value
	 * @return true if blank, false otherwise
	 */
	protected boolean isBlank(String value) {
		return (value == null || value.trim().length() == 0);
	}
}
