/*
 * Created on Dec 30, 2011
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view;

import com.google.gwt.core.shared.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.impl.images.ExternalUserImages;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;

/**
 * Column for the external user table
 * 
 * @author Vijay Silva
 */
final class UserTableColumn extends ColumnBase<UserRecord, UserRecord> {

	/* Flag indicating that view clicks are being handled */
	private final UserDataProvider dataProvider;
	private boolean handlingViewClick = false;
	private String columnWidth = null;
	private final ExternalUserImages imageBundle;

	/**
	 * Constructor
	 */
	public UserTableColumn(UserDataProvider dataProvider, String name, String displayName) {
		super(name, displayName);
		this.dataProvider = dataProvider;
		this.imageBundle = GWT.create(ExternalUserImages.class);
	}

	@Override
	public View<UserRecord> getView(UserRecord record) {
		return new ColumnView(record);
	}

	@Override
	public String getColumnWidth() {
		return this.columnWidth;
	}

	/**
	 * Set the column width
	 * 
	 * @param width The column width
	 */
	public void setColumnWidth(String width) {
		this.columnWidth = width;
	}

	/**
	 * Determine if this column can handle the click event on the view widget
	 * 
	 * @return true if handling click events on the view, false otherwise
	 */
	public boolean isHandlingViewClick() {
		return this.handlingViewClick;
	}

	/**
	 * Set if this column should handle click events on the record views
	 * 
	 * @param handlingViewClick flag indicating if click events are being handled
	 */
	public void setHandlingViewClick(boolean handlingViewClick) {
		this.handlingViewClick = handlingViewClick;
	}

	/*
	 * Get the data to show in the column view given the record
	 */
	private String getData(UserRecord record) {
		String data = null;
		if (UserDataProvider.USERNAME_COLUMN.equals(getName())) {
			data = record.getUser().getUserName();
		} else if (UserDataProvider.FIRSTNAME_COLUMN.equals(getName())) {
			data = record.getUser().getFirstName();
		} else if (UserDataProvider.LASTNAME_COLUMN.equals(getName())) {
			data = record.getUser().getLastName();
		} else {
			data = "";
		}

		return data;
	}

	/*
	 * Received notification that the view for a record has been clicked for this column
	 */
	private void handleViewClick(UserRecord record) {
		if (!isHandlingViewClick())
			return;

		/* Notify the data provider that the view has been clicked */
		dataProvider.handleRecordClick(record);
	}

	// ========================================================================
	// ===================== COLUMN VIEW FOR RECORD
	// ========================================================================

	/*
	 * Internal class for the column view
	 */
	private class ColumnView implements View<UserRecord>, ClickHandler {

		/* The record */
		private final UserRecord record;

		/*
		 * Constructor
		 */
		ColumnView(UserRecord record) {
			this.record = record;
		}

		@Override
		public Widget getWidget() {
			/* Update the displayed data and return this widget */
			LabelWidget label = new LabelWidget(getData(this.record));
			
			// display tick in external column if user is of type external
			if(UserDataProvider.EXTERNAL_COLUMN.equals(getName()) && 
					(this.record.getUser().isExternalLdapUser() || this.record.getUser().isExternallyAuthenticated())) {
				ImageWidget imageWidget = new ImageWidget();
				imageWidget.setResource(imageBundle.externalUser());
				return imageWidget;
			}			
			
			if (isHandlingViewClick())
				label.addClickHandler(this);
			return label;
		}

		@Override
		public void release() {
			/* Do nothing */
		}

		/*
		 * Use the link style if the column is handling view clicks
		 */
		@Override
		public String getStyleName() {
			return (isHandlingViewClick()) ? TableConstants.STYLE_DATA_LINK : null;
		}

		/*
		 * Handle the click of a widget generated for this view
		 */
		@Override
		public void onClick(ClickEvent event) {
			/* Got a click for a widget generated for this view */
			handleViewClick(this.record);
		}
	}
}
