/*
 * Created on Dec 30, 2011
 * 
 * (C) Copyright Ericsson Television Inc.
 */

package com.tandbergtv.neptune.ui.realm.client.tab.user.view;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUser;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserKey;
import com.tandbergtv.neptune.ui.realm.client.tab.user.model.UiUserList;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiService;
import com.tandbergtv.neptune.ui.realm.client.tab.user.service.UserUiServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.SortFeature;

/**
 * The data provider for the external user entity
 * 
 * @author Vijay Silva
 */
final class UserDataProvider implements DataProvider<UiUserKey, UserRecord> {

	private final UserTableView view;
	private UserUiServiceAsync service;
	private List<Column<?, UserRecord>> columns = null;

	/* The column names */
	public static final String USERNAME_COLUMN = "userName";
	public static final String FIRSTNAME_COLUMN = "firstName";
	public static final String LASTNAME_COLUMN = "lastName";
	public static final String EXTERNAL_COLUMN = "external";
	
	/**
	 * Constructor
	 */
	public UserDataProvider(UserTableView view) {
		this.view = view;
	}

	/*
	 * No initialization required
	 */
	@Override
	public void initialize(AsyncCallback<Void> callback) {
		callback.onSuccess(null);
	}

	/*
	 * Get the columns
	 */
	@Override
	public List<Column<?, UserRecord>> getColumns() {
		if (this.columns == null) {
			/* Build the columns if not previously built */
			this.columns = new ArrayList<Column<?, UserRecord>>();
			UserTableColumn column = null;
			String displayName = null;
		
			/* The user name column */
			displayName = view.getRealmConstants().userNameColumn();
			column = new UserTableColumn(this, USERNAME_COLUMN, displayName);
			column.setHandlingViewClick(true);
			column.setColumnWidth("31%");
			this.columns.add(column);

			/* The first name column */
			displayName = view.getRealmConstants().userFirstNameColumn();
			column = new UserTableColumn(this, FIRSTNAME_COLUMN, displayName);
			column.setColumnWidth("31%");
			this.columns.add(column);

			/* The last name column */
			displayName = view.getRealmConstants().userLastNameColumn();
			column = new UserTableColumn(this, LASTNAME_COLUMN, displayName);
			column.setColumnWidth("30%");
			this.columns.add(column);
			
			/* The external column */
			if(view.getConfiguration().isHybridMode()) {
				displayName = view.getRealmConstants().userExternalColumn();
				column = new UserTableColumn(this, EXTERNAL_COLUMN, displayName);
				column.setColumnWidth("6%");
				this.columns.add(column);
			}
		}

		return this.columns;
	}

	/**
	 * Get a column given the column name
	 * 
	 * @param columnName The column name
	 * @return The matching column, or null if no column matches
	 */
	public Column<?, UserRecord> getColumn(String columnName) {
		for (Column<?, UserRecord> column : getColumns()) {
			if (columnName.equals(column.getName()))
				return column;
		}

		return null;
	}

	@Override
	public void getRecords(final AsyncCallback<List<UserRecord>> callback) {
		PageFeature pageFeature = view.getTable().getPageFeature();
		int pageSize = pageFeature.getPageSize();
		int startIndex = (pageFeature.getPageNumber() - 1) * pageSize;

		SortFeature<UiUserKey, UserRecord> sortFeature = view.getTable().getSortFeature();
		Column<?, UserRecord> sortColumn = sortFeature.getSortColumn();
		String sortByProperty = (sortColumn != null) ? sortColumn.getName() : null;
		boolean isAscending = SortOrder.ASCENDING.equals(sortFeature.getSortOrder());

		/* Get the users */
		view.getBusyIndicator().center();
		getService().listUsers(startIndex, pageSize, sortByProperty, isAscending,
		        new NeptuneAsyncCallback<UiUserList>() {

			        @Override
			        public void onNeptuneFailure(Throwable caught) {
				        view.getBusyIndicator().hide();
				        view.getTable().getPageFeature().setRecordCount(-1);
				        callback.onFailure(caught);
			        }

			        @Override
			        public void onNeptuneSuccess(UiUserList result) {
				        view.getBusyIndicator().hide();
				        List<UserRecord> records = new ArrayList<UserRecord>();
				        for (UiUser user : result.getUsers()) {
				           	records.add(new UserRecord(user));
				        }
					        
				        view.getTable().getPageFeature().setRecordCount(result.getTotal());
				        callback.onSuccess(records);
			        }
		        });
	}

	@Override
	public void getRecord(UiUserKey key, final AsyncCallback<UserRecord> callback) {
		view.getBusyIndicator().center();
		getService().getUser(key, new NeptuneAsyncCallback<UiUser>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.getBusyIndicator().hide();
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UiUser result) {
				view.getBusyIndicator().hide();
				callback.onSuccess(new UserRecord(result));
			}
		});
	}

	@Override
	public boolean isRecordCountEnabled() {
		/* Show the record count */
		return true;
	}

	@Override
	public boolean isCheckboxEnabled() {
		/* Delete is the only batch operation */
		return view.canDeleteRecords();
	}

	/*
	 * Handle notification from the column that a record was clicked
	 */
	void handleRecordClick(UserRecord record) {
		/* Notify the view that the record was clicked */
		view.handleTableRecordClick(record);
	}

	/*
	 * Get the user service
	 */
	private UserUiServiceAsync getService() {
		if (service == null)
			service = GWT.create(UserUiService.class);

		return service;
	}

	public UserTableView getView() {
		return view;
	}
}
