package com.tandbergtv.cms.portal.util.paging;

import java.lang.reflect.Method;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.aspectj.lang.annotation.SuppressAjWarnings;

/**
 * This aspect provides paging support for any use of JPA Query. @PageSource and @PageTarget
 * provide an opt-in mechanism for this aspect. The method that is the target of @PageSource
 * must have a certain signature, see @PageSource for details.
 * 
 * @author trybak
 * 
 */
public aspect PagingAspect {
	private static final Logger LOGGER = Logger.getLogger(PagingAspect.class);

	// In
	private ThreadLocal<Integer> _start = new ThreadLocal<Integer>();
	private ThreadLocal<Integer> _length = new ThreadLocal<Integer>();
	private ThreadLocal<String> _sortBy = new ThreadLocal<String>();
	private ThreadLocal<Boolean> _isAscending = new ThreadLocal<Boolean>();

	// Out
	private ThreadLocal<Integer> _count = new ThreadLocal<Integer>();

	pointcut pageableQueryCreation(EntityManager entityManager,
			String queryString): call(Query EntityManager.createQuery(String)) && args(queryString) && target(entityManager);

	pointcut pagingTargetCall(PageTarget pageTarget): execution(@PageTarget * *.*(..)) && @annotation(pageTarget);

	pointcut pagingSourceCall(PageSource pageSource, int beginIndex,
			int length, String sortBy, boolean isAscending): execution(@PageSource * *.*(int, int, String, boolean)) && args(beginIndex, length, sortBy, isAscending) && @annotation(pageSource);

	// outer call
	@SuppressAjWarnings("adviceDidNotMatch")
	Object around(PageSource pageSource, int start, int length, String sortBy,
			boolean isAscending): pagingSourceCall(pageSource, start, length, sortBy, isAscending)
	{
		// store
		_start.set(start);
		_length.set(length);
		_sortBy.set(sortBy);
		_isAscending.set(isAscending);

		Object returnValue = proceed(pageSource, start, length, sortBy,
				isAscending);

		// set count if available
		try {
			StringBuilder builder = new StringBuilder(pageSource
					.countBeanPropertyName());
			builder.setCharAt(0, Character.toUpperCase(builder.charAt(0)));
			builder.insert(0, "set");
			if (_count.get() != null) {
				Method method = returnValue.getClass().getMethod(
						builder.toString(), int.class);
				method.invoke(returnValue, _count.get());
			}
		} catch (Exception e) {
			LOGGER.error(e);
		}

		// cleanup
		_start.set(null);
		_length.set(null);
		_sortBy.set(null);
		_isAscending.set(null);

		_count.set(null);

		return returnValue;
	}

	// inner call
	@SuppressAjWarnings("adviceDidNotMatch")
	@SuppressWarnings("unchecked")
	Query around(PageTarget pageTarget, EntityManager entityManager,
			String queryString) : pageableQueryCreation(entityManager, queryString) && !within(PagingAspect) && cflow(pagingTargetCall(pageTarget))
	{
		// if the appropriate thread locals are not available, then just proceed
		// with the call
		if (_start.get() == null || _length.get() == null
				|| _sortBy.get() == null || _isAscending.get() == null)
			return proceed(pageTarget, entityManager, queryString);

		// attach ORDER BY and ASC || DESC
		StringBuilder builder = new StringBuilder(queryString);
		builder.append(" ORDER BY ").append(pageTarget.entityName())
				.append('.').append(_sortBy.get()).append(' ');
		if (_isAscending.get())
			builder.append("ASC");
		else
			builder.append("DESC");

		Query query = proceed(pageTarget, entityManager, builder.toString());

		// set paging information
		query.setFirstResult(_start.get());
		query.setMaxResults(_length.get());

		StringBuilder countQueryBuilder = new StringBuilder();
		countQueryBuilder.append("select count(").append(
				pageTarget.entityName()).append(") from ").append(
				pageTarget.entityClass()).append(' ').append(
				pageTarget.entityName()).append(" where ").append(
				pageTarget.entityName()).append(" in (").append(queryString)
				.append(')');
		Query countQuery = entityManager.createQuery(countQueryBuilder
				.toString());

		_count.set(((Long) countQuery.getSingleResult()).intValue());

		// return new query
		return query;
	}
}
