package com.tandbergtv.neptune.widgettoolkit.client.file;

import com.google.gwt.core.shared.GWT;
import com.google.gwt.dom.client.InputElement;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.DragEnterEvent;
import com.google.gwt.event.dom.client.DragEnterHandler;
import com.google.gwt.event.dom.client.DragLeaveEvent;
import com.google.gwt.event.dom.client.DragLeaveHandler;
import com.google.gwt.event.dom.client.DragOverEvent;
import com.google.gwt.event.dom.client.DragOverHandler;
import com.google.gwt.event.dom.client.DropEvent;
import com.google.gwt.event.dom.client.DropHandler;
import com.google.gwt.i18n.client.NumberFormat;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.DockPanel;
import com.google.gwt.user.client.ui.FileUpload;
import com.google.gwt.user.client.ui.FocusPanel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.gwtext.client.widgets.ProgressBar;
import com.tandbergtv.neptune.widgettoolkit.client.file.FileUploader.UploadStatusCallback;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetConstants;
import com.tandbergtv.neptune.widgettoolkit.client.i18n.NeptuneWidgetMessages;
import com.tandbergtv.neptune.widgettoolkit.client.util.FormatUtil;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class FileUploadPopup extends DialogBox {

    private static final String STATUS_ERROR = "error";

    private static final String STATUS_CANCELLED = "cancelled";

    private static final String STATUS_CANCEL = "cancel";

    private static final String STATUS_RESUME = "resume";

    private static final String STATUS_PAUSE = "pause";

    private static final String STATUS_PAUSED = "paused";

    private static final String STATUS_COMPLETE = "complete";

    private static final String STATUS_PROGRESS = "progress";

    private static final String STYLE_PROGRESS_INPROGRESS = "progress-inprogress";

    private static final String STYLE_PROGRESS_COMPLETE = "progress-complete";

    private static final String STYLE_PROGRESS_FAILED = "progress-failed";

    private static final String STYLE_PROGRESS_CANCELLED = "progress-cancelled";

    private static final String STYLE_PROGRESS_PAUSED = "progress-paused";

    private static final String SYTLE_FILEUPLOAD_INFO_CONTAINER = "file-upload-info-container";

    private static final String STYLE_FILE_UPLOAD_POPUP = "file-upload-popup";

    private static final String STYLE_FILE_UPLOAD_CONTAINER = "file-upload-container";

    private static final String STYLE_FILE_UPLOAD_PROGRESSBAR = "file-upload-progressbar";

    private static final String STYLE_DROPZONE = "file-dropzone";

    private static final String STYLE_DROPZONE_HOVER = "file-dropzone-hover";

    private static final NeptuneWidgetConstants CONSTANTS = GWT.create(NeptuneWidgetConstants.class);

    private static final NeptuneWidgetMessages MESSAGES = GWT.create(NeptuneWidgetMessages.class);

    private FocusPanel dropZone;

    private ProgressBar progressBar;

    private FileUpload fileInput;

    private VerticalPanel mainPanel;

    private ButtonWidget pauseButton;

    private ButtonWidget resumeButton;

    private ButtonWidget cancelButton;

    private ButtonWidget okButton;

    private HorizontalContainer buttonPanel;

    private LabelWidget statusLabel;

    private LabelWidget speedLabel;

    private LabelWidget errorMessage;

    private FileUploader fileUploader;

    private File file;

    private String providerId;

    private String titleId;

    private String assetId;

    private String accessToken;

    private FileUploadCallback callback;

    private int retryAttempts = 3;

    private int retryInterval = 5000;

    /** default to 10G */
    private double fileSizeLimit = 10737418240L;

    private String uploadUrl = "/fileuploader/upload";

    public FileUploadPopup(String providerId, String accessToken, FileUploadCallback callback) {
        super(false);

        this.providerId = providerId;
        this.accessToken = accessToken;
        this.callback = callback;

        init();
    }

    public FileUploadPopup(String providerId, String accessToken) {
        this(providerId, accessToken, null);
    }

    public FileUploadPopup setUploadUrl(String uploadUrl) {
        this.uploadUrl = uploadUrl;
        return this;
    }

    public FileUploadPopup setTitleId(String titleId) {
        this.titleId = titleId;
        return this;
    }

    public FileUploadPopup setAssetId(String assetId) {
        this.assetId = assetId;
        return this;
    }

    public FileUploadPopup setRetryAttempts(int retryAttempts) {
        this.retryAttempts = retryAttempts;
        return this;
    }

    /**
     * Set the retry interval in milliseconds.
     *
     * @param retryInterval
     * @return
     */
    public FileUploadPopup setRetryInterval(int retryInterval) {
        this.retryInterval = retryInterval;
        return this;
    }

    public FileUploadPopup setFileSizeLimit(double fileSizeLimit) {
        this.fileSizeLimit = fileSizeLimit;
        return this;
    }

    private void init() {
        setModal(true);
        setGlassEnabled(true);
        setText(CONSTANTS.fileupload());

        fileInput = new FileUpload();
        fileInput.setVisible(false);
        fileInput.addChangeHandler(new ChangeHandler() {

            @Override
            public void onChange(ChangeEvent event) {
                FileInput files = fileInput.getElement().cast();
                if ((files != null) && (files.getFiles() != null) && (files.getFiles().length() > 0)) {
                    processFile((File) files.getFiles().get(0).cast());
                }
            }
        });

        dropZone = createDropZone();
        mainPanel = new VerticalPanel();
        mainPanel.addStyleName(STYLE_FILE_UPLOAD_CONTAINER);
        mainPanel.add(dropZone);

        buttonPanel = createButtons();

        DockPanel dock = new DockPanel();
        dock.setSpacing(4);
        dock.add(mainPanel, DockPanel.NORTH);
        dock.add(buttonPanel, DockPanel.SOUTH);
        dock.setCellHorizontalAlignment(buttonPanel, DockPanel.ALIGN_CENTER);
        dock.add(fileInput, DockPanel.SOUTH);
        dock.addStyleName(STYLE_FILE_UPLOAD_POPUP);

        setWidget(dock);

        statusLabel = new LabelWidget("");
        speedLabel = new LabelWidget("");
        errorMessage = new LabelWidget("");
        errorMessage.setVisible(false);
    }

    private HorizontalContainer createButtons() {
        HorizontalContainer hc = new HorizontalContainer();
        hc.setSpacing(5);

        okButton = new ButtonWidget(CONSTANTS.buttonOk());
        okButton.setStyleName(StyleNames.STYLE_EB_BTN);
        okButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
        okButton.setVisible(false);
        okButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                hide();
            }
        });

        pauseButton = new ButtonWidget(CONSTANTS.buttonPause());
        pauseButton.setStyleName(StyleNames.STYLE_EB_BTN);
        pauseButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
        pauseButton.setVisible(false);
        pauseButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                pauseUpload();
            }
        });

        resumeButton = new ButtonWidget(CONSTANTS.buttonResume());
        resumeButton.setStyleName(StyleNames.STYLE_EB_BTN);
        resumeButton.addStyleName(StyleNames.STYLE_EB_BTN_COLOR_GREEN);
        resumeButton.setVisible(false);
        resumeButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                resumeUpload();
            }
        });

        cancelButton = new ButtonWidget(CONSTANTS.buttonCancel());
        cancelButton.setStyleName(StyleNames.STYLE_EB_BTN);
        cancelButton.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                if ((fileUploader == null) || !fileUploader.isStarted()) {
                    hide();
                    return;
                }
                cancelUpload();
            }
        });

        hc.add(okButton);
        hc.add(pauseButton);
        hc.add(resumeButton);
        hc.add(cancelButton);

        return hc;
    }

    private ProgressBar createProgressBar() {
        ProgressBar pb = new ProgressBar();
        pb.addStyleName(STYLE_FILE_UPLOAD_PROGRESSBAR);
        return pb;
    }

    private FocusPanel createDropZone() {
        final FocusPanel fp = new FocusPanel();
        fp.addStyleName(STYLE_DROPZONE);
        fp.setWidget(createDropZoneLabel());
        fp.addClickHandler(new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                fileInput.getElement().<InputElement> cast().click();
            }
        });

        fp.addDragLeaveHandler(new DragLeaveHandler() {

            @Override
            public void onDragLeave(DragLeaveEvent event) {
                Object target = event.getNativeEvent().getEventTarget();
                if ((target != null) && target.equals(fp.getElement())) {
                    fp.removeStyleName(STYLE_DROPZONE_HOVER);
                }
                event.preventDefault();
                event.stopPropagation();
            }
        });

        fp.addDragEnterHandler(new DragEnterHandler() {

            @Override
            public void onDragEnter(DragEnterEvent event) {
                fp.addStyleName(STYLE_DROPZONE_HOVER);
                event.preventDefault();
                event.stopPropagation();
            }
        });

        fp.addDragOverHandler(new DragOverHandler() {

            @Override
            public void onDragOver(DragOverEvent event) {
                event.preventDefault();
                event.stopPropagation();
            }
        });

        fp.addDropHandler(new DropHandler() {

            @Override
            public void onDrop(DropEvent event) {
                fp.removeStyleName(STYLE_DROPZONE_HOVER);
                event.preventDefault();
                event.stopPropagation();

                FileInput files = event.getDataTransfer().cast();
                if ((files != null) && (files.getFiles() != null)) {
                    if (files.getFiles().length() > 1) {
                        Window.alert(CONSTANTS.fileuploadWarningOneFileAtATime());
                    } else {
                        processFile((File) files.getFiles().get(0).cast());
                    }
                }
            }
        });

        return fp;
    }

    private VerticalPanel createDropZoneLabel() {
        VerticalPanel vp = new VerticalPanel();
        vp.setWidth("100%");
        vp.setHeight("100%");
        LabelWidget label = new LabelWidget(CONSTANTS.fileuploadDialogPrompt());
        vp.add(label);
        vp.setCellVerticalAlignment(label, VerticalPanel.ALIGN_MIDDLE);
        vp.setCellHorizontalAlignment(label, HorizontalContainer.ALIGN_CENTER);
        return vp;
    }

    private void processFile(File file) {
        if (file == null) {
            return;
        }

        if (Long.parseLong(file.getSize()) > fileSizeLimit) {
            Window.alert(MESSAGES.fileuploadMaxFileSize(FormatUtil.formatBytes(fileSizeLimit)));
            return;
        }

        this.file = file;

        dropZone.setVisible(false);
        progressBar = createProgressBar();
        progressBar.setText("0.0%");
        mainPanel.remove(dropZone);

        VerticalPanel vp = new VerticalPanel();
        vp.setSpacing(5);
        vp.addStyleName(SYTLE_FILEUPLOAD_INFO_CONTAINER);
        vp.add(progressBar);
        vp.add(new LabelWidget(MESSAGES.fileuploadProgressFileName(file.getName())));
        vp.add(new LabelWidget(MESSAGES.fileuploadProgressFileType(file.getType())));
        vp.add(new LabelWidget(
                MESSAGES.fileuploadProgressFileSize(FormatUtil.formatBytes(Long.parseLong(file.getSize())))));
        speedLabel.setText(MESSAGES.fileuploadSpeed("--"));
        vp.add(speedLabel);
        statusLabel.setText(MESSAGES.fileuploadStatus(getUploadStatus(STATUS_PROGRESS)));
        vp.add(statusLabel);
        vp.add(errorMessage);

        mainPanel.add(vp);

        startUpload();
    }

    private void startUpload() {
        if (file == null) {
            return;
        }

        pauseButton.setVisible(true);

        fileUploader = new FileUploader(file, new UploadStatusCallback() {

            @Override
            public void onDone(String fileUrl) {
                pauseButton.setVisible(false);
                resumeButton.setVisible(false);
                cancelButton.setVisible(false);
                okButton.setVisible(true);
                updateProgress(1.0);
                updateProgressBarStyle(STYLE_PROGRESS_COMPLETE);
                updateStatus(STATUS_COMPLETE);
                resetSpeed();
                if (callback != null) {
                    callback.onSuccess(fileUrl);
                }
            }

            @Override
            public void onProgress(double progress) {
                updateProgress(progress);
                updateStatus(STATUS_PROGRESS);
            }

            @Override
            public void onPaused(double progress) {
                pauseButton.setVisible(false);
                updateProgress(progress);
                updateProgressBarStyle(STYLE_PROGRESS_PAUSED);
                resetSpeed();
                updateStatus(STATUS_PAUSED);
                resumeButton.setVisible(true);
            }

            @Override
            public void onPause() {
                cleanErrorMessage();
                updateStatus(STATUS_PAUSE);
                pauseButton.setVisible(false);
            }

            @Override
            public void onResume(double progress) {
                resumeButton.setVisible(false);
                updateProgress(progress);
                updateStatus(STATUS_RESUME);
                pauseButton.setVisible(true);
            }

            @Override
            public void onError(String error) {
                pauseButton.setVisible(false);
                resumeButton.setVisible(false);
                cancelButton.setVisible(true);
                updateProgressError(error);
                resetSpeed();
                updateStatus(STATUS_ERROR);
                if (callback != null) {
                    callback.onFailure();
                }
            }

            @Override
            public void onCancel() {
                pauseButton.setVisible(false);
                resumeButton.setVisible(false);
                updateStatus(STATUS_CANCEL);
            }

            @Override
            public void onCancelSent() {
                hide();
            }

            @Override
            public void onCancelled() {
                pauseButton.setVisible(false);
                resumeButton.setVisible(false);
                cancelButton.setVisible(false);
                updateProgress(1.0);
                updateProgressBarStyle(STYLE_PROGRESS_CANCELLED);
                resetSpeed();
                updateStatus(STATUS_CANCELLED);
                if (callback != null) {
                    callback.onCancelled();
                }
            }

            @Override
            public void onRetry(int retryNum, String error) {
                cleanErrorMessage();
                updateErrorMessage(error);
                statusLabel.setText(MESSAGES.fileuploadStatus(MESSAGES.fileuploadRetry(retryNum,
                        fileUploader.getRetries())));
            }

            @Override
            public void onSpeed(double speed) {
                if (speed < 0) {
                    resetSpeed();
                } else {
                    speedLabel.setText(MESSAGES.fileuploadSpeed(FormatUtil.formatBytes((long) speed)) + "/s");
                }
            }

            @Override
            public void onRetryExhausted(double progress) {
                pauseButton.setVisible(false);
                updateProgressBarStyle(STYLE_PROGRESS_PAUSED);
                resetSpeed();
                updateStatus(STATUS_PAUSED);
                resumeButton.setVisible(true);
            }
        }).setProviderId(providerId).setTitleId(titleId).setAssetId(assetId).setUploadUrl(uploadUrl)
        .setRetries(retryAttempts).setInterval(retryInterval).setAccessToken(accessToken);
        fileUploader.start();
    }

    private void resetSpeed() {
        speedLabel.setText(MESSAGES.fileuploadSpeed("--"));
    }

    private void pauseUpload() {
        if (fileUploader == null) {
            return;
        }
        fileUploader.pause(true);
    }

    private void resumeUpload() {
        if (fileUploader == null) {
            return;
        }
        fileUploader.resume();
    }

    private void cancelUpload() {
        if (fileUploader == null) {
            return;
        }
        fileUploader.cancel(true);
    }

    private void updateProgress(double progress) {
        cleanErrorMessage();
        if (progressBar == null) {
            return;
        }
        progressBar.setValue((float) progress);
        progressBar.setText(NumberFormat.getFormat("0.#").format(progress * 100) + "%");
        updateProgressBarStyle(STYLE_PROGRESS_INPROGRESS);
    }

    private void updateProgressError(String error) {
        if (progressBar == null) {
            return;
        }
        progressBar.setValue(1.0F);
        progressBar.setText("");
        updateProgressBarStyle(STYLE_PROGRESS_FAILED);
        updateErrorMessage(error);
    }

    private void updateErrorMessage(String error) {
        if ((error != null) && !"".equals(error.trim())) {
            errorMessage.setText(MESSAGES.fileuploadErrorMessage(error));
            errorMessage.setVisible(true);
        }
    }

    private void updateStatus(String status) {
        if (statusLabel != null) {
            statusLabel.setText(MESSAGES.fileuploadStatus(getUploadStatus(status)));
        }
    }

    private void cleanErrorMessage() {
        errorMessage.setText("");
        errorMessage.setVisible(false);
    }

    private String getUploadStatus(String status) {
        if (STATUS_PROGRESS.equalsIgnoreCase(status) || STATUS_RESUME.equalsIgnoreCase(status)) {
            return "Uploading";
        }
        if (STATUS_ERROR.equalsIgnoreCase(status)) {
            return "Error";
        }
        if (STATUS_COMPLETE.equalsIgnoreCase(status)) {
            return "Complete";
        }
        if (STATUS_PAUSED.equalsIgnoreCase(status)) {
            return "Paused";
        }
        if (STATUS_PAUSE.equalsIgnoreCase(status)) {
            return "Pausing";
        }
        if (STATUS_CANCEL.equalsIgnoreCase(status)) {
            return "Cancelling";
        }
        if (STATUS_CANCELLED.equalsIgnoreCase(status)) {
            return "Cancelled";
        }
        return "Unknown";
    }

    private void updateProgressBarStyle(String style) {
        progressBar.removeClass(STYLE_PROGRESS_CANCELLED);
        progressBar.removeClass(STYLE_PROGRESS_COMPLETE);
        progressBar.removeClass(STYLE_PROGRESS_INPROGRESS);
        progressBar.removeClass(STYLE_PROGRESS_PAUSED);
        progressBar.removeClass(STYLE_PROGRESS_FAILED);
        progressBar.addClass(style);
    }

    /**
     * Interface to implement follow up action of file uploading.
     *
     * @author evan
     */
    public static interface FileUploadCallback {

        void onSuccess(String fileUrl);

        void onFailure();

        void onCancelled();
    }
}
