package com.tandbergtv.cms.pmm.client;

import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.ui.SimplePanel;
import com.tandbergtv.cms.pmm.client.i18n.ContentConstants;
import com.tandbergtv.cms.pmm.client.i18n.ContentMessages;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachHandler;

/**
 * CSS:
 * portlet-alerts-alertRow - horizontal container for each alert
 * portlet-alerts-text - text
 * portlet-alerts-link - hyperlink
 * 
 * @author rprakash
 */
public class AlertListPortlet extends Portlet {
	private NeptuneSecurity security;
	private VerticalContainer container;
	private SimplePanel empty;
	private IListAlertsAsync service;
	private ContentConstants constants;
	private ContentMessages messages;
	private String dateFormat;
	private int maxLength;
	private Timer refreshTimer;
	private int refreshPeriod;
	
	// For planners the string is of the form: 
	// Expected assets for planner dated [date] from [source partner name] are late: [titles]
	// For pitch schedules the string is of the form:
	// Following titles were not distributed successfully for Site [site name] by [date]: [titles] 
	private String TITLE_NAMES_DELIMITER = ":";
	private String TRUNCATED_MSG_SUFFIX = "...";
	private String SITE_BEGINNING_DELIMITER = "Site";
	private String SITE_END_DELIMITER = "by";
	private int ALERT_TEXT_MAX_LENGTH = 255;

	
	private String SCHEDULE_BOOKMARK_URL = "#PMM.Schedule?scheduleId=";
	private String ALERT_ROW_BORDER_STYLE = "border-bottom: outset 1px";
	
	public AlertListPortlet(NeptuneSecurity security, int refreshTimeMillis, String dateFormat, 
			String maxLength) {
	        super( "ALERT_LIST" );
		this.security = security;
		this.refreshPeriod = refreshTimeMillis;
		
		service = GWT.create(IListAlerts.class);
		constants = GWT.create(ContentConstants.class);
		messages = GWT.create(ContentMessages.class);
		this.dateFormat = dateFormat;
		
		this.maxLength = (maxLength != null && !maxLength.isEmpty()) ? Integer.valueOf(maxLength) 
				: ALERT_TEXT_MAX_LENGTH;

		setTitle(constants.alertsTitle());
		setDisplayName( constants.alertsTitle() );
		
		container = new VerticalContainer();
		container.setStyleName("portlet-alerts-alertRow");
		add(container);

		empty = new SimplePanel( new LabelWidget( constants.alertsNoData() ) );
		add( empty );
		
		this.refreshTimer = new Timer() {
			@Override
			public void run() {
				reload();
			}
		};

		EventHandler handler = new EventHandler();
		addDocumentAttachHandler(handler);
		addDocumentDetachHandler(handler);
	}
	
	private void reload() {
		service.getAlerts(new NeptuneAsyncCallback<AlertList>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
			}

			@Override
			public void onNeptuneSuccess(AlertList alertList) {

				setTitle(constants.alertsTitle() + " ("
						+ alertList.getAlerts().size() + "/"
						+ alertList.getTotalAlerts() + ")");
				
				if( alertList.getAlerts().size() > 0 ) {
					/*
					 * Prepare a HTMLWidget which has a table with one column. Each
					 * row represents an alert for a schedule.
					 */
					container.clear();
					HTMLWidget html = new HTMLWidget();
					html.setStyleName("portlet-alerts-text");
					StringBuilder rows = new StringBuilder();				
					for(Alert alert : alertList.getAlerts()) {
						String alertMsg = prepareScheduleHTML(alert);
						rows.append("<tr><td style='" + ALERT_ROW_BORDER_STYLE
								+ "'>" + alertMsg + "</td></tr>");
					}
					html.setHTML("<table>" + rows.toString() + "</table>");
	
					container.add(html);
					
					empty.setVisible(false);
					container.setVisible(true);
				} else {
					empty.setVisible(true);
					container.setVisible(false);
				}
				update();
			}
		});
	}

	/**
	 * Prepares an html string based on whether the alert is for a pitch schedule or a planner.
	 * 
	 * @param alert
	 * @return
	 */
	private String prepareScheduleHTML(Alert alert) {
		String alertMsg;
		String dateLink = "";
		int allowedInvisibleTextLength = 0;

		String titles = getTitles(alert.getText());
		DateTimeFormat dateFormatter = DateTimeFormat.getFormat(dateFormat);
		String date = dateFormatter.format(alert.getDate());
		String siteName = getSiteName(alert);
		
		if (!security.isUserInRole("Schedules_View")) {
			if(alert.isPitchSchedule()) {
				alertMsg = messages.pitchScheduleAlert(siteName, dateLink, titles);
			} else {
				alertMsg = messages.plannerAlert(date, alert.getEntityName(), titles);
			}
		} else {
			dateLink = "<a href='" + SCHEDULE_BOOKMARK_URL + alert.getScheduleId() + "'>" +
				date + "</a>";
			allowedInvisibleTextLength = dateLink.length() - date.length();

			if (alert.isPitchSchedule()) {
				alertMsg = messages.pitchScheduleAlert(siteName, dateLink, titles);
			} else {
				alertMsg = messages.plannerAlert(dateLink, alert.getEntityName(), titles);
			}
		}
		
		/*
		 * If the visible message text (minus the href text) is too long, truncate it and add 
		 * continuation indicator.
		 */
		if(alertMsg.length() - dateLink.length() > maxLength) {
			int visibleTextLength = maxLength - TRUNCATED_MSG_SUFFIX.length()
					+ allowedInvisibleTextLength;
			alertMsg = alertMsg.substring(0, visibleTextLength);
			alertMsg += TRUNCATED_MSG_SUFFIX;
		}
		return alertMsg;
	}

	private String getSiteName(Alert alert) {
		// planner alerts dont have status
		if (!alert.isPitchSchedule()) {
			return "";
		}

		// the status is inside parenthesis e.g. (DISTRIBUTED)
		return alert.getText().substring(alert.getText().indexOf(SITE_BEGINNING_DELIMITER) + SITE_BEGINNING_DELIMITER.length(), alert.getText().indexOf(SITE_END_DELIMITER)).trim();
	}

	// Assumes the titles to be at the end of the message after a ":"
	private String getTitles(String text) {
		return text.substring(text.indexOf(TITLE_NAMES_DELIMITER) + 1);
	}

	/*
	 * Event Handling
	 */
	private final class EventHandler implements DocumentAttachHandler, DocumentDetachHandler {
		@Override
		public void onAttach(DocumentAttachEvent event) {
			refreshTimer.scheduleRepeating(refreshPeriod);
			reload();
		}

		@Override
		public void onDetach(DocumentDetachEvent event) {
			refreshTimer.cancel();
		}
	}
}
