package com.tandbergtv.neptune.widgettoolkit.client.remote;

import java.util.logging.Logger;

import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.rpc.InvocationException;
import com.google.gwt.user.client.rpc.StatusCodeException;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneException;
import com.tandbergtv.neptune.widgettoolkit.client.util.StringUtils;

/**
 * Base class for all {@link AsyncCallback} implementations that are used in Neptune remoting. This base class takes
 * care of processing all thrown {@link NeptuneException} instances that are specific to the Neptune runtime. All other
 * thrown exceptions are passed along. This class has to be used in every single GWT RPC call to the Neptune server.
 * 
 *
 * @param <T>
 */
public abstract class NeptuneAsyncCallback<T> implements AsyncCallback<T> {

	public static final String SESSION_TIMEOUT_FLAG = "session_timeout_error=true";


	private Logger logger = Logger.getLogger("NeptuneAsyncCallback");

	
	
	private RPCServiceCheckHandler handler = new RPCServiceCheckHandler();

	/**
	 * Overwrite the onFailure method from AsyncCall
	 */
	@Override
	public final void onFailure(Throwable caught) {
		NeptuneApplication app = NeptuneApplication.getApplication();
		
		if (caught instanceof StatusCodeException) {
			StatusCodeException statusCodeException = (StatusCodeException) caught;
			int statusCode = statusCodeException.getStatusCode();

			/**
			 * redirect request from oauth provider.
			 */
			if (statusCode == 302) {
				logger.warning("code 302, Sessin timeout, redirect to login page");
				handler.reloadBrowser();
				return;
			}

			/**
			 * Service is not ready or not accessed now. it might be in the restart process . Wait 5 seconds, reload it
			 * again. but we can use request build to check server status firstly before do a reload.
			 * 
			 * get 0 only in IE When call ajax many times, it's a bug of xmlhttprequest object in IE.
			 */
			if (statusCode == 503 || statusCode == 404 || statusCode == 0) {
				
				/**
				 * To avoid too many RPCs to handle the holding and retrying.
				 * and keep 2 RPCs call to avoid the ajax error in IE in sometime due to an known GWT json issue.
				 */
		
				handler.handleUnavailableService();
				return;
			
			}

			if (statusCode == 500 && isInternalServerErrorPage(caught)) {
				logger.warning("500 error caused by restart, ignore it");
				return;
			}
		}

		/**
		 * In case the other situation we don't send 302 well.
		 */
		if (caught instanceof InvocationException) {
			String msg = "Fail to execute RPC call with InvocationException: \n" + caught.getMessage();
			if (isLoginPage(caught)) {
				logger.warning("Session timeout, redirect to login page");
				handler.reloadBrowser();
				return;
			} else {
				logger.severe(msg);
			}
		}

		if (!app.handleThrowable(caught)) {
			onNeptuneFailure(caught);
		}

	}



	/**
	 * The 500 error page is for traditional jsp page not for RPC, so have to ignore it.
	 * 
	 * @param caught
	 * @return
	 */
	private boolean isInternalServerErrorPage(Throwable caught) {
		String msg = StringUtils.trimToEmpty(caught.getMessage());
		return msg.contains("<html") && msg.contains("500 Internal Server Error");
	}

	/**
	 * RPC will truncate the content in IE, only can get the part of content if the page is big. so have to only catch
	 * the top content to check.
	 * 
	 * @param caught
	 * @return
	 */
	private boolean isLoginPage(Throwable caught) {
		String msg = StringUtils.trimToEmpty(caught.getMessage());
		return msg.contains("DTD XHTML") && msg.contains("#loginForm") && msg.contains("j_username");
	}


	/**
	 * Overwrite the onFailure method from AsyncCall
	 */
	@Override
	public final void onSuccess(T result) {
		onNeptuneSuccess(result);
	}

	public abstract void onNeptuneFailure(Throwable caught);

	public abstract void onNeptuneSuccess(T result);
}
