/*
 * Created on Sep 1, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.basic;

import com.google.gwt.dom.client.Document;
import com.google.gwt.dom.client.NativeEvent;
import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.DomEvent;
import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.dom.client.HasBlurHandlers;
import com.google.gwt.event.dom.client.HasChangeHandlers;
import com.google.gwt.event.dom.client.HasFocusHandlers;
import com.google.gwt.event.logical.shared.HasSelectionHandlers;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.event.shared.HasHandlers;
import com.google.gwt.safehtml.shared.SafeHtmlUtils;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasName;
import com.google.gwt.user.client.ui.HasText;
import com.google.gwt.user.client.ui.ListBox;
import com.gwtext.client.core.Template;
import com.gwtext.client.data.FieldDef;
import com.gwtext.client.data.ObjectFieldDef;
import com.gwtext.client.data.Record;
import com.gwtext.client.data.RecordDef;
import com.gwtext.client.data.Store;
import com.gwtext.client.data.StringFieldDef;
import com.gwtext.client.widgets.form.ComboBox;
import com.gwtext.client.widgets.form.Field;
import com.gwtext.client.widgets.form.event.ComboBoxListener;
import com.gwtext.client.widgets.form.event.ComboBoxListenerAdapter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

/**
 * A Combo Box Widget (built using the {@link ComboBox} from GWT-Ext) with typed model.
 * 
 * @author Vijay Silva
 */
public class ComboBoxWidget<ValueType> extends Composite implements HasFocusHandlers,
        HasBlurHandlers, HasChangeHandlers, HasSelectionHandlers<ValueType>, HasHandlers, HasName,
        HasText {

	/* The combo box widget */
	private SimpleContainer container;
	private ComboBox comboBox;
	private RecordDef recordDefinition;
	private ItemKeyGenerator keyGenerator = new ItemKeyGenerator();

	/* The field names */
	private static final String ITEM_NAME_FIELD = "itemName";
	private static final String ITEM_VALUE_FIELD = "itemValue";
	private static final String ITEM_ESCAPE_NAME_FIELD = "itemEscapeName";

	/**
	 * Constructor
	 */
	public ComboBoxWidget() {
		this(-1);
	}

	/**
	 * Constructor
	 * 
	 * @param width The width to set for the combo box
	 */
	public ComboBoxWidget(int width) {
		initializeWidget(width);
		this.initWidget(container);
	}

	/*
	 * Initialize the widget
	 */
	private void initializeWidget(int width) {
		container = new SimpleContainer();
		comboBox = new BasicComboBoxWidget();

		/* Update combo box display / behavior properties */
		//The default template for rendering the items is : <div class="x-combo-list-item">{displayField}<div>
		//If the displayField contains HTML reserved characters, it will not display correctly,
		//so we should render the item with an escaped string.
		comboBox.setTpl(new Template("<div class=\"x-combo-list-item\">{" + ITEM_ESCAPE_NAME_FIELD  +"}</div>"));
		comboBox.setMode(ComboBox.LOCAL);
		comboBox.setTriggerAction(ComboBox.ALL);
		comboBox.setHideLabel(true);
		comboBox.setSelectOnFocus(true);
		comboBox.setTypeAhead(true);
		comboBox.setGrow(true);
		comboBox.setWidth(width);
		comboBox.setResizable(true);

		/* Build the store */
		FieldDef nameFieldDef = new StringFieldDef(ITEM_NAME_FIELD);
		FieldDef escapeNameFieldDef = new StringFieldDef(ITEM_ESCAPE_NAME_FIELD);
		FieldDef valueFieldDef = new ObjectFieldDef(ITEM_VALUE_FIELD);
		FieldDef[] fieldDefinitions = new FieldDef[] { nameFieldDef, escapeNameFieldDef, valueFieldDef };
		recordDefinition = new RecordDef(fieldDefinitions);
		Store store = new Store(recordDefinition);
		comboBox.setStore(store);
		comboBox.setDisplayField(ITEM_NAME_FIELD);

		/* Build the event delegator */
		EventDelegator delegator = new EventDelegator();
		comboBox.addListener(delegator);

		/* Add combo-box to the container */
		container.add(comboBox);
	}

	// ========================================================================
	// ===================== EVENT MANAGEMENT
	// ========================================================================

	@Override
	public HandlerRegistration addFocusHandler(FocusHandler handler) {
		return addDomHandler(handler, FocusEvent.getType());
	}

	@Override
	public HandlerRegistration addBlurHandler(BlurHandler handler) {
		return addDomHandler(handler, BlurEvent.getType());
	}

	@Override
	public HandlerRegistration addChangeHandler(ChangeHandler handler) {
		return addDomHandler(handler, ChangeEvent.getType());
	}

	@Override
	public HandlerRegistration addSelectionHandler(SelectionHandler<ValueType> handler) {
		return addHandler(handler, SelectionEvent.getType());
	}

	// ========================================================================
	// ===================== COMBOBOX METHODS
	// ========================================================================

	/**
	 * Get the name of the input widget
	 * 
	 * @see com.google.gwt.user.client.ui.HasName#getName()
	 * @see ComboBox#getName()
	 */
	@Override
	public String getName() {
		return comboBox.getName();
	}

	/**
	 * Set the name of the widget
	 * 
	 * @see com.google.gwt.user.client.ui.HasName#setName(String)
	 * @see ComboBox#setName(String)
	 */
	@Override
	public void setName(String name) {
		comboBox.setName(name);
	}

	/**
	 * @see ComboBox#getTabindex()
	 */
	public int getTabIndex() {
		return comboBox.getTabindex();
	}

	/**
	 * @see ComboBox#setTabIndex(int)
	 */
	public void setTabIndex(int tabIndex) {
		comboBox.setTabIndex(tabIndex);
	}

	/**
	 * Set the focus on the combo box
	 * 
	 * @see ComboBox#focus()
	 */
	public void focus() {
		comboBox.focus(true);
	}

	/**
	 * Expand the popup with selection options for the combo box
	 */
	public void expand() {
		comboBox.expand();
	}

	/**
	 * Collapse the popup with selection options for the combo box
	 */
	public void collapse() {
		comboBox.collapse();
	}

	/**
	 * Determine if the popup with selection options for the combo box is expanded
	 * 
	 * @return true if expanded, false otherwise
	 */
	public boolean isExpanded() {
		return comboBox.isExpanded();
	}

	/**
	 * Enable or disable the combo box
	 * 
	 * @param enable flag to enable to combo box.
	 * @see ComboBox#setDisabled(boolean)
	 */
	public void setEnabled(boolean enable) {
		comboBox.setDisabled(!enable);
	}

	/**
	 * Determine if this widget is enabled
	 * 
	 * @return true if enabled, false otherwise
	 * @see ComboBox#isDisabled()
	 */
	public boolean isEnabled() {
		return !comboBox.isDisabled();
	}

	/**
	 * @see ComboBox#setEditable(boolean)
	 */
	public void setEditable(boolean editable) {
		comboBox.setEditable(editable);
	}

	/**
	 * Determine if the widget is read only
	 * 
	 * @see ComboBox#isReadOnly()
	 */
	public boolean isReadOnly() {
		return comboBox.isReadOnly();
	}

	/**
	 * @see ComboBox#setReadOnly(boolean)
	 */
	public void setReadOnly(boolean readOnly) {
		comboBox.setReadOnly(readOnly);
	}

	/**
	 * @see ComboBox#getMinHeight()
	 */
	public int getMinHeight() {
		return comboBox.getMinHeight();
	}

	/**
	 * @see ComboBox#setMinHeight(int)
	 */
	public void setMinHeight(int minHeight) {
		comboBox.setMinHeight(minHeight);
	}

	/**
	 * The default value for the minimum characters before auto-complete has been changed to 1
	 * 
	 * @see ComboBox#setMinChars(int)
	 */
	public void setMinChars(int minChars) {
		comboBox.setMinChars(minChars);
	}

	/**
	 * @see ComboBox#getCaretPosition()
	 */
	public int[] getCaretPosition() {
		return comboBox.getCaretPosition();
	}

	/**
	 * @see ComboBox#setCaretPosition(int, int)
	 */
	public void setCaretPosition(int caretStart, int numToSelect) {
		comboBox.setCaretPosition(caretStart, numToSelect);
	}

	/**
	 * @see ComboBox#insertAtCaret(String)
	 */
	public void insertAtCaret(String text) {
		comboBox.insertAtCaret(text);
	}

	/**
	 * @see ComboBox#selectText()
	 */
	public void selectText() {
		comboBox.selectText();
	}

	/**
	 * @see ComboBox#selectText(int, int)
	 */
	public void selectText(int start, int end) {
		comboBox.selectText(start, end);
	}

	/**
	 * @see ComboBox#setMaxLength(int)
	 */
	public void setMaxLength(int maxLength) {
		comboBox.setMaxLength(maxLength);
	}

	/**
	 * @see ComboBox#setMaxLengthText(String)
	 */
	public void setMaxLengthText(String maxLengthText) {
		comboBox.setMaxLengthText(maxLengthText);
	}

	/**
	 * @see ComboBox#setMinLength(int)
	 */
	public void setMinLength(int minLength) {
		comboBox.setMinLength(minLength);
	}

	/**
	 * @see ComboBox#setMinLengthText(String)
	 */
	public void setMinLengthText(String minLengthText) {
		comboBox.setMinLengthText(minLengthText);
	}

	/**
	 * @see ComboBox#getHeight()
	 */
	public int getHeight() {
		return comboBox.getHeight();
	}

	/**
	 * @see ComboBox#setHeight(int)
	 */
	public void setHeight(int height) {
		comboBox.setHeight(height);
	}

	/**
	 * @see ComboBox#setHeight(String)
	 */
	public void setHeight(String height) {
		comboBox.setHeight(height);
	}

	/**
	 * @see ComboBox#getWidth()
	 */
	public int getWidth() {
		return comboBox.getWidth();
	}

	/**
	 * @see ComboBox#setWidth(int)
	 */
	public void setWidth(int width) {
		comboBox.setWidth(width);
	}

	/**
	 * @see ComboBox#setWidth(String)
	 */
	public void setWidth(String width) {
		comboBox.setWidth(width);
	}

	/**
	 * @see ComboBox#setSize(int, int)
	 */
	public void setSize(int width, int height) {
		comboBox.setSize(width, height);
	}

	/**
	 * @see ComboBox#setSize(String, String)
	 */
	public void setSize(String width, String height) {
		comboBox.setSize(width, height);
	}

	/**
	 * @see ComboBox#addClass(String)
	 */
	public void addClass(String cls) {
		comboBox.addClass(cls);
	}

	/**
	 * @see ComboBox#removeClass(String)
	 */
	public void removeClass(String cls) {
		comboBox.removeClass(cls);
	}

	/**
	 * @see ComboBox#setListClass(String)
	 */
	public void setListClass(String listClass) {
		comboBox.setListClass(listClass);
	}

	// ========================================================================
	// ===================== ITEM MANAGEMENT
	// ========================================================================

	/**
	 * Adds an item using the specified name and value.
	 * 
	 * @param itemName The display name for the item
	 * @param item The value for the item
	 */
	public void addItem(String itemName, ValueType item) {
		Record record = createRecordHelper(itemName, item);
		comboBox.getStore().add(record);
		comboBox.autoSize();
	}

	/**
	 * Inserts an item at the specified index.
	 * 
	 * @param itemName The display name for the item
	 * @param item The value for the item
	 * @param index the index at which to insert it
	 */
	public void insertItem(String itemName, ValueType item, int index) {
		Record record = createRecordHelper(itemName, item);
		comboBox.getStore().insert(index, record);
		comboBox.autoSize();
	}

	private Record createRecordHelper(String itemName, ValueType item) {
		String itemKey = keyGenerator.generate();
		String escapeItemName = SafeHtmlUtils.htmlEscape(itemName);
		return createRecord(itemKey, itemName, escapeItemName, item);
	}

	/**
	 * Sets the item (both display name and value) at the specified index overwriting the previous
	 * item at that index.
	 * 
	 * @param index The index of the item being overwritten
	 * @param itemName The new item name
	 * @param item The new item value
	 */
	public void setItem(String itemName, ValueType item, int index) {
		setItemText(index, itemName);
		setItem(index, item);
	}

	/**
	 * Removes the first item in the list box with matching item value, or does nothing if the item
	 * is not present.
	 * 
	 * @param item The item value to match
	 */
	public void removeItem(ValueType item) {
		int index = this.getIndex(item);
		if (index > -1) {
			removeItemAtIndex(index);
		}
	}

	/**
	 * Removes the item at the specified index.
	 * 
	 * @param index the (0-based) index of the item to be removed
	 * @throws IndexOutOfBoundsException if the index is out of range
	 */
	public void removeItemAtIndex(int index) {
		Record record = comboBox.getStore().getAt(index);
		comboBox.getStore().remove(record);
		comboBox.autoSize();
	}

	/**
	 * Removes all items from the combo box.
	 */
	public void clear() {
		/* Reset the key generator and clear all items */
		comboBox.clearValue();
		comboBox.getStore().removeAll();
		keyGenerator.reset();
		comboBox.autoSize();
	}

	/**
	 * Gets the number of items present in the combo box.
	 * 
	 * @return the number of items
	 */
	public int getItemCount() {
		return comboBox.getStore().getCount();
	}

	/**
	 * Gets the text associated with the item at the specified index.
	 * 
	 * @param index the index of the item whose text is to be retrieved
	 * @return the text associated with the item
	 */
	public String getItemText(int index) {
		Record record = comboBox.getStore().getAt(index);
		return record.getAsString(ITEM_NAME_FIELD);
	}

	/**
	 * Sets the text associated with the item at a given index.
	 * 
	 * @param index the index of the item to be set
	 * @param text the item's new text
	 * @see ListBox#setItemText(int, String)
	 */
	public void setItemText(int index, String text) {
		Record record = comboBox.getStore().getAt(index);
		record.beginEdit();
		record.set(ITEM_NAME_FIELD, text);
		record.endEdit();
		comboBox.autoSize();
	}

	/**
	 * Gets the value associated with the item at a given index.
	 * 
	 * @param index the index of the item to be retrieved
	 * @return the item's associated value
	 */
	@SuppressWarnings("unchecked")
	public ValueType getItem(int index) {
		Record record = comboBox.getStore().getAt(index);
		Object item = record.getAsObject(ITEM_VALUE_FIELD);
		return (ValueType) item;
	}

	/**
	 * Sets the value associated with the item at a given index. The display text for this item is
	 * not updated when the item is updated.
	 * 
	 * @param index the index of the item to be set
	 * @param value the new item value
	 */
	public void setItem(int index, ValueType value) {
		Record record = comboBox.getStore().getAt(index);
		record.beginEdit();
		record.set(ITEM_VALUE_FIELD, (Object) value);
		record.endEdit();
	}

	/**
	 * Find the index of the first matching item in the list of items. Does not match the value
	 * typed in by the user which was not added as an item.
	 * 
	 * @param item The item to match
	 * @return The index of the first matching item, or -1 if the item is not present
	 */
	public int getIndex(ValueType item) {
		int itemIndex = -1;
		for (int index = 0; index < getItemCount(); index++) {
			ValueType currentItemValue = getItem(index);

			/* The item could be null */
			if ((item == currentItemValue) || (item != null && item.equals(currentItemValue))) {
				itemIndex = index;
				break;
			}
		}
		return itemIndex;
	}

	/**
	 * Gets the currently selected item index.
	 * 
	 * @return the selected index, or -1 if none is selected, or the user has typed in a new value
	 *         not present as a combo box value
	 */
	public int getSelectedIndex() {
		String value = comboBox.getValue();
		if (value == null)
			return -1;

		int selectedIndex = -1;
		for (int index = 0; index < getItemCount(); index++) {
			String text = getItemText(index);

			/* The item could be null */
			if (value.equals(text)) {
				selectedIndex = index;
				break;
			}
		}
		return selectedIndex;
	}

	/**
	 * Sets the currently selected item index.
	 * 
	 * @param index the index of the item to be selected
	 */
	public void setSelectedIndex(int index) {
		comboBox.setValue(getItemText(index));
	}

	/**
	 * Gets the currently selected item
	 * 
	 * @return the selected item, or null if no item is selected
	 */
	public ValueType getSelectedItem() {
		int index = this.getSelectedIndex();
		return (index != -1) ? getItem(index) : null;
	}

	/**
	 * Set the item as selected. If the item is not found in the list, the selection is cleared.
	 * 
	 * @param item The item to select
	 */
	public void setSelectedItem(ValueType item) {
		int index = getIndex(item);
		if (index > -1) {
			this.setSelectedIndex(index);
		} else {
			comboBox.clearValue();
		}
	}

	/**
	 * Determine if the any of the options specified by the combo box are selected
	 * 
	 * @return true if item in the combo box options is selected, false if a new value is entered.
	 */
	public boolean isItemSelected() {
		return (getSelectedIndex() != -1);
	}

	/**
	 * Get the text displayed in the combo box
	 * 
	 * @return The combo box text
	 */
	@Override
	public String getText() {
		return comboBox.getValue();
	}

	/**
	 * Set the text to display in the combo box. If the text matches the text of any of the options
	 * in the combo box, the option value is selected
	 * 
	 * @param text The text to display
	 */
	@Override
	public void setText(String text) {
		comboBox.setValue(text);
	}

	/**
	 * Clear the text and the selected value displayed by the combo box
	 */
	public void clearText() {
		comboBox.clearValue();
	}

	/*
	 * Create a new record that can be added to the combo box store
	 */
	private Record createRecord(String key, String name, String escapeName, ValueType value) {
		return recordDefinition.createRecord(key, new Object[] { name, escapeName, value });
	}

	// ========================================================================
	// ===================== ITEM KEY MANAGEMENT
	// ========================================================================

	/*
	 * Internal class that maintains the state required for generating keys for combo box items
	 */
	private final class ItemKeyGenerator {
		/* Properties */
		private long itemCounter;

		/* The prefix to use for the item keys */
		private static final String ITEM_KEY_PREFIX = "option-";

		/*
		 * Constructs a new generator
		 */
		public ItemKeyGenerator() {
			reset();
		}

		/*
		 * Generate a new item key
		 */
		String generate() {
			return ITEM_KEY_PREFIX + (++itemCounter);
		}

		/*
		 * Reset the item key generator
		 */
		void reset() {
			itemCounter = 0;
		}
	}

	// ========================================================================
	// ===================== EVENT DELEGATOR
	// ========================================================================

	/*
	 * Event handler delegator from that handles the combo box events
	 */
	private class EventDelegator extends ComboBoxListenerAdapter implements ComboBoxListener {

		@Override
		public void onSelect(ComboBox comboBox, Record record, int index) {
			ValueType selectedItem = getSelectedItem();
			SelectionEvent.fire(ComboBoxWidget.this, selectedItem);
		}

		@Override
		public void onFocus(Field field) {
			NativeEvent nativeEvent = Document.get().createFocusEvent();
			DomEvent.fireNativeEvent(nativeEvent, ComboBoxWidget.this);
		}

		@Override
		public void onBlur(Field field) {
			NativeEvent nativeEvent = Document.get().createBlurEvent();
			DomEvent.fireNativeEvent(nativeEvent, ComboBoxWidget.this);
		}

		@Override
		public void onChange(Field field, Object newVal, Object oldVal) {
			NativeEvent nativeEvent = Document.get().createChangeEvent();
			DomEvent.fireNativeEvent(nativeEvent, ComboBoxWidget.this);
		}
	}
}
