/*
 * Created on Aug 18, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.neptune.widgettoolkit.client.widget.composite;

import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Document;
import com.google.gwt.dom.client.Element;
import com.google.gwt.event.logical.shared.AttachEvent;
import com.google.gwt.event.logical.shared.AttachEvent.Handler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.PopupPanel.PositionCallback;
import com.google.gwt.user.client.ui.UIObject;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.WidgetConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.PopupContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * A widget that works as a pop up and indicates to the user that the client is busy
 * 
 * @author Vijay Silva
 */
public class BusyIndicator extends Composite {

	
	private final SimpleContainer popupWrapper;
	private final PopupPanel glass;
	private final PopupPanel popup;
	private final Widget owningWidget;
	private ImageWidget busyImage;
	private LabelWidget busyLabel;
	private WidgetConstants constants;

	/* Style Names */
	private static final String STYLE_NAME = "nwt-BusyIndicator";
	private static final String GLASS_STYLE_NAME = "nwt-BusyIndicator-Glass";
	private static final String IMAGE_STYLE_NAME = "image";
	private static final String LABEL_STYLE_NAME = "text";
	private static final String IMAGE_URL = "neptune_widget_toolkit/images/busy-indicator.gif";
	
	
	private static final String ID_BUSY_INDICATOR_MESSAGE = "busyIndicatorMessageId";
	private static final String ID_BUSY_INDICATOR_CONTAIN = "BusyIndicatorContainId";
	/**
	 * Creates an modal popup container that displays the progress.
	 */
	public BusyIndicator() {
		this(true);
	}

	/**
	 * Creates an modal popup container that displays the progress and is associated with the owning
	 * widget.
	 * 
	 * @param owningWidget the widget that owns this indicator
	 */
	public BusyIndicator(Widget owningWidget) {
		this(true, owningWidget);
	}

	/**
	 * Creates an popup container that displays the progress.
	 * 
	 * @param modal flag indicating that this popup container should be modal
	 */
	public BusyIndicator(boolean modal) {
		this(modal, null);
	}

	/**
	 * Creates an popup container that displays the progress and is associated with the owning
	 * widget.
	 * 
	 * @param modal flag indicating that this popup container should be modal
	 * @param owningWidget the widget that owns this indicator
	 */
	public BusyIndicator(boolean modal, Widget owningWidget) {
		this.constants = GWT.create(WidgetConstants.class);

		this.glass = new PopupContainer(false, true);
		this.glass.setStyleName(GLASS_STYLE_NAME);

		this.popupWrapper = new SimpleContainer();
		
		this.popup = new PopupContainer(false, modal);
		this.popup.getElement().setId(ID_BUSY_INDICATOR_CONTAIN);
		
		initialize();
		this.initWidget(popupWrapper);

		/* If the owning widget is valid, all listener for attach events */
		this.owningWidget = owningWidget;
		if (this.owningWidget != null) {
			this.owningWidget.addAttachHandler(new Handler() {
				@Override
				public void onAttachOrDetach(AttachEvent event) {
					/* Force hiding of this indicator when the owning widget is detached */
					if (!event.isAttached())
						hide();
				}
			});
		}
	}

	/*
	 * Initialize the contents of this widget
	 */
	private void initialize() {
		/* Use the new style name */
		popup.setStylePrimaryName(STYLE_NAME);

		/* Build the contents of the popup */
		VerticalContainer contents = new VerticalContainer();
		contents.setHorizontalAlignment(HasHorizontalAlignment.ALIGN_CENTER);
		contents.setSpacing(3);
		popup.setWidget(contents);

		/* Build the image to show */
		busyImage = new ImageWidget();
		busyImage.setStylePrimaryName(IMAGE_STYLE_NAME);
		busyImage.setUrl(IMAGE_URL);
		contents.add(busyImage);

		/* Add text below the image */
		busyLabel = new LabelWidget();
		busyLabel.getElement().setId(ID_BUSY_INDICATOR_MESSAGE);
		
		busyLabel.setStylePrimaryName(LABEL_STYLE_NAME);
		contents.add(busyLabel);
		busyLabel.setText(constants.busyIndicatorMessage());
	}

	/**
	 * Get the widget that owns this busy indicator
	 * 
	 * @return the owning widget
	 */
	public Widget getOwningWidget() {
		return this.owningWidget;
	}

	/**
	 * Hide or show the busy message. If hidden, only the image will show
	 * 
	 * @param visible true to show, false to hide the busy message
	 */
	public void setBusyMessageVisible(boolean visible) {
		busyLabel.setVisible(visible);
	}

	/**
	 * Set the busy message to use when showing the busy indicator
	 * 
	 * @param message The message to use
	 */
	public void setBusyMessage(String message) {
		busyLabel.setText(message);
	}
	
	
	/**
	 * Set message color
	 * @param colorName
	 */
	public void setMessageColor(String colorName) {
		busyLabel.getElement().getStyle().setColor(colorName);
	}

	/**
	 * Override the default image with the new image specified by the URL
	 * 
	 * @param imageURL The new image URL
	 */
	public void setBusyImage(String imageURL) {
		busyImage.setUrl(imageURL);
	}

	/**
	 * Center the popup on the browser window and show the popup. If the widget is already showing,
	 * center the widget on the browser window.
	 * 
	 * @see PopupContainer#center()
	 */
	public void center() {
		if (canShow()) {
			showGlassPopup();
			popup.center();
		}
	}

	/**
	 * Shows the popup and attach it to the page.
	 * 
	 * @see PopupContainer#show()
	 */
	public void show() {
		if (canShow()) {
			showGlassPopup();
			popup.show();
		}
	}

	/**
	 * Hide the popup. Has no effect if not already visible.
	 * 
	 * @see PopupContainer#hide()
	 */
	public void hide() {
		glass.hide();
		popup.hide();
	}

	/**
	 * Determine if the popup is showing
	 * 
	 * @see PopupContainer#isShowing()
	 */
	public boolean isShowing() {
		return popup.isShowing();
	}

	/**
	 * Determine if the popup is visible
	 * 
	 * @see PopupContainer#isVisible()
	 */
	@Override
	public boolean isVisible() {
		return popup.isVisible();
	}

	/**
	 * Set the popup to be visible or not
	 * 
	 * @see PopupContainer#setVisible(boolean)
	 */
	@Override
	public void setVisible(boolean visible) {
		popup.setVisible(visible);
	}
	
	/**
	 * Set the zindex to control the diplay layer.
	 * @param index
	 */
	public void setZIndex(int index) {
		getElement().getStyle().setZIndex(index);
	}

	/**
	 * Determine if the popup is modal
	 * 
	 * @see PopupContainer#isModal()
	 */
	public boolean isModal() {
		return popup.isModal();
	}

	/**
	 * Set the modal flag for the popup
	 * 
	 * @see PopupContainer#setModal(boolean)
	 */
	public void setModal(boolean modal) {
		popup.setModal(modal);
	}

	/**
	 * Set the title for the popup
	 * 
	 * @see PopupContainer#getTitle()
	 */
	@Override
	public String getTitle() {
		return popup.getTitle();
	}

	/**
	 * Set the title for the popup
	 * 
	 * @see PopupContainer#setTitle(String)
	 */
	@Override
	public void setTitle(String title) {
		popup.setTitle(title);
	}

	/**
	 * Set the height if the popup
	 * 
	 * @see PopupContainer#setHeight(String)
	 */
	@Override
	public void setHeight(String height) {
		popup.setHeight(height);
	}

	/**
	 * Set the width of the popup
	 * 
	 * @see PopupContainer#setWidth(String)
	 */
	@Override
	public void setWidth(String width) {
		popup.setWidth(width);
	}

	/**
	 * Set the position of the popup
	 * 
	 * @see PopupContainer#setPopupPosition(int, int)
	 */
	public void setPopupPosition(int left, int top) {
		popup.setPopupPosition(left, top);
	}

	/**
	 * Set the position of the popup and show the popup
	 * 
	 * @see PopupContainer#setPopupPositionAndShow(PositionCallback)
	 */
	public void setPopupPositionAndShow(PositionCallback callback) {
		if (canShow()) {
			showGlassPopup();
			popup.setPopupPositionAndShow(callback);
		}
	}

	/**
	 * Show the popup relative to another widget
	 * 
	 * @see PopupContainer#showRelativeTo(UIObject)
	 */
	public void showRelativeTo(UIObject target) {
		if (canShow()) {
			showGlassPopup();
			popup.showRelativeTo(target);
		}
	}

	/*
	 * Determine if this busy indicator can be shown. Only show if the owning widget is attached.
	 */
	private boolean canShow() {
		if (this.owningWidget != null && !this.owningWidget.isAttached()) {
			return false;
		}

		return true;
	}

	/*
	 * Show the glass popup
	 */
	private void showGlassPopup() {
		if (popup.isModal()) {
			glass.show();
		}
	}

	/**
	 * reset the alert message for the live indicator if it exist.
	 * @param msg
	 * @return
	 */
	public static boolean resetAlertMsgForFirstLiveIndicator(String msg) {
		Element element = Document.get().getElementById(ID_BUSY_INDICATOR_CONTAIN);
		if (element != null) {
			Element textLabelWidget = Document.get().getElementById(ID_BUSY_INDICATOR_MESSAGE);
			if (textLabelWidget != null) {
				textLabelWidget.setInnerText(msg);
				textLabelWidget.getStyle().setColor("red");
				element.getParentElement().getStyle().setZIndex(9999);
				return true;
			}
			
		}
		return false;
	}

	public BusyIndicator setAlertMessage(String msg) {
		setBusyMessage(msg);
		setMessageColor("red");
		setZIndex(9999);
		return this;
	}
}
