/**
 * AMSADIManager.java
 * Created on Apr 19, 2007
 * (C) Copyright TANDBERG Television Ltd.
 */
package com.tandbergtv.amsadi;

import java.net.InetAddress;
import java.net.UnknownHostException;

import com.twc.isa.PackageModule.PackageFactory;
import com.twc.isa.PackageModule.PackageFactoryHelper;
import com.twc.isa.PackageModule.PackageHelper;
import com.twc.isa.PackageModule.TransferException;
import com.twc.isa.PackageModule.VersionException;
import com.twc.isa.PackageModule.XmlProcessingException;
import com.twc.isa.ServerModule.AdministrativeState;
import com.twc.isa.ServerModule.DuplicateServant;
import com.twc.isa.ServerModule.InvalidName;
import com.twc.isa.ServerModule.InvalidStateChange;
import com.twc.isa.ServerModule.NameNotFound;
import com.twc.isa.ServerModule.OutOfService;
import com.twc.isa.ServerModule.ServantBase;
import com.twc.isa.ServerModule.ServantBase_impl;
import com.twc.isa.ServerModule.ServantCreateFailed;
import com.twc.isa.ServerModule.UnspecifiedException;

/**
 * AMS ADI utility for package provisioning using ftp protocol
 * 
 * @author Vlada Jakobac
 * 
 */
public class AMSADIManager {

	private static final String ADIXML = "ADI.XML";
	private static final String CL10 = "CableLabsVOD1.0";
	private static final String MOD13 = "MOD1.3";

	private static final String ISA1_2 = "ISA1.2";
	private static final String ISA1_4 = "ISA1.4";

	private static final String FTPPROTOCOL = "ftp";
	
	private ProvisionProfile profile;

	public AMSADIManager(ProvisionProfile profile) {
		this.profile = profile;
	}

	/** 
	 * Provisions the package. It is required that AMS server, FTP server,
	 * and ADI version info are properly set before this method is called.
	 * 
	 * @throws AmsAdiException
	 *             Failure to provision the package
	 */
	public void provisionPackage() throws AmsAdiException {

		if (profile.getAmsServerIP() == null || profile.getAmsPort() == null
				|| profile.getAmsServerIP().trim().length() == 0
				|| profile.getAmsPort().trim().length() == 0) {
			String reason = "AMS server IP and/or port number not set";
			throw new AmsAdiException(reason);
		}
		String nameservice = "corbaloc::" + profile.getAmsServerIP() + ":"
				+ profile.getAmsPort() + "/NameService";
		String[] args = new String[2];
		args[0] = "-ORBInitRef";
		args[1] = "NameService=" + nameservice;
		org.omg.CORBA.ORB orb = null;
		orb = ServantBase_impl.orb_main(args);
		if (orb == null) {
			String reason = "Could not create org.omg.CORBA.ORB object";
			throw new AmsAdiException(reason);
		}

		String[] strFullPath = {"Factories", "PackageFactory"};
		org.omg.CORBA.Object obj = ServantBase_impl.getFactoryObject(orb,
				strFullPath);
		if (obj == null) {
			String reason = "Could not create org.omg.CORBA.Object object";
			throw new AmsAdiException(reason);
		}

		if (profile.getAdiVersion() == null
				|| profile.getAdiVODSpecVersion() == null
				|| profile.getAdiVersion().trim().length() == 0
				|| profile.getAdiVODSpecVersion().trim().length() == 0) {
			String reason = "ADI version and/or ADI VOD specification version not set";
			throw new AmsAdiException(reason);
		}
		String servantName;
		if ((profile.getAdiVersion().equals(ISA1_4))
				&& (!profile.getAdiVODSpecVersion().equals(MOD13))
				&& !profile.getAdiVODSpecVersion().equals(CL10)) {
			
			servantName = profile.getUniqueProviderId() + "/"
					+ profile.getPackageAssetId();
		} else {
			servantName = profile.getProviderName() + "::"
					+ profile.getPackageType() + "::"
					+ profile.getPackageName();
		}

		PackageFactory Packagefactory = PackageFactoryHelper.narrow(obj);
		com.twc.isa.PackageModule.Package package_ = null;
		try {
			if ((profile.getAdiVersion().equals(ISA1_4))
					&& (!profile.getAdiVODSpecVersion().equals(MOD13))
					&& !profile.getAdiVODSpecVersion().equals(CL10)) {
				ServantBase existingServant = null;
				try {
					existingServant = Packagefactory.find(servantName);
				} catch (UnspecifiedException e) {
					String reason = e.message;
					throw new AmsAdiException(reason);
				} catch (NameNotFound e) {
					String reason = e.message;
					throw new AmsAdiException(reason);
				}
				if (existingServant != null) {
					package_ = PackageHelper.narrow(existingServant);
				} else {
					package_ = PackageHelper.narrow(Packagefactory
							.createServant(servantName));
				}
			} else {
				package_ = PackageHelper.narrow(Packagefactory
						.createServant(servantName));
			}
		} catch (ServantCreateFailed e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (UnspecifiedException e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (InvalidName e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (OutOfService e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (DuplicateServant e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		}
		if (package_ == null) {
			String reason = "Failed to lookup the package servant "
					+ servantName;
			throw new AmsAdiException(reason);
		}

		InetAddress thisIp;
		try {
			thisIp = InetAddress.getLocalHost();
			if (profile.getFtpServerIP() == null
					|| profile.getFtpServerIP().trim().length() == 0) {
				profile.setFtpServerIP(thisIp.getHostAddress());
			}
		} catch (UnknownHostException e1) {
			String reason = e1.getMessage();
			throw new AmsAdiException(reason);
		}

		if (profile.getPackagedir() == null){
			String reason = "Package directory is not set";
			throw new AmsAdiException(reason);
		}
		String strURLName = FTPPROTOCOL + "://" + profile.getFtpUsername()
				+ ":" + profile.getFtpPassword() + "@"
				+ profile.getFtpServerIP() + dos2unix(profile.getPackagedir())
				+ "/" + ADIXML;

		try {
			if (profile.getAdiVersion().equals(ISA1_2)) {
				package_.provision(AdministrativeState.as_Unprovisioned,
						strURLName);
			} else {// ISA 1.3+
				package_
						.provision(AdministrativeState.as_InService, strURLName);
			}
		} catch (com.twc.isa.ServerModule.ProvisioningFailed e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (VersionException e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (InvalidStateChange e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (UnspecifiedException e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (XmlProcessingException e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		} catch (TransferException e) {
			String reason = e.message;
			throw new AmsAdiException(reason);
		}
	}

	private static String dos2unix(String s) {
		String unixstring = new String(s);
		int index;
		String nobackslash = unixstring.replace('\\', '/');
		if ((index = nobackslash.indexOf(":")) != -1) {
			return (nobackslash.substring(index + 1));
		}
		return nobackslash;
	}

}
