package com.tandbergtv.cms.portal.ui.widget.client;

import java.util.List;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;

public abstract class BaseDataProvider implements IDataProvider {
	private View view;
	private int pageSize;
	private int currentPageNum;
	private String sortColName;
	private SortOrder sortOrder;
	private int totalCount;
	private List<Row<ColumnData>> currentPage;
	
	public BaseDataProvider() {
		this.sortColName = getDefaultSortColumnName();
		this.sortOrder = getDefaultSortOrder();
	}
	
	protected void setTotalCount(int totalCount) {
		this.totalCount = totalCount;
	}
	
	public int getTotalCount() {
		return totalCount;
	}
	
	public void registerView(View view) {
		this.view = view;
	}

	private void fetchAndUpdateView() {
		if(view == null)
			return;
		getRecords(getCurrentPageBeginningIndex(), getCurrentPageEndingIndex(),
				getSortColumn(), getSortOrder(),
				new AsyncCallback<Page>() {
					public void onFailure(Throwable caught) {
						//TODO
						Window.alert("Failure in fetching records. Reason: " + caught.getMessage());
						view.update();
					}

					public void onSuccess(Page result) {
						currentPage = result.getRecords();
						setTotalCount(result.getTotalCount());
						view.update();
					}
				});
	}
	
	public int getPageSize() {
		return pageSize;
	}
	
	public void setPageSize(int pageSize) {
		if(pageSize == this.pageSize)
			return;
		this.pageSize = pageSize;
		this.currentPageNum = 1;
		fetchAndUpdateView();
	}

	public List<Row<ColumnData>> getCurrentPage() {
		return currentPage;
	}
	
	private int getCurrentPageBeginningIndex() {
		//inclusive
		return ((currentPageNum - 1) * pageSize) + 1;
	}

	private int getCurrentPageEndingIndex() {
		//inclusive
		int endIndex = getCurrentPageBeginningIndex() + pageSize - 1;
		if(endIndex > getTotalCount())
			endIndex = getTotalCount();
		return endIndex;
	}
	
	public int getNumPages() {
		return (int) Math.ceil((0f + getTotalCount()) / pageSize);
	}
	
	public boolean hasPage(int pageNum) {
		return (1 <= pageNum && pageNum <= getNumPages());
	}
	
	public void gotoPage(int pageNumber) throws PageUnavailableException {
		if(!hasPage(pageNumber))
			throw new PageUnavailableException(pageNumber);
		this.currentPageNum = pageNumber;
		fetchAndUpdateView();
	}

	public int getCurrentPageNumber() {
		return currentPageNum;
	}
	
	public boolean hasNextPage() {
		//return getCurrentPageEndingIndex() + 1 < getTotalCount();
		return hasPage(currentPageNum + 1);
	}
	
	//TODO validate with hasNextPage()
	public void gotoNextPage() {
		currentPageNum++;
		fetchAndUpdateView();
	}
	
	public boolean hasPrevPage() {
		return currentPageNum > 1;
	}

	//TODO validate with hasPrevPage()
	public void gotoPrevPage() {
		currentPageNum--;
		fetchAndUpdateView();
	}
	
	public void gotoFirstPage() {
		currentPageNum = 1;
		fetchAndUpdateView();
	}
	
	public void gotoLastPage() {
		currentPageNum = getNumPages();
		fetchAndUpdateView();
	}
	
	public void sort(String colName) {
		if(colName.equals(this.sortColName)) {
			reverseSortOrder();
		} else {
			this.sortColName = colName;
			this.sortOrder = SortOrder.ASCENDING;
		}
		gotoFirstPage();
	}
	
	private void reverseSortOrder() {
		if(sortOrder == SortOrder.ASCENDING)
			sortOrder = SortOrder.DESCENDING;
		else
			sortOrder = SortOrder.ASCENDING;
	}
	
	public String getSortColumn() {
		return sortColName;
	}
	
	public SortOrder getSortOrder() {
		return sortOrder;
	}

	public void delete(List<String> rowIds) {
		deleteRecords(rowIds, new AsyncCallback<Void>() {
			public void onFailure(Throwable caught) {
				//TODO
				Window.alert("Delete operation failed. Reason: " + caught.getMessage());
				view.update();
			}

			public void onSuccess(Void result) {
				fetchAndUpdateView();
			}});
	}
	
	public abstract Row<ColumnHeader> getHeader();
	
	//returns null when there is no default sort column
	protected abstract SortOrder getDefaultSortOrder();

	//returns null when there is no default sort column
	protected abstract String getDefaultSortColumnName();
	
	/*
	 * Both indices inclusive and are 1-based
	 * sortColName and sortOrder can be null
	 */
	protected abstract void getRecords(int beginIndex, int endIndex,
			String sortColName, SortOrder sortOrder,
			AsyncCallback<Page> callback);
	
	protected abstract void deleteRecords(List<String> recordIds, AsyncCallback<Void> callback);
}
