/**
 * 
 */
package com.ericsson.cms.categorymgmt.client;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ericsson.cms.categorymgmt.client.exception.UICategoryManagerException;
import com.ericsson.cms.categorymgmt.client.i18n.CategoryConstants;
import com.ericsson.cms.categorymgmt.client.model.CategoryDateFormat;
import com.ericsson.cms.categorymgmt.client.model.UICategoryFieldSchema;
import com.ericsson.cms.categorymgmt.client.model.UICategoryFieldSchemaDefault;
import com.ericsson.cms.categorymgmt.client.model.UICategoryHyperlink;
import com.ericsson.cms.categorymgmt.client.model.UICategoryNode;
import com.ericsson.cms.categorymgmt.client.model.UIConstants;
import com.ericsson.cms.categorymgmt.client.model.UIMetadataContent;
import com.ericsson.cms.categorymgmt.client.model.UIMetadataNode;
import com.ericsson.cms.categorymgmt.client.rpc.ICategoryFieldSchemaManagerService;
import com.ericsson.cms.categorymgmt.client.rpc.ICategoryFieldSchemaManagerServiceAsync;
import com.ericsson.cms.categorymgmt.client.validation.CategoryValidationView;
import com.ericsson.cms.categorymgmt.client.validation.UICategoryGeneralMessage;
import com.ericsson.cms.categorymgmt.client.validation.UICategoryMessage;
import com.ericsson.cms.categorymgmt.client.validation.UICategoryMessage.MessageLevel;
import com.ericsson.cms.categorymgmt.client.validation.UICategoryValidationException;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.regexp.shared.RegExp;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.IsSerializable;
import com.google.gwt.user.client.ui.AbsolutePanel;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Grid;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Hyperlink;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.Widget;
import com.google.gwt.user.datepicker.client.DatePicker;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.DateTimeInputWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.FormContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * @author eneevya
 *
 */
public class CategoryDetailPanel extends Composite implements IsSerializable{


    private static final String PATH_SEPARATOR = "/"; //$NON-NLS-1$


    private final NeptuneSecurity security;
    private final FlowPanel detailPanel;
    private final CategoryTreePanel treePanel;
    private final CategoryViewPanel parentPanel;
    private final CategoryHeaderPanel headerPanel;
    private final CategoryValidationView validationView;
    private ScrollContainer scrollContainer;
    private final FlowPanel buttonPanel;

    private final BusyIndicator busyIndicator = new BusyIndicator();

    private final ICategoryFieldSchemaManagerServiceAsync metadataService = GWT.create(ICategoryFieldSchemaManagerService.class);
    private final CategoryConstants constants = GWT.create(CategoryConstants.class);

    private Map<Object, Widget> valueMap = null;
    private UICategoryNode m_categoryNode = null;
    private boolean newCategory = false;

    private final List<UIMetadataNode> invalidNodes = new ArrayList<UIMetadataNode>();

    private final ArgClickHandler argClickHandler = new ArgClickHandler();


    public static final int CATEGORY_NAME_MAX_LENGTH = 1024;
    public static final int TEXTBOX_DEFAULT_MAX_LENGTH = 1024;


    public CategoryDetailPanel(CategoryViewPanel parentPanel, CategoryTreePanel treePanel,
            CategoryHeaderPanel headerPanel, CategoryValidationView validationView, FlowPanel buttonPanel) {
        this.parentPanel = parentPanel;
        this.treePanel = treePanel;
        this.headerPanel = headerPanel;
        this.validationView = validationView;
        this.buttonPanel = buttonPanel;
        security = NeptuneApplication.getApplication().getSecurity();


        detailPanel = new FlowPanel();
        detailPanel.addStyleName("zero-padding");
        detailPanel.setWidth(UIConstants.WIDTH_100PC);

        initWidget(detailPanel);
    }

    private VerticalContainer showHyperlinks( List<UICategoryHyperlink> links, String categoryName, boolean newCategory){
        VerticalContainer result = new VerticalContainer();
        HorizontalPanel hLinkPanel = new HorizontalPanel();
        for (final UICategoryHyperlink link : links)
        {
            String categoryParent = " " + link.getName(); //$NON-NLS-1$
            String token = UIConstants.CATEGORY_VIEW_ANCHOR + ".id=" + link.getId();

            hLinkPanel.add(new Hyperlink(categoryParent, token));
            hLinkPanel.add(new LabelWidget(UIConstants.CATEGORY_PATH_SEPARATOR + " "));
        }

        if(categoryName != null){
            hLinkPanel.add(new LabelWidget(categoryName)); //$NON-NLS-1$
        }else{
            hLinkPanel.add(new LabelWidget(constants.newCategoryTempName()));
        }

        result.add(hLinkPanel);
        return result;
    }



    private VerticalContainer showMetadataNode(UIMetadataNode metadata){

        VerticalContainer result = new VerticalContainer();

        if (metadata == null) {
            return result;
        }

        UICategoryFieldSchema uiSchema = metadata.getSchema();
        UIMetadataContent uiMetadataContent = metadata.getContent();

        if (uiSchema != null && uiMetadataContent != null) {
            String dataType = uiSchema.getDataType();
            if (UIConstants.DATATYPE_SET.equals(dataType)) {
                result.add(display(uiSchema, metadata));

            } else {
                Widget displayWidget = null;

                if (UIConstants.DATATYPE_STRING.equals(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getString());
                } else if (UIConstants.DATATYPE_STRING_CHOICE.equals(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getString());
                } else if (UIConstants.DATATYPE_INTEGER.equals(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getLong());
                } else if (UIConstants.DATATYPE_INTEGER_CHOICE.equals(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getLong());
                } else if (UIConstants.DATATYPE_DATETIMEZONE.equals(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getDate());
                } else if (UIConstants.DATATYPE_BOOLEAN.equals(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getString());
                } else if (UIConstants.DATATYPE_DECIMAL.endsWith(dataType)) {
                    displayWidget = display(uiSchema, uiMetadataContent.getDouble());
                }

                if (displayWidget != null) {
                    valueMap.put(metadata, displayWidget);
                    result.add(displayWidget);
                }

            }
        }
        return result;
    }


    private Widget display(UIMetadataNode value) {
        Widget result = null;

        if (value == null) {
            return result;
        }

        UICategoryFieldSchema uiSchema = value.getSchema();
        UIMetadataContent uiMetadataContent = value.getContent();

        if (uiSchema != null && uiMetadataContent != null) {
            String dataType = uiSchema.getDataType();

            if (UIConstants.DATATYPE_SET.equals(dataType)) {
                result = display(uiSchema, value);
            } else {
                Widget resultWidget = null;

                if (UIConstants.DATATYPE_STRING.equals(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getString());
                } else if (UIConstants.DATATYPE_STRING_CHOICE.equals(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getString());
                } else if (UIConstants.DATATYPE_INTEGER.equals(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getLong());
                } else if (UIConstants.DATATYPE_INTEGER_CHOICE.equals(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getLong());
                } else if (UIConstants.DATATYPE_DATETIMEZONE.equals(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getDate());
                } else if (UIConstants.DATATYPE_IMAGE.equals(dataType)) {
                    resultWidget = display(uiSchema, "ImageValue"); //$NON-NLS-1$
                } else if (UIConstants.DATATYPE_BOOLEAN.equals(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getString());
                } else if (UIConstants.DATATYPE_DECIMAL.endsWith(dataType)) {
                    resultWidget = display(uiSchema, uiMetadataContent.getDouble());

                }

                valueMap.put(value, resultWidget);

                HorizontalPanel valuePanel = new HorizontalPanel();
                valuePanel.add(resultWidget);

                // Display add/remove buttons (if necessary)
                if (security.isUserInRole(Permissions.CATEGORY_MODIFY)) {
                    valuePanel.add(showAddRemoveButtons(value));
                }

                result = valuePanel;
            }
        }
        return result;
    }

    private Widget display(UICategoryFieldSchema uiSchema, Double value) {
        Widget result = null;
        String strValue = null;
        if(value != null){
            strValue = value.toString();
        }

        String displayType = uiSchema.getDisplayType();

        if(UIConstants.DISPLAYTYPE_TEXT.equals(displayType)){
            TextBox textBox = new TextBox();
            textBox.setText(strValue);
            result = textBox;
        }
        else if(UIConstants.DISPLAYTYPE_TEXTAREA.equals(displayType)){
            TextArea textArea = new TextArea();
            textArea.setText(strValue);
            result = textArea;
        }
        return result;
    }

    private Widget display(UICategoryFieldSchema uiSchema, Long value) {
        Widget result = null;
        String displayType = uiSchema.getDisplayType();

        if(UIConstants.DISPLAYTYPE_TEXT.equals(displayType)){
            TextBox textBox = new TextBox();
            if (value != null) {
                textBox.setText(value.toString());
            }
            result = textBox;
        }
        else if(UIConstants.DISPLAYTYPE_TEXTAREA.equals(displayType)){
            TextArea textArea = new TextArea();
            if (value != null) {
                textArea.setText(value.toString());
            }
            result = textArea;
        }
        else if(UIConstants.DISPLAYTYPE_COMBOBOX.equals(displayType)){
            List<UICategoryFieldSchemaDefault> defaults = uiSchema.getDefaults();
            if(defaults != null){
                ListBox nameBox = new ListBox();

                nameBox.setSize(UIConstants.WIDTH_250PX, UIConstants.WIDTH_20PX);
                nameBox.addItem(UIConstants.COMBOBOX_SELECT);
                int i = 0;
                for(UICategoryFieldSchemaDefault schemaDefault: defaults)
                {
                    i++;
                    String defaultValue = schemaDefault.getKeyValue();
                    String numberValue = String.valueOf(schemaDefault.getNumberValue());

                    if (defaultValue != null && numberValue != null) {
                        nameBox.addItem(numberValue);

                        if (value != null
                                && value.equals(Long.valueOf(schemaDefault.getNumberValue().longValueExact()))) {
                            nameBox.setSelectedIndex(i);
                        }
                    }

                }
                result = nameBox;
            }
        }
        return result;
    }

    private Widget display(UICategoryFieldSchema uiSchema, String value) {
        Widget result = null;
        String displayType = uiSchema.getDisplayType() == null?UIConstants.DISPLAYTYPE_TEXT:uiSchema.getDisplayType();

        if(UIConstants.DISPLAYTYPE_TEXT.equals(displayType)){
            TextBox textBox = new TextBox();
            textBox.setMaxLength(getMaxLength(uiSchema.getMaxLen()));

            if (value != null) {
                textBox.setText(value);
            }
            result = textBox;
        }
        else if(UIConstants.DISPLAYTYPE_TEXTAREA.equals(displayType)){
            TextArea textArea = new TextArea();
            if (value != null) {
                textArea.setText(value);
            }
            result = textArea;
        }
        else if(UIConstants.DISPLAYTYPE_COMBOBOX.equals(displayType)){
            List<UICategoryFieldSchemaDefault> defaults = uiSchema.getDefaults();
            if(defaults != null){
                ListBox nameBox = new ListBox();
                nameBox.setSize(UIConstants.WIDTH_250PX, UIConstants.WIDTH_20PX);
                nameBox.addItem(UIConstants.COMBOBOX_SELECT);
                int i = 0;
                for(UICategoryFieldSchemaDefault schemaDefault: defaults)
                {
                    i++;
                    String defaultValue = schemaDefault.getStringValue();
                    if (defaultValue != null) {
                        nameBox.addItem(defaultValue);

                        if(defaultValue.equals(value)){
                            nameBox.setSelectedIndex(i);
                        }
                    }
                }
                result = nameBox;
            }
        }
        else if(UIConstants.DISPLAYTYPE_CHECKBOX.equals(displayType)){
            CheckBox nameBox = new CheckBox();
            if(UIConstants.BOOLEAN_TRUE.equals(value)) {
                nameBox.setValue(true);
            }
            result = nameBox;
        }
        else if(UIConstants.DISPLAYTYPE_HYPERLINK.equals(displayType)){
            Hyperlink nameLink = new Hyperlink();
            nameLink.setHTML(value);

            result = nameLink;
        }
        return result;
    }

    private Widget display(UICategoryFieldSchema uiSchema, Date value) {
        String displayType = uiSchema.getDisplayType();
        Widget result = null;

        if(UIConstants.DISPLAYTYPE_DATEPICKER.equals(displayType)){
            DateTimeInputWidget dateTimeWidget = new DateTimeInputWidget(new DatePicker(), value, new CategoryDateFormat(uiSchema.getValidationRegex()));
            result = dateTimeWidget;
        } else if(UIConstants.DISPLAYTYPE_TEXT.equals(displayType)){
            TextBox textBox = new TextBox();

            CategoryDateFormat dateFormat = new CategoryDateFormat(uiSchema.getValidationRegex());
            if (value != null)
            {
                String strDate = dateFormat.format(value);
                textBox.setValue(strDate);
            }

            result = textBox;
        }

        return result;
    }


    private String convertToString(UIMetadataNode node){
        if(node.getContent() != null)
        {
            if		(UIConstants.DATATYPE_STRING.equals(node.getSchema().getDataType()))		{return node.getContent().getString();}
            else if (UIConstants.DATATYPE_STRING_CHOICE.equals(node.getSchema().getDataType())) {return node.getContent().getString();}
            else if (UIConstants.DATATYPE_INTEGER.equals(node.getSchema().getDataType()))		{return node.getContent().getLong().toString();}
            else if (UIConstants.DATATYPE_INTEGER_CHOICE.equals(node.getSchema().getDataType())){return node.getContent().getLong().toString();}
            else if (UIConstants.DATATYPE_DECIMAL.equals(node.getSchema().getDataType()))		{return node.getContent().getDouble().toString();}
            else if (UIConstants.DATATYPE_BOOLEAN.equals(node.getSchema().getDataType()))		{return node.getContent().getBoolean().toString();}
            else if (UIConstants.DATATYPE_DATETIMEZONE.equals(node.getSchema().getDataType()))	{
                DateTimeFormat dtf = DateTimeFormat.getFormat(node.getSchema().getValidationRegex());
                dtf.format(node.getContent().getDate());
            }
        }
        return null;

    }

    private Widget display(UICategoryFieldSchema uiSchema, UIMetadataNode parentNode) {
        List<UIMetadataNode> values = parentNode.getContent().getList();

        Widget result = null;
        if(UIConstants.DISPLAYTYPE_SECTION.equals(uiSchema.getDisplayType())){
            RoundedDisclosureContainer disPanel = new RoundedDisclosureContainer(uiSchema.getName());
            disPanel.setOpen(true);
            if ( UIConstants.STANDARD_SECTION_NAME.equals(uiSchema.getName()) || UIConstants.CUSTOM_SECTION_NAME.equals(uiSchema.getName()) ) {
                disPanel.setWidth(UIConstants.WIDTH_DISCLOSURE_PANEL);
            }
            FormContainer fPanel = new FormContainer(HorizontalContainer.ALIGN_LEFT);
            for(UIMetadataNode value:values){
                LabelWidget name = new LabelWidget(value.getSchema().getName());
                long minOcc = value.getSchema().getMinOcc();
                boolean req = value.getSchema().isRequiredCreate();


                if(minOcc > 0 || req){
                    fPanel.addRow(name, display(value), true);
                }
                else
                {
                    fPanel.addRow(name, display(value));
                }

            }
            disPanel.add(fPanel);
            result = disPanel;
        }
        else if(UIConstants.DISPLAYTYPE_TABLE.equals(uiSchema.getDisplayType())) {
            //Use HashSets to calculate the number of unique values
            HashSet<String> colvals = new HashSet<String>();
            HashSet<Long> rowvals = new HashSet<Long>();
            List<UIMetadataNode> leftMostNodes = new ArrayList<UIMetadataNode>();
            for(UIMetadataNode value:values){
                colvals.add(value.getSchema().getName());
                rowvals.add(value.getOccNum());
            }
            //Add 2 columns to account for the +/- headings
            int colct = colvals.size();
            int rowct = rowvals.size();

            //Create the display grid
            Grid nameBox = new Grid(rowct+1, colct);
            int c = 0;
            int r = 0;
            for(UIMetadataNode column:values){
                //Format and display the headers
                nameBox.setText(0, c, column.getSchema().getName());
                nameBox.getCellFormatter().setStyleName(0, c, UIConstants.STYLE_DETAIL_TABLE_HEADER);
                nameBox.getCellFormatter().addStyleName(0, c, UIConstants.STYLE_DETAIL_TABLE_HEADER_TEXT);
                nameBox.getCellFormatter().setWidth(0, c, UIConstants.WIDTH_100PX);

                //Create the value box
                TextBox textBox = new TextBox();
                textBox.setText(convertToString(column));
                textBox.setMaxLength(getMaxLength(column.getSchema().getMaxLen()));

                //Put the value box in the proper grid cell and apply styling
                nameBox.setWidget(++r, c, textBox);
                nameBox.getCellFormatter().setWidth(r, c, UIConstants.WIDTH_100PX);

                //Add the value box to valueMap for lookup on save
                valueMap.put(column, textBox);

                //Add the right most nodes to a list for add/remove purposes ('Name' for Locale)
                if(c==0){
                    leftMostNodes.add(column);
                }

                //Move on to the next column
                if(r == rowct){
                    r = 0;
                    c++;
                }
            }

            nameBox.setBorderWidth(1);
            nameBox.setCellPadding(10);
            result =  nameBox;

            showAddRemoveButtons(parentNode, leftMostNodes, nameBox);
        }
        return result;
    }

    public void showCategory(UICategoryNode categoryNode, boolean createCategory) {



        headerPanel.updateCategoryHeaderView(categoryNode);
        valueMap = new HashMap<Object, Widget>();
        m_categoryNode = categoryNode;
        newCategory = createCategory;

        //Initialize messaging
        parentPanel.initFeedback();
        parentPanel.showFeedback();
        validationView.clearMessages();

        //Case where user tries to delete a non-existent category
        if(categoryNode == null){
            detailPanel.clear();
            detailPanel.add(new LabelWidget(constants.defaultCategoryDoesNotExistMessage()));
            return;
        }

        //Case where Root Category is selected
        if(categoryNode != null && !createCategory && categoryNode.getName().equals(UIConstants.ROOT_CATEGORY)){

            detailPanel.clear();
            detailPanel.add(new LabelWidget(constants.defaultDetailPanelMessage()));
        }
        else{

            try
            {
                detailPanel.clear();

                FlowPanel detailMainPanel = new FlowPanel();
                detailPanel.add(detailMainPanel);

                //Add the validation view

                if(categoryNode != null){
                    List<UICategoryHyperlink> links = categoryNode.getHyperlinks();
                    List<UIMetadataNode> metadata = categoryNode.getMetadata();
                    String categoryName = categoryNode.getName();

                    if(links != null){
                        detailMainPanel.add(showHyperlinks(links, categoryName, newCategory));
                    }

                    /** Form Panel for Category specific properties like Name, Internal Version & Associated Category Title Id */
                    FormContainer categoryPanel = new FormContainer(HorizontalContainer.ALIGN_LEFT);

                    LabelWidget nameLabel = new LabelWidget(constants.categoryNameLabel());
                    TextBox nameBox = new TextBox();
                    nameBox.setText(categoryNode.getName());
                    nameBox.setReadOnly(true);
                    nameBox.setMaxLength(CATEGORY_NAME_MAX_LENGTH);
                    nameBox.setWidth(UIConstants.CATEGORY_NAME_TEXTBOX_WIDTH);

                    LabelWidget internalVersionLabel = new LabelWidget(constants.internalVersionLabel());
                    TextBox versionBox = new TextBox();
                    versionBox.setText(String.valueOf(categoryNode.getInternalVersion()));
                    versionBox.setReadOnly(true);

                    /** Name can only be added for the first and then it is locked */
                    /** Internal Version is set to 1 by default for a new category */
                    if(newCategory){
                        nameBox.setReadOnly(false);
                        versionBox.setText("1");  //$NON-NLS-1$
                    }

                    valueMap.put(UIConstants.NEW_CATEGORY_KEY, nameBox);
                    valueMap.put(UIConstants.INTERNAL_VERSION_KEY, versionBox);

                    categoryPanel.addRow(nameLabel, nameBox, true);
                    categoryPanel.addRow(internalVersionLabel, versionBox, false);

                    /** display the Associated Category Title Id hyperlink only when
                     * - Selected category has an associated category title
                     *   (ASSOCIATED_CATEGORY_PACKAGE is not null)
                     * 
                     * */
                    LabelWidget associatedCatLabel = new LabelWidget(constants.categoryTitleLable());
                    if(categoryNode.getAssociateCatPackageId() != null){
                        final String associateId = categoryNode.getAssociateCatPackageId().toString();
                        Map<String, String> anchorTokens = new HashMap<String, String>();
                        anchorTokens.put("Id", associateId);
                        anchorTokens.put("returnAnchor", History.getToken());
                        String token = "Content.Search." + new AnchorTokenizer().buildAnchor(anchorTokens);
                        Hyperlink associatedCatLink = new Hyperlink(categoryNode.getAssociateCatPackageId().toString(), token);
                        categoryPanel.addRow(associatedCatLabel, associatedCatLink, false);
                    }else{
                        categoryPanel.addRow(associatedCatLabel, new LabelWidget("-"), false);
                    }

                    detailMainPanel.add(categoryPanel);

                    //Iterate the metadata list and display the values
                    FlowPanel vPanel = new FlowPanel();
                    scrollContainer = new ScrollContainer(vPanel);
                    int height = Window.getClientHeight() - UIConstants.HEIGHT_DETAIL_SCROLL_PANEL;
                    if(metadata != null){
                        scrollContainer.setWidth(UIConstants.WIDTH_SCROLL_PANEL);
                        scrollContainer.setHeight(height + "px");
                        for(UIMetadataNode m:metadata){
                            vPanel.add(showMetadataNode(m));
                        }
                    }
                    detailMainPanel.add(vPanel);
                }

                buttonPanel.clear();
                buttonPanel.setStyleName("category-button-panel");

                /** display the save button only when
                 * - user has create permission and its a new category
                 * - user has modify permission
                 * 
                 * */
                if(newCategory && security.isUserInRole(Permissions.CATEGORY_CREATE) ||
                        security.isUserInRole(Permissions.CATEGORY_MODIFY)) {
                    buttonPanel.add(createSaveButton());
                }


                /** display the Create Category Package button only when
                 * - user has create category permission
                 * - selected category is not associated to a package
                 * 
                 * */
                if (security.isUserInRole(Permissions.CATEGORY_CREATE)
                        && categoryNode.getAssociateCatPackageId() == null && !newCategory)

                {
                    buttonPanel.add(createCreateCategoryPackageButton());
                }

                /** display the Target button only when
                 * - user has target category permission
                 * - selected category is associated to a category package
                 * 
                 * */
                if(security.isUserInRole(Permissions.CATEGORY_TARGETCATEGORYTITLE)
                        && categoryNode.getAssociateCatPackageId() != null) {
                    buttonPanel.add(targetButton(categoryNode.getAssociateCatPackageId()));
                }

                /** display the Delete button only when
                 * - user has delete permission
                 * - selected node is the leaf node of the tree
                 * - selected node's object is not null
                 * 
                 * */
                if(security.isUserInRole(Permissions.CATEGORY_DELETE) && treePanel.getSelectedItem() != null &&
                        treePanel.getSelectedItem().getChildCount() == 0 &&
                        treePanel.getSelectedItem().getUserObject() != null) {
                    buttonPanel.add(deleteButton(categoryNode));
                }
                if (parentPanel != null){parentPanel.layout();}
            }
            //TODO: Handle this exception and display a meaningful text on UI, Use logger instead of System.out
            catch(Exception e)
            {
                Window.alert(e.getMessage());
                System.err.println(e.getMessage());
                e.printStackTrace(System.err);
            }
        }
    }

    /**
     * Construct the create category package button and define the create behavior
     * @return A Panel containing the create category package button
     */
    public Button createCreateCategoryPackageButton(){
        HorizontalPanel horizonButtonPanel = new HorizontalPanel();

        Button createCategoryPackageButton = new Button(constants.createCategoryPackageButton());
        createCategoryPackageButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
        createCategoryPackageButton.addStyleName("float-left");
        createCategoryPackageButton.setWidth(UIConstants.WIDTH_200PX);
        createCategoryPackageButton.addClickHandler(new ClickHandler(){
            @Override
            public void onClick(ClickEvent event){

                if(!Window.confirm(constants.createCategoryPackageConfirmMessage())){
                    return;
                }

                //Hide any previous validation messages
                clearMessages();

                //Create anchor to navigate
                Map<String, String> anchorTokens = new HashMap<String, String>();
                anchorTokens.put("type", "category"); //$NON-NLS-1$ //$NON-NLS-2$
                anchorTokens.put("returnAnchor", History.getToken()); //$NON-NLS-1$
                String token = "Content.Create." + new AnchorTokenizer().buildAnchor(anchorTokens); //$NON-NLS-1$
                History.newItem(token);
            }
        });
        horizonButtonPanel.add(createCategoryPackageButton);

        return createCategoryPackageButton;
    }

    /**
     * Construct the target button and define the target behavior
     * @return A Panel containing the target button
     */
    public Button targetButton(Long id){
        HorizontalPanel horizonButtonPanel = new HorizontalPanel();
        final String catId = id.toString();
        Button targetButton = new Button(constants.targetButton());
        targetButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
        targetButton.addStyleName("float-left");
        targetButton.addClickHandler(new ClickHandler(){
            @Override
            public void onClick(ClickEvent event){

                if(!Window.confirm(constants.targetConfirmMessage())){
                    return;
                }

                //Hide any previous validation messages
                clearMessages();

                //Create anchor to navigate
                Map<String, String> anchorTokens = new HashMap<String, String>();
                anchorTokens.put("Id", catId); //$NON-NLS-1$
                anchorTokens.put("returnAnchor", History.getToken()); //$NON-NLS-1$
                String token = "Content.Search." + new AnchorTokenizer().buildAnchor(anchorTokens) + "&Tab=Sites" ; //$NON-NLS-1$ //$NON-NLS-2$
                History.newItem(token);
            }
        });
        horizonButtonPanel.add(targetButton);

        return targetButton;
    }

    /**
     * Construct the delete button and define the delete behavior
     * @return A Panel containing the delete button
     */
    public Button deleteButton(UICategoryNode categoryNode){
        HorizontalPanel horizonButtonPanel = new HorizontalPanel();
        final String catName = categoryNode.getName();
        Button deleteButton = new Button(constants.deleteButton());
        deleteButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
        deleteButton.addStyleName("float-left");
        deleteButton.addClickHandler(new ClickHandler(){
            @Override
            public void onClick(ClickEvent event){

                if(!Window.confirm(constants.deleteCategoryConfirmMessage(catName))){
                    return;
                }

                //Hide any previous validation messages
                clearMessages();

                treePanel.deleteCategory();
                parentPanel.layout();
            }
        });
        horizonButtonPanel.add(deleteButton);

        return deleteButton;
    }

    /**
     * Construct the save button and define the save behavior
     * @return A Panel containing the save button
     */
    public Button createSaveButton(){
        HorizontalPanel horizonButtonPanel = new HorizontalPanel();

        Button saveButton = new Button(constants.saveButton());
        saveButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
        saveButton.addStyleName("float-left");
        saveButton.addClickHandler(new ClickHandler(){
            @Override
            public void onClick(ClickEvent event){
                if(!Window.confirm(constants.saveCategoryConfirmMessage())){
                    return;
                }

                //Hide any previous validation messages
                validationView.clearMessages();
                clearMessages();

                if(newCategory){
                    //This is a new category, so grab the category name
                    TextBox txtName 	= (TextBox)valueMap.get(UIConstants.NEW_CATEGORY_KEY);
                    String categoryName = txtName == null ? null : txtName.getValue() == null ? null : txtName.getValue().trim().length() == 0 ? null : txtName.getValue().trim();

                    //Reset any previous warning
                    txtName.removeStyleName(UIConstants.STYLE_VALIDATION_ERROR);

                    if(categoryName != null){
                        if (categoryName.contains(PATH_SEPARATOR))
                        {
                            txtName.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                            validationView.addValidationMessage(new UICategoryValidationException(constants.categoryName(), constants.invalidCategoryNameMessage())); //$NON-NLS-1$
                        }
                        else if (m_categoryNode.getSiblings() != null && m_categoryNode.getSiblings().contains(categoryName))
                        {
                            txtName.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                            validationView.addValidationMessage(new UICategoryValidationException(constants.categoryName(), constants.duplicateNameValidationMessage())); //$NON-NLS-1$
                        }
                        else
                        {
                            m_categoryNode.setName(categoryName);
                        }
                    } else{
                        txtName.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                        validationView.addValidationMessage(new UICategoryValidationException(constants.categoryName(), constants.emptyNameValidationMessage())); //$NON-NLS-1$
                    }
                }

                TextBox txtInternalVersion = (TextBox)valueMap.get(UIConstants.INTERNAL_VERSION_KEY);
                String intVersion = txtInternalVersion == null ? null : txtInternalVersion.getValue() == null ? null : txtInternalVersion.getValue().trim().length() == 0 ? null : txtInternalVersion.getValue().trim();
                if(intVersion != null)
                {
                    try{
                        m_categoryNode.setInternalVersion(Long.valueOf(txtInternalVersion.getValue()));
                    }
                    catch(NumberFormatException e){
                        return;
                    }
                } else{
                    return;
                }

                //Iterate through the metadata nodes and update the values
                try{
                    if(valueMap != null && m_categoryNode != null){
                        for(UIMetadataNode curMetadataNode : new ArrayList<UIMetadataNode>(m_categoryNode.getMetadata())){
                            try{
                                handleMetadataNode(curMetadataNode);
                            } catch(UICategoryValidationException ucve){
                                validationView.addValidationMessage(ucve);
                            }
                        }

                    }

                    try{
                        validateCategoryUnique(m_categoryNode);
                    } catch(UICategoryValidationException ucve){
                        //Should be a collection of validation exceptions
                        for(UICategoryValidationException curExc : ucve.getChildren()){
                            validationView.addValidationMessage(curExc);
                        }
                    }

                    invalidNodes.clear();

                } catch(UICategoryManagerException ucme){
                    validationView.addValidationMessage(new UICategoryGeneralMessage(MessageLevel.ERROR, ucme.getMessage()));
                } catch(Exception e){
                    validationView.addValidationMessage(new UICategoryGeneralMessage(MessageLevel.ERROR, constants.unanticipatedException() + e.getMessage())); //$NON-NLS-1$
                }

                if(validationView.hasErrors()){
                    validationView.refresh();
                    return;
                }

                //Send the save request to the server
                metadataService.saveCategory(m_categoryNode, newCategory, new NeptuneAsyncCallback<Long>() {
                    @Override
                    public void onNeptuneFailure(Throwable caught) {
                        if(caught instanceof UICategoryValidationException){
                            UICategoryValidationException ucve = (UICategoryValidationException)caught;
                            addValidationMessages(ucve);
                        } else{
                            validationView.addValidationMessage(new UICategoryGeneralMessage(UICategoryMessage.MessageLevel.ERROR, constants.unableToSaveCategory() + caught.getMessage())); //$NON-NLS-1$
                        }
                        validationView.refresh();
                    }

                    @Override
                    public void onNeptuneSuccess(Long result) {

                        //Set the message to display on reload
                        parentPanel.setFeedbackMessage(constants.saveCategorySuccessMessage(m_categoryNode.getName()));

                        //replacing the temporary id with the actual category id in the history
                        History.getToken().replace(UIConstants.CATEGORY_VIEW_ANCHOR + ".id=" + UIConstants.NEW_CATEGORY_TEMP_ID, UIConstants.CATEGORY_VIEW_ANCHOR + ".id="+m_categoryNode.getId());

                        if(newCategory){
                            m_categoryNode.setId(result);
                        }


                        //Re-display the updated category
                        if(m_categoryNode.getId() != null){
                            treePanel.reload(m_categoryNode.getId(), m_categoryNode.getParentId());

                            //Trigger a refresh for category updates
                            if(!newCategory){
                                History.fireCurrentHistoryState();
                            }
                        }
                    }
                });
                parentPanel.layout();

            }
        });
        horizonButtonPanel.add(saveButton);

        return saveButton;
    }
    /**
     * Traverse a UIMetadataNode tree and update it with validated user-entered data
     * @param parentNode
     * @throws UICategoryManagerException
     * @throws ParseException
     */
    private void handleMetadataNode(UIMetadataNode parentNode) throws UICategoryValidationException, UICategoryManagerException{
        UICategoryFieldSchema curFieldSchema = parentNode.getSchema();
        UIMetadataContent curContent = parentNode.getContent();
        if(UIConstants.DATATYPE_SET.equals(curFieldSchema.getDataType())){
            //Iterate the set and recursively handle the metadata nodes
            List<UIMetadataNode> childMetadataNodes = new ArrayList<UIMetadataNode>(curContent.getList());

            for(UIMetadataNode curMetadataNode : childMetadataNodes){
                try{
                    handleMetadataNode(curMetadataNode);
                } catch(UICategoryValidationException ucve){
                    this.validationView.addValidationMessage(ucve);
                }
            }

        } else if(UIConstants.DISPLAYTYPE_TEXT.equals(curFieldSchema.getDisplayType())){
            TextBox textWidget = (TextBox)valueMap.get(parentNode);
            textWidget.removeStyleName(UIConstants.STYLE_VALIDATION_ERROR);

            String strValue = textWidget.getValue() == null ? null : textWidget.getValue().trim().length() == 0 ? null : textWidget.getValue().trim();

            //Ensure required fields have values
            if((curFieldSchema.getMinOcc() > 0 || curFieldSchema.isRequiredCreate()) && strValue == null){
                textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                this.validationView.addValidationMessage(new UICategoryValidationException(curFieldSchema.getName(), constants.requiredFieldMissingMessage()));
            }

            if(strValue != null)
            {
                //Validate input length
                if(strValue.length() < curFieldSchema.getMinLen()){
                    textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                    this.validationView.addValidationMessage(new UICategoryValidationException(curFieldSchema.getName(), constants.minLengthMessage(curFieldSchema.getMinLen())));
                }
                if(curFieldSchema.getMaxLen() > 0 && strValue.length() > curFieldSchema.getMaxLen()){
                    textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                    this.validationView.addValidationMessage(new UICategoryValidationException(curFieldSchema.getName(), constants.maxLengthMessage(curFieldSchema.getMaxLen())));
                }


                //String
                if(UIConstants.DATATYPE_STRING.equals(curFieldSchema.getDataType())){

                    //Validate the entered value against a predefined pattern
                    String strPattern = curFieldSchema.getValidationRegex();
                    if(strPattern != null){
                        RegExp pattern = RegExp.compile(strPattern);
                        boolean validated = pattern.test(strValue);
                        if(!validated){
                            invalidNodes.add(parentNode);
                            textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                            this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.invalidFormatMessage("text", strPattern))); //$NON-NLS-1$
                        }
                    }

                    curContent.setString(strValue);

                } else if(UIConstants.DATATYPE_INTEGER.equals(curFieldSchema.getDataType())){

                    try{
                        curContent.setNumber(Integer.parseInt(strValue));
                    } catch(NumberFormatException nfe){
                        invalidNodes.add(parentNode);
                        textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                        this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.invalidValueMessage("integer"))); //$NON-NLS-1$
                    }
                } else if(UIConstants.DATATYPE_DECIMAL.equals(curFieldSchema.getDataType())){
                    try{
                        curContent.setNumber(Double.parseDouble(strValue));
                    } catch(NumberFormatException nfe){
                        textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                        this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.invalidValueMessage("decimal"))); //$NON-NLS-1$
                    }
                } else if(UIConstants.DATATYPE_DATETIMEZONE.equals(curFieldSchema.getDataType())){
                    String strPattern = curFieldSchema.getValidationRegex();
                    CategoryDateFormat cdf = new CategoryDateFormat(strPattern);
                    try{
                        Date dateValue = cdf.parseStrict(strValue);

                        if(dateValue != null){
                            curContent.setDate(dateValue);
                        }
                    } catch(IllegalArgumentException iae){
                        invalidNodes.add(parentNode);
                        textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                        this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.invalidFormatMessage("Date", strPattern))); //$NON-NLS-1$
                    }
                }

            } else {
                curContent.setDate(null);
                curContent.setString(null);
                curContent.setNumber((BigDecimal)null);
            }
        } else if(UIConstants.DISPLAYTYPE_DATEPICKER.equals(curFieldSchema.getDisplayType())){
            DateTimeInputWidget widget = (DateTimeInputWidget)valueMap.get(parentNode);
            widget.getTextBox().removeStyleName(UIConstants.STYLE_VALIDATION_ERROR);

            String strDate = widget.getTextBox().getValue() == null ? null : widget.getTextBox().getValue().trim().length() == 0 ? null : widget.getTextBox().getValue();

            Date dateValue = widget.getValue();

            if(curFieldSchema.isRequiredCreate() && dateValue == null){
                widget.getTextBox().addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.requiredFieldMissingMessage()));
            }

            //Validate the date's format
            if(strDate != null){
                Date validDate = widget.getFormat().parse(widget, strDate, true);

                //A null date means the format was invalid
                if(validDate == null){
                    invalidNodes.add(parentNode);
                    widget.getTextBox().addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                    CategoryDateFormat cdf = (CategoryDateFormat)widget.getFormat();
                    String pattern = cdf.getPattern();
                    this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.invalidFormatMessage("Date", pattern))); //$NON-NLS-1$
                }
            }

            if(curFieldSchema.isRequiredCreate() && dateValue == null){
                widget.getTextBox().addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                throw new UICategoryValidationException(curFieldSchema.getName(), constants.requiredFieldMissingMessage());
            }

            if(dateValue != null){
                //Set the value
                curContent.setDate(dateValue);
            } else{
                curContent.setDate(null);
            }

        } else if(UIConstants.DISPLAYTYPE_COMBOBOX.equals(curFieldSchema.getDisplayType())){
            ListBox comboWidget = (ListBox)valueMap.get(parentNode);
            comboWidget.removeStyleName(UIConstants.STYLE_VALIDATION_ERROR);
            Integer selectedIndex = comboWidget.getSelectedIndex();

            if(curFieldSchema.isRequiredCreate() && selectedIndex <= 0){
                comboWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.requiredFieldMissingMessage()));
            }

            //Only set the values if something was selected
            if(selectedIndex > 0){

                if(UIConstants.DATATYPE_INTEGER_CHOICE.equals(curFieldSchema.getDataType())){
                    //Number Choice
                    curContent.setNumber(Integer.parseInt(comboWidget.getValue(selectedIndex)));

                } else if(UIConstants.DATATYPE_STRING_CHOICE.equals(curFieldSchema.getDataType())){
                    //String Choice
                    curContent.setString(comboWidget.getValue(comboWidget.getSelectedIndex()));
                }
            } else{
                curContent.setNumber((BigDecimal)null);
                curContent.setString(null);
            }

        } else if(UIConstants.DISPLAYTYPE_CHECKBOX.equals(curFieldSchema.getDisplayType())){
            CheckBox checkWidget = (CheckBox)valueMap.get(parentNode);

            curContent.setBoolean(checkWidget.getValue());

        } else if(UIConstants.DISPLAYTYPE_TEXTAREA.equals(curFieldSchema.getDisplayType())){
            TextArea textWidget = (TextArea)valueMap.get(parentNode);
            textWidget.removeStyleName(UIConstants.STYLE_VALIDATION_ERROR);

            String strValue = textWidget.getValue() == null ? null : textWidget.getValue().trim().length() == 0 ? null : textWidget.getValue().trim() ;

            //Ensure required fields have values
            if((curFieldSchema.getMinOcc() > 0 || curFieldSchema.isRequiredCreate()) && strValue == null ){
                textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.requiredFieldMissingMessage()));
            }

            if (strValue != null)
            {
                //Validate the entered value against a predefined pattern
                String strPattern = curFieldSchema.getValidationRegex();
                if(strPattern != null){
                    RegExp pattern = RegExp.compile(strPattern);
                    boolean validated = pattern.test(strValue);
                    if(!validated){
                        invalidNodes.add(parentNode);
                        textWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                        this.validationView.addValidationMessage( new UICategoryValidationException(curFieldSchema.getName(), constants.invalidFormatMessage("text", strPattern))); //$NON-NLS-1$
                    }
                }
                curContent.setString(strValue);

            }
            else
            {
                curContent.setString(null);
            }

        } else{
            //What to do, what to do?
            throw new UICategoryManagerException(constants.unknownDisplayType() + curFieldSchema.getDisplayType()); //$NON-NLS-1$
        }
    }

    /**
     * Count the total instances of a field type in a list of metadata nodes
     * @param nodeList The list of nodes to search
     * @param name The name of the field to count
     * @return
     */
    private int nodeCount(List<UIMetadataNode> nodeList, String name){
        int count = 0;

        for(UIMetadataNode curNode : nodeList){
            UIMetadataContent content = curNode.getContent();
            UICategoryFieldSchema fieldSchema = curNode.getSchema();

            if(content.getList() != null){
                count += nodeCount(content.getList(), name);
            }

            if(name.equals(fieldSchema.getName())){
                count++;
            }
        }


        return count;
    }

    private void updateAddRemoveButtons(List<UIMetadataNode> nodeList, String fieldName){
        for(UIMetadataNode curNode : nodeList){
            UIMetadataContent content = curNode.getContent();
            UICategoryFieldSchema fieldSchema = curNode.getSchema();

            if(content.getList() != null){
                updateAddRemoveButtons(content.getList(), fieldName);
            }

            if(fieldName.equals(fieldSchema.getName())){
                Widget entryWidget = valueMap.get(curNode);
                HorizontalPanel horizPanel = (HorizontalPanel)entryWidget.getParent();
                horizPanel.remove(horizPanel.getWidgetCount()-1);
                horizPanel.add(showAddRemoveButtons(curNode));
            }
        }
    }

    private List<UIMetadataNode> getNodesByFieldName(List<UIMetadataNode> metadataList, String fieldName){
        List<UIMetadataNode> fieldNodes = new ArrayList<UIMetadataNode>();

        for(UIMetadataNode curNode : metadataList){
            UIMetadataContent content = curNode.getContent();
            UICategoryFieldSchema fieldSchema = curNode.getSchema();

            if(content.getList() != null){
                updateAddRemoveButtons(content.getList(), fieldName);
            }

            if(fieldName.equals(fieldSchema.getName())){
                fieldNodes.add(curNode);
            }
        }

        return fieldNodes;
    }

    class DisclosurePanelHeader extends HorizontalPanel
    {
        public DisclosurePanelHeader(AbsolutePanel myPanel)
        {

            add(myPanel);
        }
    }

    public void anchorChanged(String anchor) {
        if(isBlank(anchor)) {
            detailPanel.clear();
            headerPanel.refreshHeading();
            parentPanel.hideFeedback();
            treePanel.reload(new Long(0), new Long(0));
        } else{
            AnchorTokenizer at = new AnchorTokenizer();
            Map<String, String> paramMap = 	at.parseAnchor(anchor);
            Long id = Long.parseLong(paramMap.get("id"));

            if(id != null && id >= 0){
                selectCategory(id);
            }
        }

    }

    private boolean isBlank(String s) {
        return s == null || s.isEmpty();
    }

    /**
     * Create and define functionality for the buttons to add/remove field occurrences
     * @param metadataNode
     * @return
     */
    private Panel showAddRemoveButtons(UIMetadataNode metadataNode){
        HorizontalPanel horizonButtonPanel = new HorizontalPanel();

        UIMetadataContent metadataContent = metadataNode.getContent();
        UICategoryFieldSchema metadataSchema = metadataNode.getSchema();

        List<UIMetadataNode> categoryMetadataList = m_categoryNode.getMetadata();

        int nodeCount = -1;
        if(UIConstants.DISPLAYTYPE_TABLE.equals(metadataSchema.getDisplayType())){
            nodeCount = nodeCount(categoryMetadataList, metadataContent.getList().get(0).getSchema().getName());
        } else {
            nodeCount = nodeCount(categoryMetadataList, metadataSchema.getName());
        }

        /**
         * Display delete button only if
         * - a field has multiple occurrences
         * - max occurrences is not set to 1
         * */
        if(nodeCount > 1 && metadataSchema.getMaxOcc() != 1)
        {
            ArgButton deleteButton = new ArgButton(UIConstants.DELETE_BUTTON_LABEL, ArgButton.DELETE_INSTANCE, metadataNode);
            deleteButton.addClickHandler(argClickHandler);
            deleteButton.addStyleName(StyleNames.DATALOSS_BUTTON_STYLE);
            deleteButton.setTitle(constants.datalossButtonTitle());
            horizonButtonPanel.add(deleteButton);
        }


        //Display add button
        if(metadataNode.getOccNum() != null && metadataNode.getOccNum() == nodeCount &&
                (metadataNode.getOccNum() < metadataSchema.getMaxOcc() || metadataSchema.getMaxOcc() == 0)){
            ArgButton addButton = new ArgButton(UIConstants.ADD_BUTTON_LABEL, ArgButton.ADD_INSTANCE, metadataNode);
            addButton.addStyleName(StyleNames.COMMIT_BUTTON_STYLE);
            addButton.addClickHandler(argClickHandler);
            addButton.setTitle(constants.commitButtonTitle());
            horizonButtonPanel.add(addButton);
        }

        return horizonButtonPanel;
    }

    /**
     * Show the +/- buttons for a table
     * @param fieldSchema The schema of the table
     * @param metadataList The right-most column of metadata nodes
     * @param grid The Grid widget
     */
    public void showAddRemoveButtons(UIMetadataNode parentNode, List<UIMetadataNode> metadataList, Grid grid){
        if(metadataList != null){
            //Use the "left-most" node as the "master" node
            UIMetadataNode masterNode = metadataList.get(0);
            UICategoryFieldSchema fieldSchema = masterNode.getSchema();
            UICategoryFieldSchema parentSchema = parentNode.getSchema();
            if(fieldSchema != null){

                int i=1;
                if(fieldSchema.getMaxOcc() != 1 && metadataList.size() > 1){
                    grid.resizeColumns(grid.getColumnCount()+1);
                    grid.setText(0, grid.getColumnCount()-1, UIConstants.DELETE_BUTTON_LABEL);
                    grid.getCellFormatter().setStyleName(0, grid.getColumnCount()-1, UIConstants.STYLE_DETAIL_TABLE_HEADER);
                    grid.getCellFormatter().addStyleName(0, grid.getColumnCount()-1, UIConstants.STYLE_DETAIL_TABLE_HEADER_TEXT);

                    for(UIMetadataNode curMetadataNode : metadataList){
                        curMetadataNode.getSchema().setParent(parentSchema);
                        ArgButton removeButton = new ArgButton(UIConstants.DELETE_BUTTON_LABEL, ArgButton.DELETE_INSTANCE, curMetadataNode, parentNode);
                        removeButton.addClickHandler(argClickHandler);
                        removeButton.addStyleName(StyleNames.DATALOSS_BUTTON_STYLE);
                        removeButton.setTitle(constants.datalossButtonTitle());
                        grid.setWidget(i, grid.getColumnCount()-1, removeButton);
                        i++;
                    }
                }


                if(fieldSchema.getMaxOcc() == 0 || fieldSchema.getMaxOcc() > metadataList.size()){
                    grid.resizeColumns(grid.getColumnCount()+1);
                    grid.setText(0, grid.getColumnCount()-1, UIConstants.ADD_BUTTON_LABEL);
                    grid.getCellFormatter().setStyleName(0, grid.getColumnCount()-1, UIConstants.STYLE_DETAIL_TABLE_HEADER);
                    grid.getCellFormatter().addStyleName(0, grid.getColumnCount()-1, UIConstants.STYLE_DETAIL_TABLE_HEADER_TEXT);

                    ArgButton addButton = new ArgButton(UIConstants.ADD_BUTTON_LABEL, ArgButton.ADD_INSTANCE, parentNode);
                    addButton.addClickHandler(argClickHandler);
                    addButton.addStyleName(StyleNames.COMMIT_BUTTON_STYLE);
                    addButton.setTitle(constants.commitButtonTitle());
                    grid.setWidget(grid.getRowCount()-1, grid.getColumnCount()-1, addButton);
                }
            }
        }
    }

    /**
     * A box to pop up and accept input when adding field occurrences
     * @author CMeyer
     *
     */
    class NewInstanceEntryBox {
        private UIMetadataNode metadataNode = null;
        public NewInstanceEntryBox(UIMetadataNode metadataNode){
            this.metadataNode = metadataNode;
        }

        /**
         * Do the necessary processing (validation, value addition, UI re-display) when a user adds a new occurrence
         */
        public void handleSaveEvent(){

            //Create a metadata node with the user entered value
            UIMetadataNode newNode = new UIMetadataNode();
            newNode.setSchema(metadataNode.getSchema());


            UIMetadataContent newContent = new UIMetadataContent();
            String displayType = metadataNode.getSchema().getDisplayType();

            try{
                if (UIConstants.DISPLAYTYPE_TABLE.equals(displayType)) {
                    List<UICategoryFieldSchema> childrenSchema = metadataNode.getSchema().getChildren();

                    Long prevOccNum = 0l;
                    List<UIMetadataNode> parentList = metadataNode.getContent().getList();
                    if(parentList.size() > 0){
                        prevOccNum = parentList.get(parentList.size()-1).getOccNum();
                    }

                    //Insert a new row in the table
                    UIMetadataNode firstChild = metadataNode.getContent().getList().get(0);
                    Widget entryWidget = valueMap.get(firstChild);
                    Widget parentWidget = entryWidget.getParent();

                    Grid flexTable = (Grid)parentWidget;
                    int row = flexTable.getRowCount();
                    flexTable.insertRow(row);

                    for(int i=0; i<2; i++)
                    {
                        UICategoryFieldSchema curSchema = childrenSchema.get(i);

                        UIMetadataNode curMetadataNode = new UIMetadataNode();
                        curMetadataNode.setSchema(curSchema);

                        UIMetadataContent curContent = new UIMetadataContent();
                        TextBox curTextBox = new TextBox();
                        curContent.setString(curTextBox.getValue());

                        curMetadataNode.setOccNum(prevOccNum+1);

                        curMetadataNode.setContent(curContent);

                        //Insert in the proper position
                        int curIndex = 0;
                        boolean foundLikeName = false;

                        for(UIMetadataNode searchNode : parentList){


                            if(!foundLikeName && searchNode.getSchema().getName().equals(curSchema.getName())){
                                foundLikeName = true;
                            } else if (foundLikeName && !searchNode.getSchema().getName().equals(curSchema.getName())){
                                break;
                            }
                            curIndex++;
                        }

                        if(curIndex < parentList.size()){
                            parentList.add(curIndex, curMetadataNode);
                        } else{
                            parentList.add(curMetadataNode);
                        }

                        curMetadataNode.setParentList(parentList);

                        TextBox valueWidget = new TextBox();
                        valueWidget.setValue(curMetadataNode.getContent().getString());

                        flexTable.setWidget(row, i, valueWidget);
                        valueMap.put(curMetadataNode, valueWidget);
                    }

                    removeButtonColumns(flexTable);
                    showAddRemoveButtons(metadataNode, getNodesByFieldName(parentList, "Locale"), flexTable);

                } else{
                    if(UIConstants.DISPLAYTYPE_TEXT.equals(displayType)){
                        TextBox textBox = new TextBox();

                    } else if(UIConstants.DISPLAYTYPE_TEXTAREA.equals(displayType)){
                        TextArea textArea = new TextArea() ;

                    } else if(UIConstants.DISPLAYTYPE_COMBOBOX.equals(displayType)){
                        ListBox listBox = new ListBox();

                    } else if(UIConstants.DISPLAYTYPE_CHECKBOX.equals(displayType)){
                        CheckBox checkBox = new CheckBox();

                    } else if(UIConstants.DISPLAYTYPE_DATEPICKER.equals(displayType)){
                        DateTimeInputWidget dtiw = new DateTimeInputWidget(new DatePicker(), null, new CategoryDateFormat(metadataNode.getSchema().getValidationRegex()));
                    }

                    newNode.setContent(newContent);
                    newNode.setOccNum(metadataNode.getOccNum() + 1);

                    //Insert the new node after the last field occurrence
                    for(int i=0; i<metadataNode.getParentList().size(); i++){
                        if(metadataNode.equals(metadataNode.getParentList().get(i))){
                            metadataNode.getParentList().add(i+1, newNode);
                        }
                    }
                    newNode.setParentList(metadataNode.getParentList());

                    Widget entryWidget = valueMap.get(metadataNode);
                    Widget horizWidget = entryWidget.getParent();
                    Widget fcWidget = horizWidget.getParent();

                    FlexTable flexTable = (FlexTable)fcWidget;

                    //Search for the row containing the original metadataNode's widget
                    int row = findWidgetRow(flexTable, valueMap.get(metadataNode))+1;

                    flexTable.insertRow(row);

                    LabelWidget label = new LabelWidget(newNode.getSchema().getName());
                    boolean required = metadataNode.getSchema().isRequiredCreate() || metadataNode.getSchema().getMinOcc() > 0;
                    FieldLabel fieldLabel = new FieldLabel(label, required);
                    label.setStyleName("nwt-FormContainer-label");

                    Widget valueWidget = display(newNode);
                    valueWidget.setStyleName("nwt-FormContainer-value");



                    flexTable.setWidget(row, 0, fieldLabel);
                    flexTable.setWidget(row, 1, valueWidget);

                    flexTable.getCellFormatter().addStyleName(row, 0, "nwt-FormContainer-label-cell");
                    flexTable.getCellFormatter().addStyleName(row, 1, "nwt-FormContainer-value-cell");

                    //Add the new widget to the value map
                    HorizontalPanel valuePanel = (HorizontalPanel)valueWidget;
                    Widget valueMapWidget = valuePanel.getWidget(0);
                    valueMap.put(newNode, valueMapWidget);

                    updateAddRemoveButtons(metadataNode.getParentList(), newNode.getSchema().getName());
                }
            } catch(NumberFormatException nfe){
                Window.alert("Invalid number format");
            } catch(IllegalArgumentException iae){
                Window.alert("Invalid date format");
            } catch(Exception e){
                Window.alert(e.getMessage());
            }
        }
    }

    /**
     * Class to represent +/- buttons
     */
    class ArgButton extends Button{
        public static final int ADD_INSTANCE = 1;
        public static final int DELETE_INSTANCE = 2;

        private int function = -1;
        private UIMetadataNode metadataNode = null;
        private UIMetadataNode parentNode = null;

        public ArgButton(String label, int function, UIMetadataNode metadataNode){
            super(label);
            this.function = function;
            this.metadataNode = metadataNode;
        }

        public ArgButton(String label, int function, UIMetadataNode metadataNode, UIMetadataNode parentNode){
            super(label);
            this.function = function;
            this.metadataNode = metadataNode;
            this.parentNode = parentNode;
        }

        public int getFunction(){
            return this.function;
        }
        public UIMetadataNode getMetadataNode(){
            return this.metadataNode;
        }
        public UIMetadataNode getParentNode(){
            return this.parentNode;
        }
    }

    /**
     * Class to handle click events on +/- buttons
     */
    class ArgClickHandler implements ClickHandler{
        @Override
        public void onClick(ClickEvent event){
            ArgButton argButton = (ArgButton)event.getSource();

            int buttonFunction = argButton.getFunction();
            UIMetadataNode metadataNode = argButton.getMetadataNode();

            if(metadataNode != null){

                switch(buttonFunction){
                case ArgButton.ADD_INSTANCE:
                    NewInstanceEntryBox entryBox = new NewInstanceEntryBox(metadataNode);
                    entryBox.handleSaveEvent();
                    break;
                case ArgButton.DELETE_INSTANCE:

                    UICategoryFieldSchema fieldSchema = metadataNode.getSchema();
                    if(fieldSchema != null && fieldSchema.getParent() != null &&
                            fieldSchema.getParent().getDisplayType() != null &&
                            UIConstants.DISPLAYTYPE_TABLE.equals(fieldSchema.getParent().getDisplayType())){



                        List<UIMetadataNode> parentList = new ArrayList<UIMetadataNode>(metadataNode.getParentList());

                        if(parentList != null){

                            //Use HashSets to calculate the number of unique values
                            HashSet<Long> rowvals = new HashSet<Long>();

                            for(UIMetadataNode value:parentList){
                                rowvals.add(value.getOccNum());
                            }
                            //Add 2 columns to account for the +/- headings
                            int rowct = rowvals.size();

                            int r=0;
                            int rowToRemove=-1;
                            for(UIMetadataNode curMetadataNode : parentList){
                                if(curMetadataNode.getOccNum() != null && curMetadataNode.getOccNum() == metadataNode.getOccNum()){
                                    removeNodeFromParent(curMetadataNode, false);

                                    if(rowToRemove < 0){
                                        //Add one to account for the header row
                                        rowToRemove = r+1;
                                    }
                                }

                                //Reached the bottom of the table, so reset the row counter
                                if(++r == rowct){
                                    r=0;
                                }
                            }

                            //Obtain a reference to the table's display Grid
                            UIMetadataNode refNode = parentList.get(0);
                            Widget entryWidget = valueMap.get(refNode);
                            Widget gridWidget = entryWidget.getParent();
                            Grid grid = (Grid)gridWidget;
                            grid.removeRow(rowToRemove);

                            //Remove any +/- button columns already in the Grid
                            removeButtonColumns(grid);

                            //Display applicable +/- buttons
                            UIMetadataNode parentNode = argButton.getParentNode();
                            if(parentNode != null){
                                showAddRemoveButtons(parentNode, getNodesByFieldName(refNode.getParentList(), "Locale"), grid);
                            }
                        }

                    } else{
                        //Remove the metadata node from the category
                        removeNodeFromParent(metadataNode, true);

                        //Remove from the UI
                        Widget entryWidget = valueMap.get(metadataNode);
                        Widget horizWidget = entryWidget.getParent();
                        Widget fcWidget = horizWidget.getParent();

                        FlexTable flexTable = (FlexTable)fcWidget;

                        int rowToRemove = findWidgetRow(flexTable, entryWidget);

                        flexTable.removeRow(rowToRemove);

                        //Redisplay the applicable +/- buttons for the field grouping
                        updateAddRemoveButtons(metadataNode.getParentList(), metadataNode.getSchema().getName());
                    }

                    break;
                }

            }
        }
    }

    /**
     * Remove a field from a category and update the ordering
     * @param metadataNode
     */
    protected void removeNodeFromParent(UIMetadataNode metadataNode, boolean updateOccNum){
        //Remove the selected field from the associated list

        List<UIMetadataNode> parentList = metadataNode.getParentList();
        if(parentList != null){
            parentList.remove(metadataNode);
        }

        if(updateOccNum){
            updateOccNums(metadataNode.getSchema().getName(), parentList);
        }
    }

    /**
     * Reset a field's occurrence numbers
     * @param fieldSchemaName
     * @param metadataList
     */
    protected void updateOccNums(String fieldSchemaName, List<UIMetadataNode> metadataList){
        //Update the remaining field type's occurrence numbers
        long curOccNum = 1;
        for(UIMetadataNode curMetadataNode : metadataList){
            if(fieldSchemaName.equals(curMetadataNode.getSchema().getName())){
                curMetadataNode.setOccNum(curOccNum);
                curOccNum++;
            }
        }
    }

    /**
     * Recursively add validation messages and their children to the validation view
     * @param ve
     */
    private void addValidationMessages(UICategoryValidationException ve)
    {
        validationView.addValidationMessage(ve);
        if (ve.getChildren() != null)
        {
            for(UICategoryValidationException child: ve.getChildren())
            {
                addValidationMessages(child);
            }
        }
    }


    /**
     * Validate a list of metadata nodes against configured uniqueness settings
     * @param metadataList
     * @return A Set of metadata nodes that violate uniqueness settings
     */
    private Set<String> validateUnique(List<UIMetadataNode> metadataList){
        Set<String> duplicateOccs = new HashSet<String>();

        for(UIMetadataNode metadataNode : metadataList){
            if(UIConstants.DATATYPE_SET.equals(metadataNode.getSchema().getDataType())){
                duplicateOccs.addAll(validateUnique(metadataNode.getContent().getList()));
            } else{
                if(!invalidNodes.contains(metadataNode)){
                    UIMetadataContent umc = metadataNode.getContent();
                    String strNode = metadataNode.getSchema().getName() + ":" + umc.getString() + ":" + umc.getNumber() + ":" + umc.getBoolean() + ":" + umc.getImage() + ":" + umc.getDate(); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$

                    if(findDuplicateValues(metadataNode, metadataList)){
                        Widget mappedWidget = valueMap.get(metadataNode);
                        Widget duplicateWidget;

                        if(UIConstants.DISPLAYTYPE_DATEPICKER.equals(metadataNode.getSchema().getDisplayType())){
                            DateTimeInputWidget dtiw = (DateTimeInputWidget)mappedWidget;
                            duplicateWidget = dtiw.getTextBox();
                        } else{
                            duplicateWidget = mappedWidget;
                        }

                        if(duplicateWidget != null){
                            duplicateWidget.addStyleName(UIConstants.STYLE_VALIDATION_ERROR);
                        }

                        duplicateOccs.add(metadataNode.getSchema().getId() + "]" + metadataNode.getSchema().getName()); //$NON-NLS-1$
                    }
                }
            }
        }

        return duplicateOccs;
    }

    private boolean findDuplicateValues(UIMetadataNode metadataNode, List<UIMetadataNode> metadataList){
        boolean foundDuplicate = false;

        UIMetadataContent umc = metadataNode.getContent();
        if(umc.hasValue()){
            String metadataStr = metadataNode.getSchema().getName() + ":" + umc.getString() + ":" + umc.getNumber() + ":" + umc.getBoolean() + ":" + umc.getImage() + ":" + umc.getDate();

            for(UIMetadataNode curMetadataNode : metadataList){
                if(metadataNode != curMetadataNode){
                    if(UIConstants.DATATYPE_SET.equals(curMetadataNode.getSchema().getDataType())){
                        foundDuplicate = foundDuplicate || findDuplicateValues(metadataNode, curMetadataNode.getContent().getList());
                    } else{
                        UIMetadataContent curUmc = curMetadataNode.getContent();
                        if(curUmc.hasValue()){
                            String strNode = curMetadataNode.getSchema().getName() + ":" + curUmc.getString() + ":" + curUmc.getNumber() + ":" + curUmc.getBoolean() + ":" + curUmc.getImage() + ":" + curUmc.getDate();

                            if(metadataStr.equals(strNode)){
                                foundDuplicate = true;
                                break;
                            }
                        }
                    }
                }
            }
        }
        return foundDuplicate;
    }

    /**
     * Validate a category against configured uniqueness settings
     * @param categoryNode The UICategoryNode to validate
     */
    private void validateCategoryUnique(UICategoryNode categoryNode) throws UICategoryValidationException{
        //Validate uniqueness
        Set<String> duplicates = validateUnique(categoryNode.getMetadata());
        UICategoryValidationException ucve = new UICategoryValidationException();
        List<UICategoryValidationException> validationList = new ArrayList<UICategoryValidationException>();
        for(String duplicateField : duplicates){
            String duplicateName = duplicateField.substring(duplicateField.indexOf("]")+1); //$NON-NLS-1$
            validationList.add(new UICategoryValidationException(duplicateName, constants.uniqueMessage()));
        }

        if(validationList.size() > 0){
            ucve.setChildren(validationList);
            throw ucve;
        }
    }

    private void clearMessages(){
        this.validationView.setVisible(false);
        parentPanel.hideFeedback();
    }

    /**
     * Load a category from persistence and display in the UI
     * @param id The ID of the category to load
     */
    protected void selectCategory(long id){
        busyIndicator.center();

        metadataService.getUICategoryNode(id, new NeptuneAsyncCallback<UICategoryNode>() {

            @Override
            public void onNeptuneFailure(Throwable caught) {
                Window.alert(caught.toString());
                StringBuffer strBuff = new StringBuffer();
                StackTraceElement[] stackTrace = caught.getStackTrace();
                for(int i=0; i<stackTrace.length; i++){
                    strBuff.append(stackTrace[i].toString());
                }

                busyIndicator.hide();
            }

            @Override
            public void onNeptuneSuccess(UICategoryNode result) {
                if(result != null){
                    treePanel.select(result.getId(), result.getParentId());
                    showCategory(result, false);
                }
                busyIndicator.hide();
            }});
    }

    private int getMaxLength(long len){
        if(len <= 0) {
            return TEXTBOX_DEFAULT_MAX_LENGTH;
        } else {
            return (int)len;
        }
    }

    /**
     * Remove any +/- columns from a table-displaying Grid
     * @param grid
     */
    private void removeButtonColumns(Grid grid){
        int colsToPare = 0;
        for(int i=0; i<grid.getCellCount(0); i++){
            if("+".equals(grid.getText(0, i)) || "-".equals(grid.getText(0,i))){
                colsToPare++;
            }
        }

        grid.resizeColumns(grid.getColumnCount() - colsToPare);
    }

    private int findWidgetRow(FlexTable table, Widget widget){
        int row=-1;

        for(int i=0; i<table.getRowCount(); i++){
            Widget valueWidget = table.getWidget(i,1);

            if(valueWidget instanceof HorizontalPanel){
                HorizontalPanel horizPanel = (HorizontalPanel)table.getWidget(i, 1);
                if(horizPanel.getWidget(0) != null && horizPanel.getWidget(0).equals(widget)){
                    row = i;
                    break;
                }
            }
        }

        return row;
    }

    private static final class FieldLabel extends Composite {

        private static final String STYLE_REQUIRED = "nwt-FormContainer-required";

        private final HorizontalPanel panel;
        private final Label requiredLabel;
        private boolean required = false;

        public FieldLabel(Label label, boolean required) {
            this.panel = new HorizontalPanel();

            /* Build the required label */
            this.requiredLabel = new LabelWidget();
            this.requiredLabel.setStyleName(STYLE_REQUIRED);
            panel.add(this.requiredLabel);

            /* Add the field label */
            panel.add(label);

            /* Initialize the composite */
            this.initWidget(panel);

            /* Update the label */
            setRequired(required);
        }

        /**
         * Check if the label is marked required
         * 
         * @return true if marked required, false otherwise
         */
        public boolean isRequired() {
            return required;
        }

        /**
         * Mark the label as required (or not required)
         * 
         * @param required true to mark as required, false otherwise
         */
        public void setRequired(boolean required) {
            this.required = required;

            /* Set the text */
            String text = this.required ? "*" : "";
            this.requiredLabel.setText(text);

            /* Set the cell width */
            String width = this.required ? "10" : "";
            panel.setCellWidth(this.requiredLabel, width);
        }
    }
}
