/**
 * 
 */
package com.ericsson.cms.categorymgmt.client;


import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.ericsson.cms.categorymgmt.client.i18n.CategoryConstants;
import com.ericsson.cms.categorymgmt.client.model.UICategoryNode;
import com.ericsson.cms.categorymgmt.client.model.UICategoryTreeNode;
import com.ericsson.cms.categorymgmt.client.model.UIConstants;
import com.ericsson.cms.categorymgmt.client.rpc.ICategoryFieldSchemaManagerService;
import com.ericsson.cms.categorymgmt.client.rpc.ICategoryFieldSchemaManagerServiceAsync;
import com.ericsson.cms.categorymgmt.client.rpc.ICategoryManagerService;
import com.ericsson.cms.categorymgmt.client.rpc.ICategoryManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ContextMenuEvent;
import com.google.gwt.event.dom.client.ContextMenuHandler;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.event.logical.shared.OpenEvent;
import com.google.gwt.event.logical.shared.OpenHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.IsSerializable;
import com.google.gwt.user.client.ui.TreeItem;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.MenuBarWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TreeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.PopupContainer;


/**
 * @author Anuj Jain
 *
 */
public class CategoryTreePanel extends TreeWidget implements IsSerializable{
    private NeptuneSecurity security;
    private TreeItem categoryRoot;
    private CategoryDetailPanel detailView;
    private boolean loaded;

    private final Set<UICategoryTreeNode> openedCategoryNodes = new HashSet<UICategoryTreeNode>();
    private final BusyIndicator busyIndicator = new BusyIndicator();
    private final PopupContainer contextMenuContainer = new PopupContainer(true);

    private final ICategoryManagerServiceAsync service 					= GWT.create(ICategoryManagerService.class);
    private final ICategoryFieldSchemaManagerServiceAsync metadataService = GWT.create(ICategoryFieldSchemaManagerService.class);
    private final CategoryConstants constants 							= GWT.create(CategoryConstants.class);

    public CategoryTreePanel(NeptuneSecurity security){
        this.security = security;

        addOpenHandler(new OpenHandler<TreeItem>(){
            @Override
            public void onOpen(OpenEvent<TreeItem> event) {
                UICategoryTreeNode nodeData = (UICategoryTreeNode) event.getTarget().getUserObject();
                if(nodeData == null) {
                    return;
                }
                openedCategoryNodes.add(nodeData);
                TreeItem openedItem = event.getTarget();
                openedItem.setWidget(CategoryTreeItem.getWidget(nodeData));
            }
        });

        addCloseHandler(new CloseHandler<TreeItem>() {
            @Override
            public void onClose(CloseEvent<TreeItem> event) {
                UICategoryTreeNode nodeData = (UICategoryTreeNode) event.getTarget().getUserObject();
                //if categoryRoot, do nothing
                if(nodeData == null) {
                    return;
                }
                openedCategoryNodes.remove(nodeData);
                TreeItem closedItem = event.getTarget();
                //If the node is collapsed it should also reflect the error status of any of its descendant nodes
                //i.e isLocalError parameter set to false
                closedItem.setWidget(CategoryTreeItem.getWidget(nodeData));
            }});

        addSelectionHandler(new SelectionHandler<TreeItem>() {
            @Override
            public void onSelection(SelectionEvent<TreeItem> event){
                TreeItem selectedItem = event.getSelectedItem();
                UICategoryTreeNode nodeData = (UICategoryTreeNode) selectedItem.getUserObject();
                Long id = 0L;
                if(nodeData != null){
                    id = nodeData.getId();
                }

                History.newItem(buildAnchor(id));
            }});

        setupContextMenu();
        categoryRoot = new CategoryTreeItem(constants.labelCategoryTreeRoot());
        addItem(categoryRoot);

    }

    public void setDetailView(CategoryDetailPanel detailView) {
        this.detailView = detailView;

    }

    private void setupContextMenu() {
        addDomHandler(new ContextMenuHandler() {
            @Override
            public void onContextMenu(ContextMenuEvent event) {
                showContextMenu();
                // Prevent the browser context menu from showing
                event.preventDefault();
                event.stopPropagation();
            }
        }, ContextMenuEvent.getType());
    }

    private void showContextMenu() {

        TreeItem selectedItem = getSelectedItem();
        if(selectedItem == null) {
            return;
        }

        MenuBarWidget menu = new MenuBarWidget(true);
        contextMenuContainer.setWidget(menu);

        boolean hasMenuItems = false;
        //build the menu
        if (security.isUserInRole(Permissions.CATEGORY_CREATE)) {
            hasMenuItems = true;
            // MENU: Create Category
            menu.addItem(constants.contextMenuCreateCategory(), new Command() {
                @Override
                public void execute() {
                    busyIndicator.show();

                    TreeItem selectedItem = getSelectedItem();
                    UICategoryTreeNode treeNode = (UICategoryTreeNode) selectedItem.getUserObject();
                    long parentId = 0;
                    if (treeNode != null) {
                        parentId = treeNode.getId();
                    }
                    hideContextMenu();
                    metadataService.getDefaultUICategory(parentId, new NeptuneAsyncCallback<UICategoryNode>() {

                        @Override
                        public void onNeptuneFailure(Throwable caught) {
                            busyIndicator.hide();
                            Window.alert("Unable to load Field Schema: " + caught.getMessage());
                        }

                        @Override
                        public void onNeptuneSuccess(UICategoryNode result) {
                            loaded = true;
                            detailView.showCategory(result, true);
                            long tempId = Long.valueOf(UIConstants.NEW_CATEGORY_TEMP_ID);
                            // adding temporary id in the history, this is required by the hyperlink & tree to navigate
                            // back to the parent
                            History.newItem(buildAnchor(tempId));
                            busyIndicator.hide();
                        }
                    });
                }
            });
        }

        /** Display 'delete category' option only if:
         * 1. User has delete permission
         * 2. Category does not have any sub category
         * 3. Category is not the root category ('Categories')
         *  */
        if(security.isUserInRole(Permissions.CATEGORY_DELETE)  &&
                getSelectedItem().getChildCount() == 0 &&
                getSelectedItem().getUserObject() != null) {
            hasMenuItems = true;
            //MENU: Delete Category
            menu.addItem(constants.contextMenuDeleteCategory(), new Command() {
                @Override
                public void execute() {
                    if(!Window.confirm(constants.deleteCategoryConfirmMessage(getSelectedItem().getText()))){
                        return;
                    }
                    deleteCategory();
                }});
        }

        //display
        if(hasMenuItems) {
            Widget relativeWidget = selectedItem != null ? selectedItem.getWidget() : getItem(0).getWidget();
            contextMenuContainer.setPopupPosition(relativeWidget.getAbsoluteLeft(),
                    relativeWidget.getAbsoluteTop() + relativeWidget.getOffsetHeight());
            contextMenuContainer.show();
        }

    }

    public void deleteCategory(){

        TreeItem selectedItem = getSelectedItem();
        UICategoryTreeNode treeNode = (UICategoryTreeNode)selectedItem.getUserObject();
        long id = 0;
        long tempId = 0L;
        if(treeNode.getParent() != null){
            tempId = treeNode.getParent().getId();
        }
        final long parentId = tempId;
        if(treeNode != null) {
            id = treeNode.getId();
        }
        hideContextMenu();

        service.deleteCategory(id, new NeptuneAsyncCallback<Boolean>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                busyIndicator.hide();
                Window.alert(caught.getMessage());
            }

            @Override
            public void onNeptuneSuccess(Boolean result) {
                Window.alert(constants.deleteCategorySuccessMessage());
                metadataService.getUICategoryNode(parentId, new NeptuneAsyncCallback<UICategoryNode>() {

                    @Override
                    public void onNeptuneFailure(Throwable caught) {
                        busyIndicator.hide();
                        Window.alert(caught.toString());
                        StringBuffer strBuff = new StringBuffer();
                        StackTraceElement[] stackTrace = caught.getStackTrace();
                        for(int i=0; i<stackTrace.length; i++){
                            strBuff.append(stackTrace[i].toString());
                        }
                    }

                    @Override
                    public void onNeptuneSuccess(UICategoryNode result) {
                        loaded = false;

                        long id = result.getId();

                        History.newItem(buildAnchor(id));

                        select(parentId, result.getParentId());
                        busyIndicator.hide();
                    }});
            }});
    }



    private void hideContextMenu() {
        contextMenuContainer.hide();
    }
    public void reload(final Long categoryIdToSelect, final Long parentCategoryIdToSelect) {
        busyIndicator.center();
        service.getCategoryTree(new NeptuneAsyncCallback<List<UICategoryTreeNode>>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                busyIndicator.hide();
                Window.alert("Unable to load Category Tree");
            }

            @Override
            public void onNeptuneSuccess(List<UICategoryTreeNode> result) {
                loaded = true;
                renderTree(result);

                if(categoryIdToSelect != 0) {
                    selectTreeItem(categoryIdToSelect, parentCategoryIdToSelect);
                } else {
                    selectDefaultCategory();
                }

                busyIndicator.hide();
            }});




    }

    public void select(Long categoryIdToSelect, Long parentCategoryIdToSelect) {
        if(!loaded) {
            reload(categoryIdToSelect, parentCategoryIdToSelect);
        } else {
            if(categoryIdToSelect == 0) {
                clearSelection();
            } else {
                selectTreeItem(categoryIdToSelect, parentCategoryIdToSelect);
            }
        }
    }

    private void renderTree(List<UICategoryTreeNode> treeData) {
        categoryRoot.removeItems();
        addToTreeItem(categoryRoot, treeData);
        categoryRoot.setState(true);

        //maintain open state
        Iterator<UICategoryTreeNode> openedCategoryNodeIter = openedCategoryNodes.iterator();
        while(openedCategoryNodeIter.hasNext()) {
            UICategoryTreeNode openedSiteNode = openedCategoryNodeIter.next();

            //find the tree item in the tree that matches the opened node
            boolean foundAndHasChildren = false;
            Iterator<TreeItem> treeItemIterator = treeItemIterator();
            while(treeItemIterator.hasNext()) {
                TreeItem treeItem = treeItemIterator.next();
                if(openedSiteNode.equals(treeItem.getUserObject())) {
                    if(treeItem.getChildCount() > 0) {
                        //set the tree item to be open
                        treeItem.setState(true, false);
                        foundAndHasChildren = true;
                    }
                    break;
                }
            }
            if(!foundAndHasChildren) {

                // the category node is not there anymore or
                // it does not have children anymore, so, remove from opened set

                openedCategoryNodeIter.remove();
            }
        }
    }

    //adds children recursively
    private void addToTreeItem(TreeItem parentTreeItem, List<UICategoryTreeNode> treeData) {
        for(UICategoryTreeNode nodeData : treeData) {
            CategoryTreeItem treeItem = new CategoryTreeItem(nodeData);
            parentTreeItem.addItem(treeItem);
            if(nodeData.hasChildren()) {
                addToTreeItem(treeItem, nodeData.getChildren());
            }
        }
    }

    private void selectTreeItem(long id, Long parentCategoryId) {
        CategoryTreeItem item = null;
        if(parentCategoryId != 0) {
            CategoryTreeItem parentItem = findFirstTreeItemById(parentCategoryId, categoryRoot);
            if(parentItem != null) {
                item = findFirstTreeItemById(id, parentItem);
            }
        } else {
            item = findFirstTreeItemById(id, categoryRoot);
        }

        if(item != null) {
            // Select item
            item.getTree().setSelectedItem(item, true);
            // Make sure it is expanded and visible
            item.getTree().ensureSelectedItemVisible();
        }
    }


    private void clearSelection() {
        clearSelection(categoryRoot);
    }

    private boolean clearSelection(TreeItem item) {
        if(item == null) {
            return false;
        }

        if(item.isSelected()) {
            item.setSelected(false);
            return true;
        }

        //recurse
        if(item.getChildCount() > 0) {
            for(int i = 0; i < item.getChildCount(); i++) {
                boolean cleared = clearSelection(item.getChild(i));
                if(cleared) {
                    return true;
                }
            }
        }

        return false;
    }

    private CategoryTreeItem findFirstTreeItemById(long id, TreeItem root) {
        // Validation
        if(root == null) {
            return null;
        }

        UICategoryTreeNode uiNode = (UICategoryTreeNode)root.getUserObject();
        // This is the item we are looking for
        if(root instanceof CategoryTreeItem && uiNode != null && id == uiNode.getId()) {
            return (CategoryTreeItem)root;
        }

        // Iterate over children and recursively try to find a node depth first
        if(root.getChildCount() > 0) {
            for(int i = 0; i < root.getChildCount(); i++) {
                CategoryTreeItem item = findFirstTreeItemById(id, root.getChild(i));
                // return first child
                if(item != null) {
                    return item;
                }
            }
        }

        return null;
    }

    private static class CategoryTreeItem extends TreeItem{
        private static final String STYLE_TREEITEM = "category-treeItem";
        private static final String STYLE_TREEITEM_SELECTED = "category-treeItem-selected";

        public CategoryTreeItem(String text){
            super(new LabelWidget(text));
        }

        public CategoryTreeItem(UICategoryTreeNode nodeData){
            super(getWidget(nodeData));
            setUserObject(nodeData);
        }

        private static Widget getWidget(UICategoryTreeNode nodeData) {
            HorizontalContainer container = new HorizontalContainer();
            LabelWidget label = new LabelWidget(nodeData.getName());
            container.setSpacing(2);
            container.addStyleName(STYLE_TREEITEM);
            container.add(label);
            return container;
        }

        @Override
        public void setSelected(boolean selected) {
            super.setSelected(selected);
            if(selected) {
                getWidget().addStyleName(STYLE_TREEITEM_SELECTED);
            } else {
                getWidget().removeStyleName(STYLE_TREEITEM_SELECTED);
            }
        }
    }

    private String buildAnchor(long id){
        Map<String,String> paramMap = new HashMap<String,String>();
        paramMap.put("id", String.valueOf(id));

        AnchorTokenizer at = new AnchorTokenizer();
        String paramAnchor = at.buildAnchor(paramMap);

        return UIConstants.CATEGORY_VIEW_ANCHOR + "." + paramAnchor;
    }

    /**
     * This will select the first node in the category tree
     * by default
     * 
     * */
    public void selectDefaultCategory(){
        if(categoryRoot.getChildCount() > 0)
        {
            TreeItem child = categoryRoot.getChild(0);
            UICategoryTreeNode treeNode = (UICategoryTreeNode)child.getUserObject();
            long id = treeNode.getId();

            selectTreeItem(id, 0l);
            metadataService.getUICategoryNode(id, new NeptuneAsyncCallback<UICategoryNode>() {

                @Override
                public void onNeptuneFailure(Throwable caught) {
                    busyIndicator.hide();
                }

                @Override
                public void onNeptuneSuccess(UICategoryNode result) {
                    loaded = true;
                    detailView.showCategory(result, false);
                    busyIndicator.hide();
                }
            });
        }

    }
}
