package com.tandbergtv.cms.portal.content.client.contentclass;

import java.util.Collection;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.tandbergtv.cms.portal.content.client.contentclass.model.ContentClassValidator;
import com.tandbergtv.cms.portal.content.client.contentclass.model.SaveStatus;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClass;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClassPartner;
import com.tandbergtv.cms.portal.content.client.contentclass.service.IContentClassViewService;
import com.tandbergtv.cms.portal.content.client.contentclass.service.IContentClassViewServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;

/**
 * Handles communication with the server via RPC and updates views
 * @author eyevkar
 */
public class ContentClassController implements IContentClassController {
	// PRC Services
	private IContentClassViewServiceAsync service = GWT.create(IContentClassViewService.class);
	private ContentClassView view;
	
	/**
	 * Constructor
	 */
	public ContentClassController(ContentClassView view) {
		this.view = view;
	}

	@Override
	public void getContentClassList() {
		view.showEmptyView();
		view.showBusyIndicator();

		service.getContentClassListWithoutMetadata(new NeptuneAsyncCallback<List<UIContentClass>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.onError(ContentClassCommand.List, caught.getMessage());
				view.hideBusyindicator();
			}
			
			@Override
			public void onNeptuneSuccess(List<UIContentClass> result) {
				view.showContentClassListView(result);
				view.hideBusyindicator();
			}
		});
	}

	
	@Override
	public void saveContentClass(UIContentClass contentClass) 
	{
		view.showBusyIndicator();
		
		// Validate
		ContentClassValidator validator = new ContentClassValidator(ContentClassValidator.VALIDATOR_GLOBAL_CC);
		validator.validate(contentClass);
		if(validator.hasErrors()) 
		{
			view.onError(ContentClassCommand.Save, validator.getErrors());
			view.hideBusyindicator();
			return;
		}
		
		// Get data from the server
		service.saveAndValidateContentClass(contentClass, new NeptuneAsyncCallback<SaveStatus>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				view.onError(ContentClassCommand.Save, caught.getMessage());
				view.hideBusyindicator();
			}

			@Override
			public void onNeptuneSuccess(SaveStatus status) 
			{
				if(status.errors.isEmpty())
				{
					view.onAfterContentClassSave(status.contentClassId);
					view.hideBusyindicator();
				}
				else
				{
					view.onCCPError(status.errors);
					view.hideBusyindicator();
				}
			}
		});
	}

	
	@Override
	public void deleteContentClasses(Collection<Long> ids) {
		view.showBusyIndicator();
		
		// Get data from the server
		service.deleteContentClassesById(ids, new NeptuneAsyncCallback<List<UIContentClass>>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.onError(ContentClassCommand.Delete, caught.getMessage());
				view.hideBusyindicator();
			}

			@Override
			public void onNeptuneSuccess(List<UIContentClass> result) {
				view.showContentClassListView(result);
				view.hideBusyindicator();
			}
		});
	}

	@Override
	public void getNewContentClass() {
		view.showEmptyView();
		view.showBusyIndicator();
		
		// Get data from the server
		service.getNewContentClass(new NeptuneAsyncCallback<UIContentClass>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.onError(ContentClassCommand.New, caught.getMessage());
				view.hideBusyindicator();
			}

			@Override
			public void onNeptuneSuccess(UIContentClass result) {
				view.showContentClassNewView(result);
				view.hideBusyindicator();
			}
		});
	}

	@Override
	public void getContentClass(long id) {
		view.showEmptyView();
		view.showBusyIndicator();
		
		// Get data from the server
		service.getContentClassById(id, new NeptuneAsyncCallback<UIContentClass>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.onError(ContentClassCommand.Get, caught.getMessage());
				view.hideBusyindicator();
			}

			@Override
			public void onNeptuneSuccess(UIContentClass result) {
				view.showContentClassEditView(result);
				view.hideBusyindicator();
			}
		});
	}

	@Override
	public void getContentClassPartner(long contentClassId, long partnerId) {
		view.showEmptyView();
		view.showBusyIndicator();

		// Get data from the server
		service.getContentClassPartner(contentClassId, partnerId, new NeptuneAsyncCallback<UIContentClassPartner>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.onError(ContentClassCommand.GetPartner, caught.getMessage());
				view.hideBusyindicator();
			}

			@Override
			public void onNeptuneSuccess(UIContentClassPartner result) {
				view.showContentClassPartnerEditView(result);
				view.hideBusyindicator();
			}
		});
	}

	@Override
	public void saveContentClassPartner(UIContentClassPartner ccp) {
		view.showBusyIndicator();
		
		// Validate
		ContentClassValidator validator = new ContentClassValidator(ContentClassValidator.VALIDATOR_PARTNER_CC);
		validator.validate(ccp);
		if(validator.hasErrors()) {
			view.onError(ContentClassCommand.SavePartner, validator.getErrors());
			view.hideBusyindicator();
			return;
		}
		
		// Get data from the server
		service.saveContentClassPartner(ccp, new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				view.onError(ContentClassCommand.SavePartner, caught.getMessage());
				view.hideBusyindicator();
			}

			@Override
			public void onNeptuneSuccess(Void v) {
				view.onAfterContentClassPartnerSave();
				view.hideBusyindicator();
			}
		});
	}

}
