package com.tandbergtv.cms.portal.content.client.contentclass;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.tandbergtv.cms.portal.content.client.Anchors;
import com.tandbergtv.cms.portal.content.client.Permissions;
import com.tandbergtv.cms.portal.content.client.bundle.ContentClassConstants;
import com.tandbergtv.cms.portal.content.client.contentclass.model.CCPValidationError;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClass;
import com.tandbergtv.cms.portal.content.client.contentclass.model.ValidationError;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;


/**
 * A view to edit or create new content class.
 * @author eyevkar
 */
public class ContentClassEditView extends Composite
{
	ContentClassConstants constants = GWT.create(ContentClassConstants.class);
	
	// Controller
	private IContentClassController controller;
	// Content class currently displayed in the view
	private UIContentClass contentClass;
	
	private SimpleContainer rootContainer;
	
	// Main tab
	private ContentClassMainView mainView;
	// Metadata tab
	private ContentClassMetadataView metadataView;
	// View title
	private HeaderPanel header;
	private ResizableContainer resizableContainer;

	private ButtonWidget btnCopy;
	private ButtonWidget btnSave;
	// TODO: Remove when SanjayPanel supports single footer shared by all panels.
	// For now, each tab has its own copy of buttons.
	private ButtonWidget btnCopy2;
	private ButtonWidget btnSave2;
	
	private static final String STYLE = "ContentClassEditView";
	
	private static final String TAB_PROPERTIES = "Properties";
	private static final String TAB_METADATA = "Metadata";

	/**
	 * Constructor
	 */
	public ContentClassEditView(IContentClassController pController) 
	{
		this.controller = pController;
		
		rootContainer = new SimpleContainer();
		this.initWidget(rootContainer);

		this.addStyleName(STYLE);
		
		// Screen title
		header = new HeaderPanel("");
		
		// Properties Tab
		mainView = new ContentClassMainView();
		resizableContainer = new ResizableContainer(mainView, TAB_PROPERTIES, null);
		resizableContainer.addHeightOffsettingWidget(header);
		btnSave = addSaveButton(TAB_PROPERTIES);
		btnCopy = addCopyButton(TAB_PROPERTIES);
		addCancelButton(TAB_PROPERTIES);
		
		// Metadata Tab
		metadataView = new ContentClassMetadataView();
		resizableContainer.addTab(metadataView.getLeftWidget(), metadataView.getRightWidget(), TAB_METADATA);
		btnSave2 = addSaveButton(TAB_METADATA);
		btnCopy2 = addCopyButton(TAB_METADATA);
		addCancelButton(TAB_METADATA);
		
		// Main panel
		VerticalPanel vp = new VerticalPanel();
		vp.add(header);
		vp.add(resizableContainer);
		
		rootContainer.setWidget(vp);
	}

	
	private void onSaveClick()
	{
		String name = mainView.getName();
		if(name == null || name.trim().length() < 1) 
		{
			resizableContainer.selectTab(TAB_PROPERTIES);
			resizableContainer.setErrorMessage(constants.errorBlankCCName());
			return;
		}

		if(controller != null) 
		{
			contentClass.setName(name);
			contentClass.setDescription(mainView.getDescription());
			contentClass.setSpecialUsageId(mainView.getSpecialUsageId());
			controller.saveContentClass(contentClass);
		}
	}
	
	private void onCopyClick()
	{
		resizableContainer.reset();
		
		contentClass.setId(0);
		contentClass.setDefault(false);
		// Copy latest values from main view
		contentClass.setName(mainView.getName());
		contentClass.setDescription(mainView.getDescription());
		contentClass.setSpecialUsageId(mainView.getSpecialUsageId());
		// Update name
		String oldName = contentClass.getName();
		if(oldName == null || !oldName.startsWith("Copy of ")) 
		{
			contentClass.setName("Copy of " + oldName);
		}
		
		History.newItem(Anchors.CONTENT_CLASS_NEW, false);
		setNewTitle();
		mainView.refresh(contentClass, true);
		resizableContainer.selectTab(TAB_PROPERTIES);
		resizableContainer.reset();
	}

	private void onCancelClick()
	{
		History.newItem(Anchors.CONTENT_CLASS_LIST);
	}
	
	
	private ButtonWidget addSaveButton(String tabName)
	{
		// Save
		ButtonWidget btnSave = new ButtonWidget("Save");
		btnSave.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		btnSave.addClickHandler(new ClickHandler() 
		{
			@Override
			public void onClick(ClickEvent event) 
			{
				onSaveClick();
			}
		});
		resizableContainer.addButton(tabName, btnSave);
		
		return btnSave;
	}
	
	
	private ButtonWidget addCopyButton(String tabName)
	{
		ButtonWidget btnCopy = new ButtonWidget("Copy");
		btnCopy.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		btnCopy.addClickHandler(new ClickHandler() 
		{
			@Override
			public void onClick(ClickEvent event) 
			{
				onCopyClick();
			}
		});
		resizableContainer.addButton(tabName, btnCopy);
		
		return btnCopy;
	}
	
	
	private void addCancelButton(String tabName)
	{
		ButtonWidget btnCancel = new ButtonWidget("Cancel");
		btnCancel.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		btnCancel.addClickHandler(new ClickHandler() 
		{
			@Override
			public void onClick(ClickEvent event) 
			{
				onCancelClick();
			}
		});
		resizableContainer.addButton(tabName, btnCancel);
	}
	
	
	/**
	 * Refreshes the view with content class
	 * @param contentClass
	 * @param isNew true if a new content class is being created
	 */
	public void refresh(UIContentClass contentClass, boolean isNew) 
	{
		resizableContainer.reset();
		this.contentClass = contentClass;

		boolean isEditable;
		// Permissions
		if(isNew) 
		{
			isEditable = ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_CREATE);
			setNewTitle();
		}
		else 
		{
			isEditable = ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_MODIFY);
			setEditTitle(contentClass.getName());
		}
		
		btnSave.setEnabled(isEditable);
		btnSave2.setEnabled(isEditable);
		// Only enable copy for PACKAGE content class
		btnCopy.setEnabled(contentClass.getTypeId() == 0);
		btnCopy2.setEnabled(contentClass.getTypeId() == 0);
		
		mainView.refresh(contentClass, isEditable);
		metadataView.refresh(contentClass, isEditable);

		resizableContainer.selectTab(TAB_PROPERTIES);
		Scheduler.get().scheduleDeferred(new ScheduledCommand() 
		{
			@Override
			public void execute() 
			{
				resizableContainer.updateSize();
			}
		});
	}
	
	
	/**
	 * Sets title for new title view
	 */
	public void setNewTitle() 
	{
		header.setLabel("New Content Class");
		btnSave.setEnabled(ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_CREATE));
		btnCopy.setEnabled(false);
	}


	/**
	 * Sets title for edit title view
	 */
	public void setEditTitle(String name)
	{
		header.setLabel("Edit Content Class: " + name);
		btnCopy.setEnabled(contentClass.getTypeId() == 0 
				&& ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_CREATE));
	}


	/**
	 * Gets called after content class is saved
	 * @param id
	 */
	public void afterSave(Long id) 
	{
		// New content class was saved (old id = 0)
		if(contentClass.getId() == 0) 
		{
			contentClass.setId(id);
			History.newItem(Anchors.CONTENT_CLASS_EDIT + "." + id, false);
		}
		
		resizableContainer.setInfoMessage(constants.ccSaved());
		setEditTitle(contentClass.getName());
	}


	public void onError(String message) 
	{
		resizableContainer.setErrorMessage(message);
	}


	public void onError(List<ValidationError> errors) 
	{
		if(errors == null || errors.isEmpty()) return;
		
		StringBuilder bld = new StringBuilder();
		bld.append("<p>Please fix the following validation errors before saving:</p>");
		bld.append("<ul>");
		
		for(int i = 0; i < errors.size(); i++) 
		{
			bld.append("<li>");
			ValidationError error = errors.get(i);
			
			if(error.getField() != null)
			{
				bld.append(error.getField().getDisplayName());
				bld.append(" ");
			}
			
			bld.append("[");
			bld.append(error.getAsset().getDisplayName());
			bld.append("]: ");
			bld.append(error.getMessage());
			
			bld.append("</li>");
		}
		bld.append("</ul>");
		
		resizableContainer.setErrorHTML(bld.toString());
	}


	public void onCCPError(List<CCPValidationError> errors) 
	{
		if(errors == null || errors.isEmpty()) return;

		StringBuilder bld = new StringBuilder();
		bld.append("<p>Could not disable fields with default values at the partner level:</p>");

		for(int i = 0; i < errors.size(); i++) 
		{
			CCPValidationError error = errors.get(i);

			bld.append("<li>");
			bld.append("Partner: " + error.partnerId + ", Asset: " + error.assetPath + ", Field: " + error.fieldPath);
			bld.append("</li>");
		}
		
		resizableContainer.setErrorHTML(bld.toString());
	}

}
