package com.tandbergtv.cms.portal.content.client.contentclass;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HasAlignment;
import com.google.gwt.user.client.ui.HasHorizontalAlignment.HorizontalAlignmentConstant;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.tandbergtv.cms.portal.content.client.Permissions;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClass;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.ResizableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;


/**
 * Shows a list of content classes in a grid
 * @author eyevkar
 */
public class ContentClassListView extends Composite 
{
	private IContentClassController controller;
	
	private Set<CheckBox> checkboxes = new HashSet<CheckBox>();
	
	private SimpleContainer rootContainer;
	private FlexTable table;

	ResizableContainer resizableContainer;
	
	private static final String TAB_NAME = "Content Classes";
	
	/**
	 * Construcotor
	 */
	public ContentClassListView(final IContentClassController controller) 
	{
		this.controller = controller;
		rootContainer = new SimpleContainer();
		this.initWidget(rootContainer);
		this.addStyleName("ContentClassListView");
		
		VerticalPanel mainPanel = new VerticalPanel();
		
		// Title
		HeaderPanel header = new HeaderPanel("Content Class Administration");
		mainPanel.add(header);
		
		// Data grid
		table = new FlexTable();
		table.addStyleName("cc-list-table");
		createTableHeader();
		resizableContainer = new ResizableContainer(table, TAB_NAME, null);
		resizableContainer.addHeightOffsettingWidget(header);
		mainPanel.add(resizableContainer);
		
		// Buttons
		addButtons(resizableContainer);
		
		rootContainer.setWidget(mainPanel);
	}
	
	
	private void addButtons(final ResizableContainer resizableContainer) 
	{
		// New
		ButtonWidget btnNew = new ButtonWidget("Create");
		btnNew.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		if(ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_CREATE)) 
		{
			btnNew.addClickHandler(new ClickHandler() 
			{
				@Override
				public void onClick(ClickEvent event) 
				{
					History.newItem("Content.ContentClass.New");
				}
			});
		}
		else 
		{
			btnNew.setEnabled(false);
		}
		
		// Delete
		ButtonWidget btnDelete = new ButtonWidget("Delete");
		btnDelete.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		if(ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_DELETE)) 
		{
			btnDelete.addClickHandler(new ClickHandler() 
			{
				@Override
				public void onClick(ClickEvent event) {
					List<Long> selectedIds = getSelectedContentClassIds();
					if(selectedIds.isEmpty()) 
					{
						resizableContainer.setInfoMessage("No content class is selected. There is nothing to delete.");
						return;
					}
					if(controller != null && !selectedIds.isEmpty()) 
					{
						if(Window.confirm("Are you sure you want to delete selected content classes?")) 
						{
							controller.deleteContentClasses(selectedIds);
						}
					}
				}
			});
		}
		else 
		{
			btnDelete.setEnabled(false);
		}
		
		resizableContainer.addButton(TAB_NAME, btnNew);
		resizableContainer.addButton(TAB_NAME, btnDelete);
	}
	
	private void createTableHeader() 
	{
		final CheckBox mcb = new CheckBox();
		mcb.addClickHandler(new ClickHandler() 
		{
			@Override
			public void onClick(ClickEvent event) 
			{
				boolean selected = mcb.getValue();
				for(CheckBox cb: checkboxes) 
				{
					cb.setValue(selected);
				}
			}
		});
		table.setWidget(0, 0, mcb);
		table.setWidget(0, 1, new Label("ID"));
		table.setWidget(0, 2, new Label("Name"));
		table.setWidget(0, 3, new Label("Description"));
		table.setWidget(0, 4, new Label("Special Usage"));
		
		table.getRowFormatter().setStyleName(0, "tr-header");
	}
	
	
	private void createTableDataRow(int row, final UIContentClass contentClass) 
	{
		// Add checkbox for non-default PACKAGE content classes
		if(contentClass.getTypeId() == 0 && !contentClass.isDefault())
		{
			CheckBox cb = new CheckBox();
			cb.setFormValue(""+contentClass.getId());
			checkboxes.add(cb);
			table.setWidget(row, 0, cb);
		}		
		// ID
		table.setWidget(row, 1, new HyperlinkWidget("" + contentClass.getId(), "Content.ContentClass.Edit." + contentClass.getId()));
		// Name
		table.setWidget(row, 2, new Label(contentClass.getName()));
		
		// Description
		String descr = contentClass.getDescription();
		// If description is longer than 100 chars, show first 100 chars followed by ...
		if(descr != null && descr.length() > 100) {
			descr = descr.substring(0, 100) + "...";
		}
		table.setWidget(row, 3, new Label(descr));
		
		// Special usage
		String lbl = "";
		switch(contentClass.getSpecialUsageId()) 
		{
			case 1: lbl = "Pre Roll"; break;
			case 2: lbl = "Post Roll"; break;
		}
		table.setWidget(row, 4, new Label(lbl));
		
		table.getRowFormatter().setStyleName(row, "tr-body");
		table.getCellFormatter().setHorizontalAlignment(row, 0, HasAlignment.ALIGN_CENTER);
		table.getCellFormatter().setHorizontalAlignment(row, 1, HasAlignment.ALIGN_RIGHT);
	}
	
	
	public void refresh(List<UIContentClass> list) 
	{
		resizableContainer.getMessageArea().reset();
		
		checkboxes.clear();
		table.removeAllRows();
		createTableHeader();
		
		if(list != null) 
		{
			Collections.sort(list);
			for(int i = 0; i < list.size(); i++) 
			{
				UIContentClass cc = list.get(i);
				createTableDataRow(i+1, cc);
			}
		}
		
		Scheduler.get().scheduleDeferred(new ScheduledCommand() 
		{
			@Override
			public void execute() 
			{
				resizableContainer.updateSize();
			}
		});
	}
	
	private List<Long> getSelectedContentClassIds() {
		List<Long> ids = new ArrayList<Long>();

		for(CheckBox cb: checkboxes) {
			if(cb.getValue()) {
				Long id = Long.parseLong(cb.getFormValue());
				ids.add(id);
			}
		}

		return ids;
	}
	
	public void onError(ContentClassCommand cmd, String message) 
	{
		resizableContainer.setErrorMessage(message);
	}

}
