package com.tandbergtv.cms.portal.content.client.contentclass.dialog;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.MenuBar;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.Permissions;
import com.tandbergtv.cms.portal.content.client.bundle.IContentClassResourceBundle;
import com.tandbergtv.cms.portal.content.client.contentclass.model.KeyValuePair;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClass;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClassField;
import com.tandbergtv.cms.portal.content.client.contentclass.widget.ContentClassFieldWidget;
import com.tandbergtv.cms.portal.content.client.contentclass.widget.ImageAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.DateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;


/**
 * A dialog to filter suggested values of a content class field 
 * @author eyevkar
 */
public class FieldPropertiesDialog extends DialogBox 
{
	private static final String TIP_MIN = "Enter positive integer from 1 to 99. Leave the field blank to use default (1) value.";
	private static final String TIP_MAX = "Enter positive integer from 1 to 99. Leave the field blank to use default (unlimited) value.";
	private static final String ERROR_STYLE = "WidgetValidationError";

	private static final IContentClassResourceBundle bundle = GWT.create(IContentClassResourceBundle.class);
	
	private ContentClassFieldWidget widget;

	private MessageArea messageArea;
	
	private TextBox txtName;
	private TextBox txtDataType;
	
	private ListBox cbFieldType;
	
	private TextBox txtMin;
	private boolean minIsValid = true;
	
	private TextBox txtMax;
	private boolean maxIsValid = true;
	
	private CheckBox cbBatch;
	
	private CheckBox cbAutofillable;
	private ListBox listAutofillProvider;

	private FlexTable optionTable;
	
	private TextBox txtDisplayFormat;
	private boolean displayFormatIsValid = true;
	
	private Button btnSave;
	
	/**
	 * Constructor
	 * @param widget
	 */
	public FieldPropertiesDialog(final ContentClassFieldWidget widget) 
	{
		super();
		
		this.widget = widget;
		this.addStyleName("FieldPropertiesDialog");
		
		setModal(true);
		// Dialog box's caption
	    setText("Properties");
	    //setAnimationEnabled(true);
	    setGlassEnabled(true);
	    
	    VerticalPanel rootPanel = new VerticalPanel();
	    
		messageArea = new MessageArea();
		rootPanel.add(messageArea);
		messageArea.reset();
		
	    // Main view
	    Panel mainView = createMainView();
	    
	    // Content
    	Panel optionsView = createOptionsView();
    	
    	TabContainer tabContainer = new TabContainer();
		tabContainer.add(mainView, "Main");
		tabContainer.add(optionsView, "Options");
		tabContainer.selectTab(0);
    	
		rootPanel.add(tabContainer);

		// Disable options tab and field type selector for DATE and TIME data types
		UIContentClassField.DataType dataType = widget.getField().getDataType(); 
		if(dataType == UIContentClassField.DataType.DATE || dataType == UIContentClassField.DataType.TIME)
		{
			tabContainer.getTabBar().setTabEnabled(1, false);
			cbFieldType.setEnabled(false);
			cbFieldType.setSelectedIndex(0);
		}
		else if(dataType == UIContentClassField.DataType.BOOLEAN)
		{
			cbFieldType.setEnabled(false);
			cbFieldType.setSelectedIndex(2);
		}
		
	    // Buttons
	    Widget buttonPanel = createButtonPanel();
	    rootPanel.add(buttonPanel);

	    // Refresh UI with data
	    refresh();
	    
	    setWidget(rootPanel);
	}

	
	private ContentClassFieldWidget getContentClassFieldWidget() 
	{
		return widget;
	}
	
	
	private void refresh() 
	{
		if(widget == null) return;
		UIContentClassField field = widget.getField();
		if(field == null) return;
	
		boolean canSave = ClientAuthorizationManager.isAuthorized(Permissions.CONTENT_CLASS_VIEW, Permissions.CONTENT_CLASS_MODIFY);
		btnSave.setEnabled(canSave);
		
		txtName.setValue(field.getDisplayName());
		txtDataType.setValue(field.getDataType().name());

		// Field Type
		String type = field.getFieldType().toString();
		for(int i = 1; i < cbFieldType.getItemCount(); i++) 
		{
			if(cbFieldType.getValue(i).equals(type)) 
			{
				cbFieldType.setSelectedIndex(i);
				break;
			}
		}
		
		// Cardinality
		if(field.isMultiValue()) 
		{
			txtMin.setValue(field.getMin() == null ? "" : String.valueOf(field.getMin()));
			txtMax.setValue(field.getMax() == null ? "" : String.valueOf(field.getMax()));
		}
		
		// Batch
		if(cbBatch != null) 
		{
			cbBatch.setValue(field.isIncludeInBatch());
		}
		
		// Auto fill checkbox
		if(cbAutofillable != null) 
		{
			cbAutofillable.setValue(field.isAutofillable());
		
			// Autofill dropdown
			String provider = field.getAutofillProvider();
			if(provider != null) 
			{
				for(int i = 1; i < listAutofillProvider.getItemCount(); i++) 
				{
					if(listAutofillProvider.getValue(i).equals(provider)) 
					{
						listAutofillProvider.setSelectedIndex(i);
						break;
					}
				}
			}
		}
		
		// Display Format
		if(UIContentClassField.DataType.DATE == field.getDataType() || UIContentClassField.DataType.TIME == field.getDataType()) 
		{
			txtDisplayFormat.setValue(field.getDisplayFormat());
			cbFieldType.setEnabled(false);
		}

		// If the field has suggested values (options)
		if(field.canEditOptions()) 
		{
		    refreshOptionTable(field); 
		}
	}
	

	
	private void refreshOptionTableHeader(UIContentClassField field)
	{
		// Header check box
		final CheckBox mcb = new CheckBox();
		mcb.addClickHandler(new ClickHandler() 
		{
			@Override
			public void onClick(ClickEvent event) 
			{
				boolean selected = mcb.getValue();
				
				for(int row = 1; row < optionTable.getRowCount(); row++) 
				{
					CheckBox cb = (CheckBox)optionTable.getWidget(row, 0);
					cb.setValue(selected);
				}
			}
		});
		
		// Header
		optionTable.setWidget(0, 0, mcb);
		optionTable.setWidget(0, 1, new Label("Value"));
		optionTable.setWidget(0, 2, new Label("Display Name"));

		// Set CSS styles
		for(int i = 0; i <= 2; i++) 
		{
			optionTable.getCellFormatter().setStyleName(0, i, "portlet-tableHeader");
		}

		// Only add 3rd column for options which are not generated by options provider
		if(widget.getField().canAddOptions())
		{
			Anchor headerContextMenuAnchor = new ImageAnchor(createContextMenuImage());
			HeaderContextMenuClickHandler headerContextMenuHandler = new HeaderContextMenuClickHandler();
			headerContextMenuAnchor.addClickHandler(headerContextMenuHandler);
			optionTable.setWidget(0, 3, headerContextMenuAnchor);
			
			// TODO: Fix CSS. Context menu image doesn't show for some reason if CSS style is "portlet-tableHeader". 
			optionTable.getCellFormatter().setStyleName(0, 3, "portlet-table-text");
		}
	}
	
	
	private void refreshOptionTableData(UIContentClassField field)
	{
	    // Data
	    List<KeyValuePair> list = field.getOptions();
	    if(list == null) return;
	    
	    // Elements to be excluded from list of options in edit title screen
	    Set<String> excludedItems = field.getFilteredOptionKeys(); 
	    
	    int row = 1;
	    for(KeyValuePair kvp: list) 
	    {
	    	CheckBox cb = new CheckBox();
		    // If this item is not in the list of excluded items, select it
	    	if(excludedItems == null || !excludedItems.contains(kvp.getValue())) 
	    	{
	    		cb.setValue(true);
	    	}
	    	
	    	optionTable.setWidget(row, 0, cb);
		    optionTable.setWidget(row, 1, new Label(kvp.getValue()));
		    optionTable.setWidget(row, 2, new Label(kvp.getKey()));

		    // Set CSS styles
			for(int i = 0; i < 3; i++) 
			{
				optionTable.getCellFormatter().setStyleName(row, i, "portlet-table-text");
			}

		    // Only add 3rd column for options which are not generated by options provider 
		    if(widget.getField().canAddOptions())
		    {
				ImageAnchor contextMenuAnchor = new ImageAnchor(createContextMenuImage());
				ContextMenuClickHandler contextMenuHandler = new ContextMenuClickHandler(row, kvp.getValue(), kvp.getKey());
				contextMenuAnchor.setContextMenuClickHandler(contextMenuHandler);
		    	optionTable.setWidget(row, 3, contextMenuAnchor);
		    	optionTable.getCellFormatter().setStyleName(row, 3, "portlet-table-text");
		    }
		    
	    	++row;
	    }
	}
	
	
	private void refreshOptionTable(UIContentClassField field)
	{
		optionTable.clear();

		refreshOptionTableHeader(field);
		refreshOptionTableData(field);
	}
	
	
	private Panel createMainView() 
	{
		VerticalPanel panel = new VerticalPanel();
		panel.addStyleName("ContentClassEditView");
		
		if(widget == null || widget.getField() == null) return panel;
		
		final UIContentClassField field = widget.getField();
		
		FlexTable table = new FlexTable();
		table.addStyleName("FormPanel");
		table.addStyleName("FormFieldTable");
		
		int row = 0;

		// Name
		txtName = new TextBox();
		txtName.setEnabled(false);
		table.setWidget(row, 0, new Label("Field Name"));
		table.setWidget(row, 1, txtName);
		row++;
		
		// Data Type
		txtDataType = new TextBox();
		txtDataType.setEnabled(false);
		table.setWidget(row, 0, new Label("Data Type"));
		table.setWidget(row, 1, txtDataType);
		row++;
		
		// Field Type
		cbFieldType = new ListBox();
		cbFieldType.setVisibleItemCount(1);
		cbFieldType.addItem("Text Box", "TEXT_BOX");
		cbFieldType.addItem("Text Area", "TEXT_AREA");
		cbFieldType.addItem("Drop Down", "DROP_DOWN");
		cbFieldType.addItem("Combo Box", "COMBO_BOX");
		
		table.setWidget(row, 0, new Label("Field Type"));
		table.setWidget(row, 1, cbFieldType);
		row++;
		
		UIContentClassField uiField = widget.getField(); 
		
		// Cardinality text boxes are only available if a field supports multiple values
		if(uiField.isMultiValue()) 
		{
			// Min
			createMinField();
			table.setWidget(row, 0, new Label("Min Occurrence"));
			table.setWidget(row, 1, txtMin);
			row++;

			// Max
			createMaxField();
			table.setWidget(row, 0, new Label("Max Occurrence"));
			table.setWidget(row, 1, txtMax);
			row++;
		}
		
		// Batch checkbox only available for fields with cardinality = 1
		if(uiField.isMultiValue() == false || (uiField.getMax() != null && uiField.getMax() == 1))
		{
			cbBatch = new CheckBox();
			table.setWidget(row, 0, new Label("Include in Batch"));
			table.setWidget(row, 1, cbBatch);
			row++;
		}

		// Autofill is only available for fields which can have default value (editable)
		if(uiField.canHaveDefaultValue())
		{
			// Auto fill checkbox
			cbAutofillable = new CheckBox();
			
			//Autofill providers
			listAutofillProvider = new ListBox();
			// Empty value
			listAutofillProvider.addItem("", (String) null);
			// Values from content class
			if(widget != null) {
				UIContentClass cc = widget.getContentClass(); 
				if(cc != null) {
					List<KeyValuePair> providers = cc.getAutofillProviders();
					if(providers != null) {
						for(KeyValuePair kvp: providers) {
							listAutofillProvider.addItem(kvp.getValue(), kvp.getKey());
						}
					}
				}
			}
			listAutofillProvider.setSelectedIndex(0);
			
			table.setWidget(row, 0, new Label("Auto Fillable"));
			table.setWidget(row, 1, cbAutofillable);
			row++;
			table.setWidget(row, 0, new Label("Autofill Provider"));
			table.setWidget(row, 1, listAutofillProvider);
			row++;
		}
		
		// Display pattern
		if(UIContentClassField.DataType.DATE == field.getDataType()) 
		{		
			txtDisplayFormat = new TextBox();
			addDateChangeHandler(field);
			table.setWidget(row, 0, new Label("Display Format"));
			table.setWidget(row, 1, txtDisplayFormat);
			row++;
		}
		else if(UIContentClassField.DataType.TIME == field.getDataType())
		{
			txtDisplayFormat = new TextBox();
			addTimeChangeHandler(field);
			table.setWidget(row, 0, new Label("Display Format"));
			table.setWidget(row, 1, txtDisplayFormat);
			row++;
		}
		
		panel.add(table);
		return panel;
	}
	
	
	private void createMinField()
	{
		// Min
		txtMin = new TextBox();
		txtMin.setMaxLength(2);
		txtMin.setTitle(TIP_MIN);
		txtMin.addValueChangeHandler(new ValueChangeHandler<String>() 
		{
			@Override
			public void onValueChange(ValueChangeEvent<String> event) 
			{
				Widget widget = (Widget)event.getSource();
				String value = trim(event.getValue());
				
				if(!isEmpty(value))
				{
					Integer iVal = parseInt(value);
					if(iVal == null || iVal < 1) {
						widget.addStyleName(ERROR_STYLE);
						minIsValid = false;
						return;
					}
				}
				
				widget.removeStyleName(ERROR_STYLE);
				minIsValid = true;
			}
		});		
	}
	
	
	private void createMaxField()
	{
		// Max
		txtMax = new TextBox();
		txtMax.setMaxLength(2);
		txtMax.setTitle(TIP_MAX);
		txtMax.addValueChangeHandler(new ValueChangeHandler<String>() {
			@Override
			public void onValueChange(ValueChangeEvent<String> event) {
				Widget widget = (Widget)event.getSource();
				String value = trim(event.getValue());
				
				if(!isEmpty(value))
				{
					Integer iVal = parseInt(value);
					if(iVal == null || iVal < 1) {
						widget.addStyleName(ERROR_STYLE);
						maxIsValid = false;
						return;
					}
				}
				
				widget.removeStyleName(ERROR_STYLE);
				maxIsValid = true;
			}
		});
	}
	
	
	private void addDateChangeHandler(final UIContentClassField field)
	{
		txtDisplayFormat.addValueChangeHandler(new ValueChangeHandler<String>() 
		{
			@Override
			public void onValueChange(ValueChangeEvent<String> event) {
				Widget widget = (Widget)event.getSource();
				String value = trim(event.getValue());
				
				DateTimeFormat df = null;
				
				if(!isEmpty(value))
				{
					try
					{
						// Validate format
						df = DateTimeFormat.getFormat(value);
						df.format(new Date());
					}
					catch(Exception ex)
					{
						widget.addStyleName(ERROR_STYLE);
						displayFormatIsValid = false;
						return;
					}
				}
				
				widget.removeStyleName(ERROR_STYLE);
				displayFormatIsValid = true;
				field.setDisplayFormat(value);
				
				ContentClassFieldWidget ccWidget = getContentClassFieldWidget();
				CCDateWidget dateWidget = (CCDateWidget)ccWidget.getGwtWidget();
				DateConverter dc = (DateConverter)dateWidget.getDateConverter();					
				dc.setOutputFormat(value);
				
				String displayValue = null;
				
				try
				{
					Date date = dc.getDateForInputValue(field.getValue());
					displayValue = dc.getDisplayValue(date);
				}
				catch(Exception ex)
				{
				}
				
				dateWidget.setTextValue(displayValue);
			}
		});			
	}
	
	
	private void addTimeChangeHandler(final UIContentClassField field)
	{
		txtDisplayFormat.addValueChangeHandler(new ValueChangeHandler<String>() 
		{
			@Override
			public void onValueChange(ValueChangeEvent<String> event) {
				Widget widget = (Widget)event.getSource();
				String value = trim(event.getValue());
				
				DateTimeFormat df = null;
				
				if(!isEmpty(value))
				{
					try
					{
						// Validate format
						df = DateTimeFormat.getFormat(value);
						df.format(new Date());
					}
					catch(Exception ex)
					{
						widget.addStyleName(ERROR_STYLE);
						displayFormatIsValid = false;
						return;
					}
				}
				
				widget.removeStyleName(ERROR_STYLE);
				displayFormatIsValid = true;
				field.setDisplayFormat(value);
				
				field.setValue(null);
				ContentClassFieldWidget ccWidget = getContentClassFieldWidget();
				TextBox timeWidget = (TextBox)ccWidget.getGwtWidget();
				
				if(isEmpty(value))
				{
					timeWidget.setTitle("Please enter a time with format: HH:mm:ss");
				}
				else
				{
					timeWidget.setTitle("Please enter a time with format: " + value);
				}

				// Clear the value
				timeWidget.setText("");
			}
		});
	}
	
	
	private Panel createOptionsView() {
	    // Main content panel
	    VerticalPanel contentPanel = new VerticalPanel();
	    contentPanel.addStyleName("Content");
	    
	    // List box
	    createOptionTable();
	    
	    // Flex table with radio buttons and a list
	    VerticalPanel formPanel = new VerticalPanel();
	    formPanel.addStyleName("FormFieldTable");
	    formPanel.add(optionTable);
	    
	    contentPanel.add(formPanel);
	    
	    return contentPanel;
	}
	
	private void createOptionTable() {
	    // Create table header
	    optionTable = new FlexTable();
	    optionTable.addStyleName("portletTable");
	}
	
	private Widget createButtonPanel() 
	{
	    // Buttons
	    // Save
	    btnSave = new Button("Save");
	    btnSave.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
	    btnSave.addClickHandler(new ClickHandler() 
	    {
	    	public void onClick(ClickEvent event) 
	    	{
	    		onSave();
	        }
	    });
	    
	    // Cancel
	    Button btnCancel = new Button("Cancel");
	    btnCancel.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
	    btnCancel.addClickHandler(new ClickHandler() 
	    {
	    	public void onClick(ClickEvent event) 
	    	{
	    		onCancel();
	        }
	    });

	    // Button panel
	    HorizontalPanel buttonPanel = new HorizontalPanel();
	    buttonPanel.addStyleName("ButtonPanel");
	    buttonPanel.add(btnSave);
	    buttonPanel.add(btnCancel);
	    return buttonPanel;
	}
	
	private boolean validate() 
	{
		// Validate options
		Set<String> valueSet = new HashSet<String>();
		Set<String> nameSet = new HashSet<String>();
		
		// row = 0 is the table header
		for(int row = 1; row < optionTable.getRowCount(); row++) 
		{
			CheckBox cb = (CheckBox)optionTable.getWidget(row, 0);
			Label lblValue = (Label)optionTable.getWidget(row, 1);
			Label lblDisplayName = (Label)optionTable.getWidget(row, 2);

			String value = lblValue.getText();
			String displayName = lblDisplayName.getText();
			
			// Validate values
			if(valueSet.contains(value))
			{
				messageArea.setErrorMessage("Option value is not unique: " + value);
				return false;
			}
			else
			{
				valueSet.add(value);
			}
			
			// Validate names
			if(nameSet.contains(displayName))
			{
				messageArea.setErrorMessage("Option name is not unique: " + displayName);
				return false;
			}
			else
			{
				nameSet.add(displayName);
			}
		}

		messageArea.reset();
		
		if(!minIsValid) return false;
		if(!maxIsValid) return false;
		if(!displayFormatIsValid) return false;
		
		return true;
	}
	
	private void onSave() 
	{
		if(!validate()) 
		{
			return;
		}
		
		setMainValues();
		setOptions();
		
		widget.refresh();
		
		// Hide the dialog
		this.hide();
	}
	
	private void onCancel() 
	{
		// Hide the dialog
		this.hide();
	}
	

	private void setMainValues() 
	{
		UIContentClassField uiField = widget.getField();
		
		// FieldType
		int idx = cbFieldType.getSelectedIndex();
		String value = cbFieldType.getValue(idx);
		uiField.setFieldType(value);
		
		
		// Cardinality
		if(uiField.isMultiValue()) 
		{
			uiField.setMin(parseInt(txtMin.getValue()));
			uiField.setMax(parseInt(txtMax.getValue()));
		}

		// Batch
		if(cbBatch != null) 
		{
			uiField.setIncludeInBatch(cbBatch.getValue());
		}
		
		// Auto fill checkbox
		if(cbAutofillable != null) 
		{
			uiField.setAutofillable(cbAutofillable.getValue());
		
			// Autofill provider
			try 
			{
				idx = listAutofillProvider.getSelectedIndex();
				value = listAutofillProvider.getValue(idx);
				uiField.setAutofillProvider(value);
			}
			catch(Exception ex) 
			{
				uiField.setAutofillProvider(null);
			}
		}
		
		// Dsiplay format
		if(UIContentClassField.DataType.DATE == uiField.getDataType() || UIContentClassField.DataType.TIME == uiField.getDataType()) 
		{
			uiField.setDisplayFormat(trim(txtDisplayFormat.getValue()));
		}
	}
	
	
	private static String trim(String str) 
	{
		return (str == null) ? null : str.trim();
	}
	
	private static boolean isEmpty(String str)
	{
		return (str == null || str.length() < 1);
	}
	
	private static Integer parseInt(String str) 
	{
		if(str == null) return null;
		str = str.trim();
		try {
			return Integer.parseInt(str);
		}
		catch(Exception ex) {
			return null;
		}
	}
	
	private void setOptions() 
	{
		if(optionTable == null) return;
		
		List<KeyValuePair> options = new ArrayList<KeyValuePair>();
		Set<String> filteredKeys = new HashSet<String>();
	
		for(int row = 1; row < optionTable.getRowCount(); row++) 
		{
			CheckBox cb = (CheckBox)optionTable.getWidget(row, 0);
			Label lblValue = (Label)optionTable.getWidget(row, 1);
			Label lblDisplayName = (Label)optionTable.getWidget(row, 2);

			String value = lblValue.getText();
			String displayName = lblDisplayName.getText();
			options.add(new KeyValuePair(displayName, value));
			
			// Check box is not selected. Exclude this item.
			if(cb.getValue() == false)
			{
				filteredKeys.add(value);
			}
		}

		widget.getField().setOptions(options);
		widget.getField().setFilteredOptionKeys(filteredKeys);
		
		// If option corresponding to old default value was disabled, set default value = null
		String oldValue = widget.getField().getValue();
		if(oldValue != null && filteredKeys.contains(oldValue)) widget.getField().setValue(null);
	}
	
	private Image createContextMenuImage() {
		Image img = new Image(bundle.contextMenuImage());
		img.addStyleName("ContextMenuImage");
		return img;
	}

	
	private void deleteOption(int index)
	{
		if(index > 0) 
		{
			optionTable.removeRow(index);
			// Update indices
			updateIndices();
		}
	}
	
	
	private void updateIndices()
	{
		for(int row = 1; row < optionTable.getRowCount(); row++) 
		{
			ImageAnchor anchor = (ImageAnchor)optionTable.getWidget(row, 3);
			ContextMenuClickHandler handler = anchor.getContextMenuClickHandler();
			handler.setIndex(row);
		}
	}


	private void changeOption(int rowIndex, String value, String displayName)
	{
		if(rowIndex > 0 && rowIndex < optionTable.getRowCount())
		{
			Label lblValue = (Label)optionTable.getWidget(rowIndex, 1);
			lblValue.setText(value.trim());

			Label lblName = (Label)optionTable.getWidget(rowIndex, 2);
			lblName.setText(displayName.trim());
		}
	}
	
	
	/**
	 * 
	 * @param newValue new value
	 * @param newDisplayName new display name
	 * @param currentOptionIndex
	 * @return
	 */
	private String validateOptionBase(String newValue, String newDisplayName)
	{
		if(newValue == null || newValue.trim().length() < 1)
		{
			return "Value could not be empty";
		}
		
		if(newDisplayName == null || newDisplayName.trim().length() < 1)
		{
			return "Display name could not be empty";
		}

		// Validate typed values
		UIContentClassField.DataType type = this.widget.getField().getDataType();
		
		if(type == UIContentClassField.DataType.BOOLEAN)
		{
			if( !(newValue.equalsIgnoreCase("true") || newValue.equalsIgnoreCase("false")) )
			{
				return "Invalid boolean value. Valid values are 'true' or 'false'";
			}
		}
		else if(type == UIContentClassField.DataType.INTEGER)
		{
			try 
			{ 
				Integer.parseInt(newValue); 
			}
			catch(Exception ex)
			{
				return "Invalid integer value";
			}
		}
		else if(type == UIContentClassField.DataType.FLOAT)
		{
			try 
			{ 
				Double.parseDouble(newValue); 
			}
			catch(Exception ex)
			{
				return "Invalid float value";
			}
		}

		return null;
	}
	

	private String validateOptionNew(String newValue, String newDisplayName)
	{
		String error = validateOptionBase(newValue, newDisplayName);
		if(error != null)
		{
			return error;
		}
		
		// Validate uniqueness
		newValue = newValue.trim();
		newDisplayName = newDisplayName.trim();
		
		// row = 0 is the table header
		for(int row = 1; row < optionTable.getRowCount(); row++) 
		{
			CheckBox cb = (CheckBox)optionTable.getWidget(row, 0);
			Label lblValue = (Label)optionTable.getWidget(row, 1);
			Label lblDisplayName = (Label)optionTable.getWidget(row, 2);

			if(newValue.equals(lblValue.getText()))
			{
				return "Value is not unique";
			}

			if(newDisplayName.equals(lblDisplayName.getText()))
			{
				return "Display Name is not unique";
			}
		}
		
		return null;
	}
	

	private String validateOptionEdit(String newValue, String newDisplayName,
			int optionIndex, String oldValue, String oldDisplayName)
	{
		String error = validateOptionBase(newValue, newDisplayName);
		if(error != null)
		{
			return error;
		}
		
		// Validate uniqueness
		newValue = newValue.trim();
		newDisplayName = newDisplayName.trim();

		boolean skipOldValueCheck = newValue.equals(oldValue);
		boolean skipOldDisplayNameCheck = newDisplayName.equals(oldDisplayName);
		
		// row = 0 is the table header
		for(int row = 1; row < optionTable.getRowCount(); row++) 
		{
			CheckBox cb = (CheckBox)optionTable.getWidget(row, 0);
			Label lblValue = (Label)optionTable.getWidget(row, 1);
			Label lblDisplayName = (Label)optionTable.getWidget(row, 2);

			if(newValue.equals(lblValue.getText()))
			{
				if(!(skipOldValueCheck && row == optionIndex))
				{
					return "Value is not unique";
				}
			}

			if(newDisplayName.equals(lblDisplayName.getText()))
			{
				if(!(skipOldDisplayNameCheck && row == optionIndex))
				{
					return "Display Name is not unique";
				}
			}
		}

		return null;
	}
	
	
	private void addOption(String value, String displayName)
	{
    	CheckBox cb = new CheckBox();
		cb.setValue(true);
    	
		ImageAnchor contextMenuAnchor = new ImageAnchor(createContextMenuImage());
		ContextMenuClickHandler contextMenuHandler = new ContextMenuClickHandler(optionTable.getRowCount(), 
				value.trim(), displayName.trim());
		contextMenuAnchor.setContextMenuClickHandler(contextMenuHandler);

		int row = optionTable.getRowCount();
		
    	optionTable.setWidget(row, 0, cb);
	    optionTable.setWidget(row, 1, new Label(value));
	    optionTable.setWidget(row, 2, new Label(displayName));
	    optionTable.setWidget(row, 3, contextMenuAnchor);
	    
	    // Set CSS styles
		for(int i = 0; i <= 3; i++) {
			optionTable.getCellFormatter().setStyleName(row, i, "portlet-table-text");
		}		
	}
	
	////////////////////////////////////////////////////////////////
	
	private class HeaderContextMenuClickHandler implements ClickHandler, IEditOptionDialogListener
	{
		HeaderContextMenuClickHandler() {
		}
		
		@Override
		public void onClick(ClickEvent event) {
			final PopupPanel menuPanel = new PopupPanel(true);
			menuPanel.addStyleName("ContentClassPopupPanel");

			int x = event.getNativeEvent().getClientX() + Window.getScrollLeft();
			int y = event.getNativeEvent().getClientY() + Window.getScrollTop();

			menuPanel.setPopupPosition(x, y);
			
			MenuBar menu = new MenuBar(true);

			menu.addItem("Add New...", new Command() {
				@Override
				public void execute() {
					menuPanel.hide();
					
					EditOptionDialog dialog = new EditOptionDialog(HeaderContextMenuClickHandler.this);
					dialog.refresh();
					dialog.center();
				}
			});
			
			menuPanel.add(menu);
			menuPanel.show();
		}
		
		@Override
		public String onSave(String newValue, String newDisplayName) 
		{
			// Validate
			String validationError = validateOptionNew(newValue, newDisplayName);
			if(validationError != null) return validationError;
			
			// No errors
			addOption(newValue, newDisplayName);
			return null;
		}
	}
	
	////////////////////////////////////////////////////////////////
	
	public class ContextMenuClickHandler implements ClickHandler, IEditOptionDialogListener 
	{
		private String optionValue;
		private String optionDisplayName;
		private int optionIndex = -1;
		private int lastCommand;
		
		private static final int CMD_EDIT = 1;
		
		
		ContextMenuClickHandler(int index, String value, String displayName)
		{
			this.optionIndex = index;
			this.optionValue = value;
			this.optionDisplayName = displayName;
		}
		
		public void setIndex(int index)
		{
			this.optionIndex = index;
		}
		
		@Override
		public void onClick(ClickEvent event) {
			final PopupPanel menuPanel = new PopupPanel(true);
			menuPanel.addStyleName("ContentClassPopupPanel");

			int x = event.getNativeEvent().getClientX() + Window.getScrollLeft();
			int y = event.getNativeEvent().getClientY() + Window.getScrollTop();

			menuPanel.setPopupPosition(x, y);
			
			MenuBar menu = new MenuBar(true);

			menu.addItem("Edit...", new Command() {
				@Override
				public void execute() {
					menuPanel.hide();
					
					lastCommand = CMD_EDIT;
					
					EditOptionDialog dialog = new EditOptionDialog(ContextMenuClickHandler.this);
					dialog.refresh(optionValue, optionDisplayName);
					dialog.center();
				}
			});
			
			menu.addItem("Delete", new Command() {
				@Override
				public void execute() {
					deleteOption(optionIndex);
					menuPanel.hide();
				}
			});
			
			menuPanel.add(menu);
			menuPanel.show();
		}

		@Override
		public String onSave(String newValue, String newDisplayName) 
		{
			switch(lastCommand)
			{
			case CMD_EDIT:
				// Validate
				String validationError = validateOptionEdit(newValue, newDisplayName, 
						optionIndex, this.optionValue, this.optionDisplayName);
				if(validationError != null) return validationError;
				
				// No errors
				this.optionValue = newValue.trim();
				this.optionDisplayName = newDisplayName.trim();

				changeOption(optionIndex, newValue, newDisplayName);
				return null;
			}
			
			return null;
		}		
	}
}
