package com.tandbergtv.cms.portal.content.client.contentclass.model;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.tandbergtv.cms.portal.ui.title.client.view.datatype.factory.DataTypeWidgetMessages;

/**
 * Validates content class metadata on the client.
 * @author eyevkar
 */
public class ContentClassValidator {
	public static final byte VALIDATOR_GLOBAL_CC = 0;
	public static final byte VALIDATOR_PARTNER_CC = 1;

	private static DataTypeWidgetMessages messages = GWT.create(DataTypeWidgetMessages.class);
	
	private List<ValidationError> errors;
	private int maxErrors;
	// An asset is being processed 
	private UIContentClassAsset currentAsset;
	
	private byte validatorType = VALIDATOR_GLOBAL_CC;
	
	/**
	 * Constructor
	 */
	public ContentClassValidator(byte validatorType) {
		this.validatorType = validatorType;
		errors = new ArrayList<ValidationError>();
		maxErrors = 10;
	}
	
	public boolean hasErrors() {
		return !errors.isEmpty();
	}
	
	public List<ValidationError> getErrors() {
		return errors;
	}
	
	/**
	 * Validates a content class.
	 * @param cc
	 */
	public void validate(UIContentClass cc) {
		errors.clear();
		currentAsset = null;
		
		if(cc == null) return;
		
		validateAsset(cc.getRootAsset());
	}

	/**
	 * Validates a content class partner.
	 * @param cc
	 */
	public void validate(UIContentClassPartner ccp) {
		errors.clear();
		currentAsset = null;
		
		if(ccp == null) return;
		
		validateAsset(ccp.getRootAsset());
	}

	
	private void validateAsset(UIContentClassAsset asset) {
		if(asset == null) return;
		if(errors.size() >= maxErrors) return;
		
		currentAsset = asset;
		
		// Validate asset cardinality for global CC
		if(validatorType == VALIDATOR_GLOBAL_CC) {
			validateAssetCardinality(asset);
		}
		
		// Validate fields
		validateFields(asset.getFields());
		
		// Validate child assets
		List<UIContentClassAsset> childAssets = asset.getAssets();
		if(childAssets == null) return;
		
		for(UIContentClassAsset childAsset: childAssets) {
			validateAsset(childAsset);
		}
	}

	
	private void validateAssetCardinality(UIContentClassAsset asset)
	{
		boolean parseError = false;
		
		if(asset.strMin != null)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Invalid Min Occurrence value. Enter an integer value greater than or equal to 0.");
			errors.add(error);
			parseError = true;
		}

		if(asset.strMax != null)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Invalid Max Occurrence value. Enter an integer value greater than or equal to 1.");
			errors.add(error);
			parseError = true;
		}
		
		if(asset.strBatchMin != null)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Invalid Min Batch Occurrence value. Enter an integer value greater than or equal to 0.");
			errors.add(error);
			parseError = true;
		}

		if(asset.strBatchMax != null)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Invalid Max Batch Occurrence value. Enter an integer value greater than or equal to 0.");
			errors.add(error);
			parseError = true;
		}

		if(parseError) return;
		
		if(asset.getMax() > asset.getMaxAllowed())
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Max Occurrence value could not be greater than " + asset.getMaxAllowed());
			errors.add(error);
		}
		
		if(asset.getMin() > asset.getMax())
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Max Occurrence value could not be less than Min Occurrence");
			errors.add(error);
		}
		
		if(asset.getMin() < 0)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Min Occurrence value should be between 0 and " + asset.getMaxAllowed());
			errors.add(error);			
		}
		
		if(asset.getMax() < 0)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Max Occurrence value should be between 0 and " + asset.getMaxAllowed());
			errors.add(error);			
		}
		
		/////////////////////////////////////////////////////////////////////
		// Btach
		/////////////////////////////////////////////////////////////////////
		
		if(asset.getBatchMax() > asset.getMaxAllowed())
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Max Batch Occurrence value could not be greater than " + asset.getMaxAllowed());
			errors.add(error);
		}
		
		if(asset.getBatchMin() > asset.getBatchMax())
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Max Batch Occurrence value could not be less than Min Occurrence");
			errors.add(error);
		}
		
		if(asset.getBatchMin() < 0)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Min Batch Occurrence value should be between 0 and " + asset.getMaxAllowed());
			errors.add(error);			
		}
		
		if(asset.getBatchMax() < 0)
		{
			ValidationError error = new ValidationError(currentAsset, null);
			error.setMessage("Max Batch Occurrence value should be between 0 and " + asset.getMaxAllowed());
			errors.add(error);			
		}
		
	}
	
	private void validateFields(List<UIContentClassField> fields) {
		if(fields == null) return;
		
		for(UIContentClassField field: fields) {
			if(errors.size() >= maxErrors) return;
			if(field instanceof UIContentClassFieldGroup) {
				validateFields(((UIContentClassFieldGroup)field).getFields());
			}
			else {
				validateField(field);
			}
		}
	}
	
	private void validateField(UIContentClassField field) {
		if(field == null) return;
		
		switch(field.getDataType()) {
		case DATE:
			if(!field.isValid()) {
				ValidationError error = new ValidationError(currentAsset, field);
				
				String displayFormatPattern = field.getDisplayFormat();
				if (displayFormatPattern == null || displayFormatPattern.trim().length() == 0) {
					displayFormatPattern = messages.dateWidgetFormat();
				}

				error.setMessage(messages.dateWidgetErrorToolTip(displayFormatPattern));
				errors.add(error);
			}
			break;
		case TIME:
			if(!field.isValid()) 
			{
				ValidationError error = new ValidationError(currentAsset, field);
			
				if(field.isTimeCodeType())
				{
					error.setMessage(messages.timeCodeWidgetErrorToolTip());
				}
				else
				{
					String displayFormatPattern = field.getDisplayFormat();
					if(displayFormatPattern == null || displayFormatPattern.trim().length() == 0) 
					{
						displayFormatPattern = messages.timeWidgetFormat();
					}
					error.setMessage(messages.timeWidgetErrorToolTip(displayFormatPattern));
				}				
				
				errors.add(error);
			}
			break;
		case INTEGER:
			if(!field.isValid()) {
				ValidationError error = new ValidationError(currentAsset, field);
				error.setMessage(messages.integerWidgetErrorToolTip());
				errors.add(error);
			}
			break;
		case FLOAT:
			if(!field.isValid()) {
				ValidationError error = new ValidationError(currentAsset, field);
				error.setMessage(messages.floatWidgetErrorToolTip());
				errors.add(error);
			}
			break;
		default:
			break;
		}
	}

	
	/**
	 * Validates value in timecode format: 
	 * @param value
	 * @return
	 */
	public static boolean isTimeCodeValid(String value)
	{
		if(value.length() != 11) return false;
		
		if(value.charAt(2) != ':') return false;
		if(value.charAt(5) != ':') return false;
		if(value.charAt(8) != ':') return false;
		
		try
		{
			int hh = Integer.parseInt(value.substring(0,2));
			int mm = Integer.parseInt(value.substring(3,5));
			int ss = Integer.parseInt(value.substring(6,8));
			Integer.parseInt(value.substring(9));
			
			if(hh > 23) return false;
			if(mm > 60) return false;
			if(ss > 60) return false;
			
			return true;
		}
		catch(Exception ex)
		{
			return false;
		}
	}
	
}
