package com.tandbergtv.cms.portal.content.client.contentclass.model;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.rpc.IsSerializable;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIDateFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UITimeFieldDefinition;

/**
 * UI model of content class field 
 * @author eyevkar
 */
public class UIContentClassField implements IsSerializable 
{
	public enum DataType implements IsSerializable { STRING, BOOLEAN, INTEGER, FLOAT, DATE, TIME }
	public enum FieldType implements IsSerializable { TEXT_BOX, TEXT_AREA, DROP_DOWN, COMBO_BOX };

	private static final KeyValuePair.KeyComparator keyComparator = new KeyValuePair.KeyComparator();
	
	private static final DateTimeFormat dateFormat = getFormat(UIDateFieldDefinition.DATE_PATTERN);
	private static final DateTimeFormat timeFormat = getFormat(UITimeFieldDefinition.TIME_PATTERN);
	
	private String path;
	private String displayName;

	private String displayFormat;
	private DataType dataType = DataType.STRING;
	private FieldType fieldType = FieldType.TEXT_BOX;

	// None of the field's parameters can be edited, 
	// including locking, disabling, editing cardinality, etc.
	private boolean readOnly;
	
	private Integer min;
	private Integer max;
	private boolean multiValue;

	private boolean includeInBatch;
	
	private boolean isAutofillable;
	private String autofillProvider;

	private boolean isAttribute;
	private boolean isRequired;
	
	/** 
	 * All sugested values / options for this field 
	 */
	private List<KeyValuePair> options;

	private String value;
	private boolean isLocked;

	private boolean disabled = true;
	private boolean hideInUI;

	private boolean canAddOptions = false;
	
	
	/* 
	 * If this UIField is used to display content class partner information, the following values represent 
	 * corresponding values from a content class, which partner is overriding.
	 * NOTE: if parent (content class) value is locked, content class partner could not unlock it.
	 */
	private String parentValue;
	private boolean isParentLocked;
	
	/**
	 * Keys of filetered subset of sugested values. If null, show all suggested values in GUI.
	 */
	private Set<String> filteredOptionKeys;
	
	/**
	 * Default constructor. Required by GWT to serialize the object.
	 */
	public UIContentClassField() {
	}
	
	/**
	 * Constructor
	 */
	public UIContentClassField(String path) {
		this.path = path;
	}

	public String getPath() {
		return path;
	}

	public String getValue() {
		return value;
	}

	public void setValue(String value) {
		this.value = value;
	}

	public String getDisplayName() {
		return displayName;
	}
	
	public void setDisplayName(String displayName) {
		this.displayName = displayName;
	}

	public boolean isLocked() {
		return isLocked;
	}

	public void setLocked(boolean isLocked) 
	{
		this.isLocked = isLocked;
	}

	public List<KeyValuePair> getOptions() 
	{
		return options;
	}

	public boolean hasOptions() 
	{
		return (options != null && !options.isEmpty());
	}

	
	public boolean canEditOptions() 
	{
		if(dataType == UIContentClassField.DataType.DATE || dataType == UIContentClassField.DataType.TIME) return false;
		return true;
	}

	
	public void setOptions(List<KeyValuePair> suggestedValues) 
	{
		this.options = suggestedValues;
		
		// Sort
		// Key = Display label
		if(options != null)
		{
			Collections.sort(this.options, keyComparator);
		}
	}

	public Set<String> getFilteredOptionKeys() 
	{
		return filteredOptionKeys;
	}
	
	public void setFilteredOptionKeys(Collection<String> keys) 
	{
		// Add keys
		if(keys != null && !keys.isEmpty()) 
		{
			// Lazily initialize
			if(filteredOptionKeys == null) 
			{
				filteredOptionKeys = new HashSet<String>();
			}
			// Clear keys
			filteredOptionKeys.clear();
			// Add new keys
			filteredOptionKeys.addAll(keys);
		}
		else {
			filteredOptionKeys = null;
		}
	}

	public boolean hasFilteredOptions() 
	{
		return (filteredOptionKeys != null && !filteredOptionKeys.isEmpty());
	}
	
	public void addFilteredOptionKeys(Collection<String> keys) {
		if(keys == null || keys.isEmpty()) return;
		// Lazily initialize
		if(filteredOptionKeys == null) {
			filteredOptionKeys = new HashSet<String>();
		}
		filteredOptionKeys.addAll(keys);
	}
	
	public DataType getDataType() {
		return dataType;
	}

	public void setDataType(DataType dataType) {
		this.dataType = dataType;
	}

	public void setDataType(String strDataType) {
		try {
			this.dataType = DataType.valueOf(strDataType.toUpperCase());
		}
		catch(Exception ex) {
			this.dataType = DataType.STRING;
		}
	}

	public FieldType getFieldType() {
		return fieldType;
	}

	public void setFieldType(FieldType fieldType) {
		this.fieldType = fieldType;
	}

	public void setFieldType(String strFieldType) 
	{
		try 
		{
			this.fieldType = FieldType.valueOf(strFieldType.toUpperCase());
		}
		catch(Exception ex) 
		{
			this.fieldType = FieldType.TEXT_BOX;
		}
	}
	
	public String getDisplayFormat() {
		return displayFormat;
	}

	public void setDisplayFormat(String displayFormat) {
		this.displayFormat = displayFormat;
	}

	public String getParentValue() {
		return parentValue;
	}

	public void setParentValue(String parentValue) {
		this.parentValue = parentValue;
	}

	public boolean isParentLocked() {
		return isParentLocked;
	}

	public void setParentLocked(boolean isParentLocked) {
		this.isParentLocked = isParentLocked;
	}

	public boolean isEmpty() {
		return (value == null || value.trim().length() < 1);
	}

	public boolean isReadOnly() {
		return readOnly;
	}

	public void setReadOnly(boolean readOnly) {
		this.readOnly = readOnly;
	}

	
	public boolean hasNonDefaultValues() 
	{
		// Field has default value
		if(!isEmpty()) return true;

		// Min / Max
		if(getMin() != null || getMax() != null) return true;

		// Locked / Hidden
		if(isLocked()) return true;
		
		// Auto fill
		if(isAutofillable()) return true;
		if(autofillProvider != null && autofillProvider.trim().length() > 0) return true;
		
		// Display Format
		if(displayFormat != null && displayFormat.trim().length() > 0) return true;
		
		// Batch
		if(isIncludeInBatch()) return true;
		
		// Field has suggested values (options)
		if(options!= null && !options.isEmpty()) return true;
		
		// Field has filtered suggested values
		if(hasFilteredOptions()) return true;

		return false;
	}


	public boolean hasNonDefaultPartnerValues() 
	{
		// Field has default value
		if(!isEmpty()) return true;

		// Locked
		if(isLocked()) return true;
		
		// Field has filtered suggested values
		if(hasFilteredOptions()) return true;

		return false;
	}
	
	
	public boolean isValid() 
	{
		switch(this.dataType) 
		{
		case STRING:
			return true;
		case INTEGER:
			if(isEmpty()) return true;
			try {
				Integer.parseInt(value.trim());
				return true;
			}
			catch(Exception ex) {
				return false;
			}
		case FLOAT:
			if(isEmpty()) return true;
			try {
				Double.parseDouble(value.trim());
				return true;
			}
			catch(Exception ex) {
				return false;
			}
		case DATE:
			if(isEmpty()) return true;
			try {
				dateFormat.parse(value.trim());
				return true;
			}
			catch(Exception ex) {
				return false;
			}
		case TIME:
			if(isEmpty()) return true;
			
			// Time code
			if(isTimeCodeType())
			{
				return ContentClassValidator.isTimeCodeValid(value.trim());
			}
			// Regular time
			else
			{
				try 
				{
					if(displayFormat == null || displayFormat.isEmpty())
					{
						timeFormat.parse(value.trim());
					}
					else
					{
						DateTimeFormat customFormat = getFormat(displayFormat);
						customFormat.parse(value.trim());
					}
					
					return true;
				}
				catch(Exception ex) 
				{
					return false;
				}
			}
		}
		
		return true;
	}

	private static DateTimeFormat getFormat(String format) {
		if (GWT.isClient())
			return DateTimeFormat.getFormat(format);
		return null;
	}

	public boolean isDisabled() {
		return disabled;
	}
	
	public boolean isEnabled()
	{
		return !isDisabled();
	}

	public void setDisabled(boolean isHidden) {
		this.disabled = isHidden;
	}

	
	public boolean hideInUI() {
		return hideInUI;
	}

	public void setHideInUI(boolean hide) {
		this.hideInUI = hide;
	}
	
	
	public Integer getMin() {
		return min;
	}

	public void setMin(Integer min) {
		this.min = min;
	}

	public Integer getMax() {
		return max;
	}

	public void setMax(Integer max) {
		this.max = max;
	}

	public boolean isIncludeInBatch() {
		return includeInBatch;
	}

	public void setIncludeInBatch(boolean includeInBatch) {
		this.includeInBatch = includeInBatch;
	}

	public boolean isAutofillable() {
		return isAutofillable;
	}

	public void setAutofillable(boolean isAutofillable) {
		this.isAutofillable = isAutofillable;
	}

	public String getAutofillProvider() {
		return autofillProvider;
	}

	public void setAutofillProvider(String autofillProvider) {
		this.autofillProvider = autofillProvider;
	}


	public boolean isMultiValue() {
		return multiValue;
	}

	public void setMultiValue(boolean multiValue) {
		this.multiValue = multiValue;
	}

	/**
	 * A field can not have default value if
	 *     1. a field is read-only 
	 * @return
	 */
	public boolean canHaveDefaultValue() {
		if(isReadOnly()) return false;
		return true;
	}

	public boolean isAttribute() {
		return isAttribute;
	}

	public void setAttribute(boolean isAttribute) {
		this.isAttribute = isAttribute;
	}

	public boolean isRequired() {
		return isRequired;
	}

	public void setRequired(boolean isRequired) {
		this.isRequired = isRequired;
	}

	public boolean canAddOptions() {
		return canAddOptions;
	}

	public void setCanAddOptions(boolean canAddOptions) {
		this.canAddOptions = canAddOptions;
	}

	// TODO: Create new type: TIMECODE
	public boolean isTimeCodeType()
	{
		if(path.endsWith("/TimeCodeOne") || path.endsWith("/TimeCodeTwo") 
				|| path.endsWith("/TimeCodeThree") || path.endsWith("/TimeCodeFour"))
		{
			return true;
		}
		else
		{
			return false;
		}
	}
}
