package com.tandbergtv.cms.portal.content.client.contentclass.widget;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.MenuBar;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.bundle.IContentClassResourceBundle;
import com.tandbergtv.cms.portal.content.client.contentclass.dialog.PartnerFieldPropertiesDialog;
import com.tandbergtv.cms.portal.content.client.contentclass.dialog.FieldPropertiesDialog;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClass;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClassField;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ComboBoxWidget;

/**
 * Displays content class field
 * @author eyevkar
 */
public class ContentClassFieldWidget extends Composite 
{
	public enum Type { ContentClass, ContentClassPartner };
	
	private static final IContentClassResourceBundle bundle = GWT.create(IContentClassResourceBundle.class);
	private SimplePanel rootPanel;
	private HorizontalPanel hp;
	
	// Image icons
	private Image lockImage;
	private Image disabledImage;
	private Image hideInUIOnlyImage;
	private Image autofillImage; 
	
	// Label widget corresponding to this field
	private HTML label;
	
	private Widget simpleWidget;
	
	// Data model
	private Type type;
	private UIContentClassField field;
	
	private UIContentClass contentClass;
	
	// Field type before field properties dialog was shown
	private UIContentClassField.FieldType prevFieldType;
	
	/**
	 * Constructor
	 * @param field data source
	 */
	public ContentClassFieldWidget(Type type, UIContentClassField field, 
			boolean isEditable, final UIContentClass cc, HTML label) 
	{
		this.type = type;
		this.field = field;
		this.contentClass = cc;
		this.prevFieldType = field.getFieldType();
		
		this.label = label;
		
		this.rootPanel = new SimplePanel();
		this.initWidget(rootPanel);
		
		// Find out if the widget is editable
		boolean widgetIsEditable = isEditable;
		if(isEditable) 
		{
			widgetIsEditable = field.canHaveDefaultValue();
		}

		// Create widget
		simpleWidget = ContentClassWidgetFactory.createSimpleFieldWidget(field, widgetIsEditable);
		
		// Create horizontal panel
		hp = new HorizontalPanel();
		// Add widget
		hp.add(simpleWidget);
		
		// Add context menu
		if(widgetIsEditable) 
		{
			/*
			 * There are only two possible menu items 1. Lock/Unlock and  2. Edit Options...
			 * If none of these options are available, don't show the menu anchor.
			 * Lock/Unlock is only available if field.isParentLock() == false
			 * Edit Options is only available if field.hasSuggestedValues() == true
			 */
			if(field.isParentLocked() == false || field.hasOptions()) 
			{
				Anchor contextMenuAnchor = new ImageAnchor(createContextMenuImage());
				ContextMenuClickHandler contextMenuHandler = new ContextMenuClickHandler(this);
				contextMenuAnchor.addClickHandler(contextMenuHandler);
				hp.add(contextMenuAnchor);
			}
		}
		// Non-editable
		else {
			// Even if value is not editable content class can be locked or disabled
			if(type == Type.ContentClass) {
				Anchor contextMenuAnchor = new ImageAnchor(createContextMenuImage());
				ContextMenuClickHandler contextMenuHandler = new ContextMenuClickHandler(this);
				contextMenuAnchor.addClickHandler(contextMenuHandler);
				hp.add(contextMenuAnchor);
			}
		}

		// Disabled image
		disabledImage = new Image(bundle.getDisabledImage());
		hp.add(disabledImage);
		if(field.isDisabled()) 
		{
			this.disable();
		}
		else 
		{
			this.enable();
		}

		// Hide in UI Only image
		hideInUIOnlyImage = new Image(bundle.getHideImage());
		hp.add(hideInUIOnlyImage);
		if(field.hideInUI()) 
		{
			this.hideInUI(true);
		}
		else
		{
			this.hideInUI(false);
		}

		// Autofillable image
		autofillImage = new Image(bundle.getGearImage());
		autofillImage.setTitle("Auto generated field");
		hp.add(autofillImage);
		refreshAutofillIcon();
		
		// Lock image
		lockImage = new Image(bundle.getLockImage());
		hp.add(lockImage);
		if(field.isLocked() || field.isParentLocked()) 
		{
			this.lock();
		}
		else 
		{
			this.unlock();
		}
		if(field.isParentLocked()) 
		{
			lockImage.setTitle("Locked by Content Class");
		}
		
		// Info
		if(field.getParentValue() != null) 
		{
			Image infoImage = new Image(bundle.getInfoImage());
			infoImage.setTitle("Content Class Value: " + field.getParentValue());
			hp.add(infoImage);
		}
		
		rootPanel.setWidget(hp);
	}
	
	
	public UIContentClassField getField() 
	{
		return field;
	}

	public UIContentClass getContentClass() 
	{
		return contentClass;
	}

	public Widget getGwtWidget()
	{
		return simpleWidget;
	}
	
	/**
	 * Locks the field (changes the model and updates look & feel - lock icon)
	 */
	public void lock() 
	{
		field.setLocked(true);
		lockImage.setVisible(true);
	}

	/**
	 * Unlocks the field. (changes the model and updates look & feel - lock icon)
	 */
	public void unlock() 
	{
		field.setLocked(false);
		lockImage.setVisible(false);
	}

	public void disable() 
	{
		field.setDisabled(true);
		disabledImage.setVisible(true);
	}
	
	public void enable() 
	{
		field.setDisabled(false);
		disabledImage.setVisible(false);
	}
	
	public void setRequired(boolean flag)
	{
		field.setRequired(flag);
		
		String html = null;
		if(flag == true)
		{
			html = "<span class='nwt-FormContainer-required'>* </span>" + field.getDisplayName();
		}
		else
		{
			html = field.getDisplayName();
		}
		
		// Add xpath tooltip
		html = "<span title='" + field.getPath() + "'>" + html + "</span>";

		label.setHTML(html);
	}
	
	private void refreshAutofillIcon()
	{
		autofillImage.setVisible(field.isAutofillable());
	}
	
	
	public void hideInUI(boolean hide) 
	{
		field.setHideInUI(hide);
		hideInUIOnlyImage.setVisible(hide);
	}
	
	/**
	 * Checks if this field is locked
	 * @return
	 */
	public boolean isLocked() {
		return field.isLocked();
	}
	
	/**
	 * Checks if the widget has suggested values (if it is a drop down box)
	 * @return
	 */
	public boolean canEditOptions()
	{
		UIContentClassField.DataType dataType = field.getDataType(); 
		if(dataType == UIContentClassField.DataType.DATE || dataType == UIContentClassField.DataType.TIME) return false;
		return true;
	}
	
	/**
	 * Reloads widget's data, e.g. list box items.
	 */
	public void refresh() 
	{
		// Recreate the widget if field type changed. 
		if(field.getFieldType() != prevFieldType)
		{
			prevFieldType = field.getFieldType();
			simpleWidget = ContentClassWidgetFactory.createSimpleFieldWidget(field, true);
			hp.remove(0);
			hp.insert(simpleWidget, 0);
		}
		
		if(simpleWidget instanceof ListBox) 
		{
			ContentClassWidgetFactory.reloadData((ListBox)simpleWidget, field);
		}
		else  if(simpleWidget instanceof ComboBoxWidget) 
		{
			ContentClassWidgetFactory.reloadData((ComboBoxWidget)simpleWidget, field);
		}
 
		
		refreshAutofillIcon();
	}
	
	private Image createContextMenuImage() 
	{
		Image img = new Image(bundle.contextMenuImage());
		img.addStyleName("ContextMenuImage");
		return img;
	}

	/**
	 * Click handler for field context menu 
	 * @author eyevkar
	 */
	private static class ContextMenuClickHandler implements ClickHandler {
		private ContentClassFieldWidget widget;
		
		/**
		 * Constructor
		 * @param field
		 */
		public ContextMenuClickHandler(ContentClassFieldWidget widget) {
			this.widget = widget;
		}
		
		@Override
		public void onClick(ClickEvent event) {
			final PopupPanel menuPanel = new PopupPanel(true);
			menuPanel.addStyleName("ContentClassPopupPanel");

			int x = event.getNativeEvent().getClientX() + Window.getScrollLeft();
			int y = event.getNativeEvent().getClientY() + Window.getScrollTop();

			menuPanel.setPopupPosition(x, y);
			
			MenuBar menu = new MenuBar(true);
			
			// Lock/Unlock command
			// NOTE: Only available if parent is unlocked. 
			// Content class partner could not override content class lock flag.
			if(!widget.getField().isParentLocked()) {
				if(widget.isLocked()) {
					menu.addItem("Unlock", new Command() {
						@Override
						public void execute() {
							widget.unlock();
							menuPanel.hide();
						}
					});
				}
				else {
					menu.addItem("Lock", new Command() {
						@Override
						public void execute() {
							widget.lock();
							menuPanel.hide();
						}
					});
				}
			}

			// Disable/Enable command (only available in gloabal content class)
			if(widget.type == Type.ContentClass) 
			{
				if(widget.getField().isDisabled()) {
					menu.addItem("Enable", new Command() {
						@Override
						public void execute() {
							widget.enable();
							menuPanel.hide();
						}
					});
				}
				else {
					menu.addItem("Disable", new Command() {
						@Override
						public void execute() {
							widget.disable();
							menuPanel.hide();
						}
					});
				}
			}

			// Hide in UI
			if(widget.type == Type.ContentClass) 
			{
				if(widget.getField().hideInUI()) 
				{
					menu.addItem("Show in UI", new Command() 
					{
						@Override
						public void execute() 
						{
							widget.hideInUI(false);
							menuPanel.hide();
						}
					});
				}
				else 
				{
					menu.addItem("Hide in UI", new Command() 
					{
						@Override
						public void execute() 
						{
							widget.hideInUI(true);
							menuPanel.hide();
						}
					});
				}
			}
			
			// Required flag. Content class only. Disabled in partner override.
			if(widget.type == Type.ContentClass) 
			{
				if(widget.getField().isRequired()) 
				{
					menu.addItem("Make Optional", new Command() 
					{
						@Override
						public void execute() 
						{
							widget.setRequired(false);
							menuPanel.hide();
						}
					});
				}
				else 
				{
					menu.addItem("Make Required", new Command() 
					{
						@Override
						public void execute() 
						{
							widget.setRequired(true);
							menuPanel.hide();
						}
					});
				}
			}
			
			// Properties
			if(widget.type == Type.ContentClass) 
			{
				menu.addItem("Properties...", new Command() 
				{
					@Override
					public void execute() 
					{
						menuPanel.hide();
						FieldPropertiesDialog dialog = new FieldPropertiesDialog(widget);
						dialog.center();
					}
				});
			}
			
			if(widget.type == Type.ContentClassPartner && widget.field.getOptions() != null) 
			{
				menu.addItem("Edit Options...", new Command() 
				{
					@Override
					public void execute() 
					{
						menuPanel.hide();
						PartnerFieldPropertiesDialog dialog = new PartnerFieldPropertiesDialog(widget);
						dialog.center();
					}
				});
			}
			
			menuPanel.add(menu);
			menuPanel.show();
		}
	}

}
