package com.tandbergtv.cms.portal.content.client.contentclass.widget;

import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.contentclass.model.ContentClassValidator;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClassField;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ComboBoxWidget;

/**
 * Handles changes to values of text box, text area and drop-down box of the following data types:
 * STRING, INTEGER, FLOAT, BOOLEAN, TIME.
 * NOTE: This handler is not used for DATE data type. 
 * DATE data type is handled by {@link ContentClassDateValueChangeHandler}
 * @author eyevkar
 */
public class ContentClassValueChangeHandler implements ValueChangeHandler<String>, ChangeHandler 
{
	private static final String ERROR_STYLE = "WidgetValidationError";
	private UIContentClassField field; 
	
	/**
	 * Constructor
	 * @param field
	 */
	public ContentClassValueChangeHandler(UIContentClassField field) 
	{
		this.field = field;
	}

	/**
	 * Handles changes in text box and text area
	 */
	@Override
	public void onValueChange(ValueChangeEvent<String> event) 
	{
		Widget widget = (Widget)event.getSource();
		String value = event.getValue();
		
		setValueAndValidate(widget, value);
	}

	/**
	 * Handles changes in dropdowns
	 */
	@Override
	public void onChange(ChangeEvent event) 
	{
		// Get reference to the widget that fired this event
		Object source = event.getSource();
		// If List Box fired this event, get selected value
		if(source instanceof ListBox) 
		{
			ListBox listBox = (ListBox)source;
			int idx = listBox.getSelectedIndex();
			if(idx >= 0) 
			{
				String value = listBox.getValue(idx);
				field.setValue(value);
			}
		}
		else if(source instanceof ComboBoxWidget)
		{
			ComboBoxWidget comboBox = (ComboBoxWidget)source;
			String value = comboBox.getText();
			field.setValue(value);
		}
	}

	private void setValueAndValidate(Widget widget, String value) 
	{
		// Trim value if it is not null
		if(value != null) 
		{
			value = value.trim();
		}

		// If value is blank, remove error indicators.
		if(value == null || value.length() < 1) 
		{
			field.setValue(null);
			widget.removeStyleName(ERROR_STYLE);
			return;
		}
		else 
		{
			// Set trimmed value as is
			field.setValue(value);
		}

		// Validate non-STRING and non-BOOLEAN types
		// NOTE: DATE type is handled by {@link ContentClassDateValueChangeHandler} 
		if(field.getDataType() == UIContentClassField.DataType.INTEGER) 
		{
			try 
			{
				Long.parseLong(value);
				widget.removeStyleName(ERROR_STYLE);
			}
			catch(Exception ex) 
			{
				widget.addStyleName(ERROR_STYLE);
			}
		}
		else if(field.getDataType() == UIContentClassField.DataType.FLOAT) 
		{
			try 
			{
				Float.parseFloat(value);
				widget.removeStyleName(ERROR_STYLE);
			}
			catch(Exception ex) 
			{
				widget.addStyleName(ERROR_STYLE);
			}
		}
		else if(field.getDataType() == UIContentClassField.DataType.TIME)
		{
			if(field.isTimeCodeType())
			{
				if(ContentClassValidator.isTimeCodeValid(value))
				{
					widget.removeStyleName(ERROR_STYLE);
				}
				else
				{
					widget.addStyleName(ERROR_STYLE);
				}
			}
			else
			{
				try 
				{
					String dispFormat = field.getDisplayFormat();
					String formatString = (dispFormat == null || dispFormat.isEmpty()) ? "HH:mm:ss" : dispFormat;
					DateTimeFormat displayFormat = DateTimeFormat.getFormat(formatString);
					displayFormat.parseStrict(value);
					widget.removeStyleName(ERROR_STYLE);
				}
				catch(Exception ex) 
				{
					widget.addStyleName(ERROR_STYLE);
				}
			}
		}
	}
}
