package com.tandbergtv.cms.portal.content.client.contentclass.widget;

import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.TextArea;
import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.contentclass.dialog.CCDateWidget;
import com.tandbergtv.cms.portal.content.client.contentclass.model.KeyValuePair;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClassField;
import com.tandbergtv.cms.portal.content.client.contentclass.model.UIContentClassField.FieldType;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIDateFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.view.datatype.factory.DataTypeWidgetMessages;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ComboBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.DatePickerWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedComboBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.DateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IDateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.StringConverter;


/**
 * Creates different widgets on Content Class screens
 * @author eyevkar
 */
public class ContentClassWidgetFactory {
	private static DataTypeWidgetMessages messages = GWT.create(DataTypeWidgetMessages.class);
	private static final String ERROR_STYLE = "WidgetValidationError";
	private static final String READ_ONLY_STYLE = "WidgetReadOnly";
	
	/**
	 * Creates simple widgets such as text box, text area or dropdown box.
	 * @param asset
	 * @param field
	 * @return new widget
	 */
	public static Widget createSimpleFieldWidget(UIContentClassField field, boolean isEditable) {
		// Create Date widget
		if(field.getDataType() == UIContentClassField.DataType.DATE) {
			return createDateWidget(field, isEditable);
		}
		// Create Time widget
		else if(field.getDataType() == UIContentClassField.DataType.TIME) {
			return createTimeWidget(field, isEditable);
		}
		// Create Boolean widget
		else if(field.getDataType() == UIContentClassField.DataType.BOOLEAN) {
			return createBooleanWidget(field, isEditable);
		}
		else if(field.getDataType() == UIContentClassField.DataType.INTEGER) {
			return createIntegerWidget(field, isEditable);
		}
		else if(field.getDataType() == UIContentClassField.DataType.FLOAT) {
			return createFloatWidget(field, isEditable);
		}
		else if(field.getDataType() == UIContentClassField.DataType.STRING) {
			return createStringWidget(field, isEditable);
		}
		else {
			return new Label();
		}
	}

	private static Widget createDateWidget(UIContentClassField field, boolean isEditable) {
		String displayFormatPattern = field.getDisplayFormat();
		if (displayFormatPattern == null || displayFormatPattern.trim().length() == 0) {
			displayFormatPattern = messages.dateWidgetFormat();
		}

		// Create date formats
		DateTimeFormat displayFormat = DateTimeFormat.getFormat(displayFormatPattern);
		DateTimeFormat inputFormat = DateTimeFormat.getFormat(UIDateFieldDefinition.DATE_PATTERN);

		// Create date box widget
		IDateConverter dateConverter = new DateConverter(inputFormat.getPattern(), displayFormat.getPattern());
		CCDateWidget dateBox = new CCDateWidget(new DatePickerWidget(), null, dateConverter);
		dateBox.setEnabled(isEditable);
		
		if(!isEditable) {
			dateBox.addStyleName(READ_ONLY_STYLE);
		}
		
		if(isEditable) {
			ContentClassDateValueChangeHandler handler = new ContentClassDateValueChangeHandler(field);
			dateBox.addValueChangeHandler(handler);
		}
		
		// Set value
		if(field.isValid()) {
			dateBox.setValue(field.getValue());
		}
		else {
			dateBox.setTextValue(field.getValue());
		}
		
		dateBox.setTitle(messages.dateWidgetToolTip(displayFormatPattern));
		return dateBox;
	}
	
	private static Widget createTimeWidget(UIContentClassField field, boolean isEditable) 
	{
		String displayFormatPattern = field.getDisplayFormat();
		if(displayFormatPattern == null || displayFormatPattern.trim().length() == 0) 
		{
			displayFormatPattern = messages.timeWidgetFormat();
		}
		
		// Single-line text field
		TextBox textBox = new TextBox();
		textBox.setEnabled(isEditable);

		if(!isEditable) 
		{
			textBox.addStyleName(READ_ONLY_STYLE);
		}
		
		if(field.isValid()) 
		{
			if(!field.isEmpty())
			{
				if(field.isTimeCodeType())
				{
					textBox.setValue(field.getValue());
				}
				else
				{
					textBox.setValue(field.getValue());
				}
			}
		}
		else 
		{
			textBox.setValue(field.getValue());
			textBox.addStyleName(ERROR_STYLE);
		}
		
		// Add change handler to update model (field)
		if(isEditable) 
		{
			ContentClassValueChangeHandler changeHandler = new ContentClassValueChangeHandler(field);
			textBox.addValueChangeHandler(changeHandler);
		}
		
		if(field.isTimeCodeType())
		{
			textBox.setTitle(messages.timeCodeWidgetToolTip());
		}
		else
		{
			textBox.setTitle(messages.timeWidgetToolTip(displayFormatPattern));
		}

		return textBox; 
	}
	
	private static Widget createFloatWidget(UIContentClassField field, boolean isEditable) {
		Widget widget = createStringWidget(field, isEditable);
		widget.setTitle(messages.floatWidgetToolTip());

		if(!field.isValid()) {
			widget.addStyleName(ERROR_STYLE);
		}

		return widget;
	}
	
	private static Widget createIntegerWidget(UIContentClassField field, boolean isEditable) {
		Widget widget = createStringWidget(field, isEditable);
		widget.setTitle(messages.integerWidgetToolTip());
		
		if(!field.isValid()) {
			widget.addStyleName(ERROR_STYLE);
		}
		
		return widget;
	}
	
	private static Widget createStringWidget(UIContentClassField field, boolean isEditable) 
	{
		ContentClassValueChangeHandler changeHandler = null;
		if(isEditable) 
		{
			changeHandler = new ContentClassValueChangeHandler(field);
		}

		FieldType fieldType = field.getFieldType();
		
		// Drop-down box
		if(fieldType == FieldType.DROP_DOWN)
		{
			ListBox listBox = new ListBox();
			listBox.setVisibleItemCount(1);
			
			List<KeyValuePair> options = field.getOptions();
			ContentClassWidgetFactory.reloadData(listBox, field);
			
			// Add change handler to update model (field)
			if(isEditable) 
			{
				listBox.addChangeHandler(changeHandler);
			}
			else
			{
				listBox.addStyleName(READ_ONLY_STYLE);
			}
			
			listBox.setTitle(messages.stringWidgetToolTip());
			listBox.setEnabled(isEditable);
			
			return listBox;
		}
		// Combo box
		else if(fieldType == FieldType.COMBO_BOX)
		{
			ComboBoxWidget<String> comboBox = new ComboBoxWidget<String>();
			ContentClassWidgetFactory.reloadData(comboBox, field);
			
			comboBox.setTitle(messages.stringWidgetToolTip());
			comboBox.setEnabled(isEditable);

			// Add change handler to update model (field)
			if(isEditable) 
			{
				comboBox.addChangeHandler(changeHandler);
			}
			else
			{
				comboBox.addStyleName(READ_ONLY_STYLE);
			}
			
			return comboBox;
		}
		// Text Area
		else if(field.getFieldType() == FieldType.TEXT_AREA) 
		{
			TextArea textArea = new TextArea();
			textArea.setVisibleLines(4);
			textArea.setText(field.getValue());
			
			// Add change handler to update model (field)
			if(isEditable) 
			{
				textArea.addValueChangeHandler(changeHandler);
			}
			
			textArea.setTitle(messages.stringWidgetToolTip());
			textArea.setEnabled(isEditable);
			
			if(!isEditable) 
			{
				textArea.addStyleName(READ_ONLY_STYLE);
			}
			
			return textArea;
		}
		// Single-line text field
		else
		{
			TextBox textBox = new TextBox();
			textBox.setText(field.getValue());
			
			// Add change handler to update model (field)
			if(isEditable) 
			{
				textBox.addValueChangeHandler(changeHandler);
			}
			
			textBox.setTitle(messages.stringWidgetToolTip());
			textBox.setEnabled(isEditable);
			
			if(!isEditable) 
			{
				textBox.addStyleName(READ_ONLY_STYLE);
			}
			
			return textBox;
		}
	}

	
	private static Widget createBooleanWidget(UIContentClassField field, boolean isEditable) 
	{
		ListBox listBox = new ListBox();
		listBox.setVisibleItemCount(1);
		
		if(field.hasOptions()) 
		{
			ContentClassWidgetFactory.reloadData(listBox, field);
		}
		else 
		{
			listBox.addItem("", "");
			listBox.addItem(messages.booleanWidgetTrueValue(), "true");
			listBox.addItem(messages.booleanWidgetFalseValue(), "false");
			if("true".equals(field.getValue())) 
			{
				listBox.setSelectedIndex(1);
			}
			else if("false".equals(field.getValue())) 
			{
				listBox.setSelectedIndex(2);
			}
		}
		
		// Add change handler to update model (field)
		if(isEditable) 
		{
			ContentClassValueChangeHandler changeHandler = new ContentClassValueChangeHandler(field);
			listBox.addChangeHandler(changeHandler);
		}		
		
		listBox.setTitle(messages.booleanWidgetToolTip());
		
		listBox.setEnabled(isEditable);
		
		if(!isEditable) {
			listBox.addStyleName(READ_ONLY_STYLE);
		}
		
		return listBox;
	}
	
	/**
	 * Clears old values and adds new from UIContentClassField
	 * @param listBox list box to reload
	 * @param field data source
	 */
	public static void reloadData(ListBox listBox, UIContentClassField field) 
	{
		if(listBox == null || field == null) return;
		
		List<KeyValuePair> options = field.getOptions();
		Set<String> excludedKeys = field.getFilteredOptionKeys();
		String selectedValue = field.getValue(); 

		// Clear old values
		listBox.clear();
		
		// Insert empty value
		listBox.addItem("", "");

		// Create an item for each option
		if(options != null)
		{
			for(KeyValuePair item: options) {
				// If the key of this item is in excluded keys set, don't display it.
				if(excludedKeys != null && excludedKeys.contains(item.getValue())) 
				{
					continue;
				}
				
				listBox.addItem(item.getKey(), item.getValue());
				// If item's key = field's value, select the item
				if(item.getValue().equals(selectedValue)) 
				{
					// Select last item
					listBox.setSelectedIndex(listBox.getItemCount()-1);
				}
			}
		}
	}


	public static void reloadData(ComboBoxWidget<String> comboBox, UIContentClassField field) 
	{
		if(comboBox == null || field == null) return;
		
		List<KeyValuePair> options = field.getOptions();
		Set<String> excludedKeys = field.getFilteredOptionKeys();
		String selectedValue = field.getValue(); 

		// Clear old values
		comboBox.clear();
		
		// Create an item for each option
		if(options != null)
		{
			for(KeyValuePair item: options) {
				// If the key of this item is in excluded keys set, don't display it.
				if(excludedKeys != null && excludedKeys.contains(item.getValue())) 
				{
					continue;
				}
				
				comboBox.addItem(item.getKey(), item.getValue());
				// If item's key = field's value, select the item
				if(item.getValue().equals(selectedValue)) 
				{
					// Select last item
					comboBox.setSelectedIndex(comboBox.getItemCount()-1);
				}
			}
		}
		
		comboBox.setText(field.getValue());
	}

}
