package com.tandbergtv.cms.portal.content.client.lookupkey.view;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.cms.portal.content.client.i18n.ContentConstants;
import com.tandbergtv.cms.portal.content.client.lookupkey.model.UiLookupKey;
import com.tandbergtv.cms.portal.content.client.rpc.lookupkey.ILookupKeyService;
import com.tandbergtv.cms.portal.content.client.rpc.lookupkey.ILookupKeyServiceAsync;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on lookup keys.
 * 
 * @author Daniel Paranhos
 * 
 */
public class LookupKeysDataProvider implements DataProvider<Long, UiLookupKeyRecord> {

    // columns for this table
    private final List<Column<?, UiLookupKeyRecord>> columns;
    private final ColumnBase<String, UiLookupKeyRecord> nameColumn;
    private final ColumnBase<String, UiLookupKeyRecord> descriptionColumn;
    private final ColumnBase<String, UiLookupKeyRecord> assetTypesColumn;
    
    private int COLUMN_SIZE_LIMIT = 100;

    // RPC service to get data from the backend
    ILookupKeyServiceAsync lookupKeyService = GWT.create( ILookupKeyService.class );

    private ContentConstants constants = (ContentConstants) GWT.create( ContentConstants.class );

    private SortFeatureImpl<Long, UiLookupKeyRecord> parentSortFeature;

    // column names
    private static String NAME_COLUMN = "name";
    private static String DESCRIPTION_COLUMN = "description";
    private static String ASSET_TYPES_COLUMN = "assetTypes";

    // lists panel
    private IBusyNotifier busyNotifier;

    public LookupKeysDataProvider() {
        // create the columns - name, description and assetTypes
        columns = new ArrayList<Column<?, UiLookupKeyRecord>>();

        nameColumn = new ColumnBase<String, UiLookupKeyRecord>( NAME_COLUMN, constants.lookupKeyNameColumn() ) {
            public View<String> getView( UiLookupKeyRecord record ) {
                return new LabelStringView( record.getLookupKey().getName() );
            }
        };
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );

        descriptionColumn = new ColumnBase<String, UiLookupKeyRecord>( DESCRIPTION_COLUMN,
                constants.lookupKeyDescriptionColumn() ) {
            public View<String> getView( UiLookupKeyRecord record ) {
                return new LimitedLabelStringView( record.getLookupKey().getDescription(), COLUMN_SIZE_LIMIT );
            }
        };
        descriptionColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( descriptionColumn );

        assetTypesColumn = new ColumnBase<String, UiLookupKeyRecord>( ASSET_TYPES_COLUMN,
                constants.lookupKeyAssetTypesColumn() ) {
            public View<String> getView( UiLookupKeyRecord record ) {
                return new LimitedLabelStringView( record.getLookupKey().getAssetTypesAsString(), COLUMN_SIZE_LIMIT );
            }
        };
        assetTypesColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( assetTypesColumn );
    }

    public ColumnBase<String, UiLookupKeyRecord> getDefaultSortColumn() {
        return nameColumn;
    }

    public List<Column<?, UiLookupKeyRecord>> getSortColumns() {
        List<Column<?, UiLookupKeyRecord>> sortColumns = new ArrayList<Column<?, UiLookupKeyRecord>>();
        sortColumns.add( nameColumn );
        sortColumns.add( descriptionColumn );

        return sortColumns;
    }

    @Override
    public List<Column<?, UiLookupKeyRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiLookupKeyRecord> callback ) {
        notifyBusyStart();
        lookupKeyService.get( key, new NeptuneAsyncCallback<UiLookupKey>() {

            @Override
            public void onNeptuneFailure( Throwable caught ) {
                notifyBusyFinish();
                callback.onFailure( caught );
            }

            @Override
            public void onNeptuneSuccess( UiLookupKey result ) {
                notifyBusyFinish();
                callback.onSuccess( new UiLookupKeyRecord( result ) );
            }
        } );
    }

    @Override
    public void getRecords( final AsyncCallback<List<UiLookupKeyRecord>> callback ) {
        // get all lookup keys in the system and prepare ui objects
        notifyBusyStart();
        lookupKeyService.getLookupKeys( parentSortFeature.getSortColumnName(),
                parentSortFeature.isAscending(), new NeptuneAsyncCallback<List<UiLookupKey>>() {
                    public void onNeptuneFailure( Throwable caught ) {
                        notifyBusyFinish();
                        callback.onFailure( caught );
                    }

                    public void onNeptuneSuccess( List<UiLookupKey> result ) {
                        notifyBusyFinish();
                        callback.onSuccess( prepareUIObjects( result ) );
                    }

                    private List<UiLookupKeyRecord> prepareUIObjects( List<UiLookupKey> result ) {
                        List<UiLookupKeyRecord> records = new ArrayList<UiLookupKeyRecord>();
                        for ( UiLookupKey list : result ) {
                            records.add( new UiLookupKeyRecord( list ) );
                        }
                        return records;
                    }
                } );
    }

    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return false;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return false;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiLookupKeyRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiLookupKeyRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
            busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
            busyNotifier.doneBeingBusy();
        }
    }

}
