package com.tandbergtv.cms.portal.content.client.portlet.activity;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Timer;
import com.tandbergtv.cms.portal.content.client.portlet.PortletConstants;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.Portlet;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentAttachHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.DocumentDetachHandler;

/**
 * Activity portlet lists most recent activity for any title (master or site)
 * that is currently being processed by a work order in the system.
 * The activity portlet only shows current activity.
 * 
 * @author eyevkar
 */
public class ActivityPortlet extends Portlet {
	ActivityPortletConfig config;
	
	private NeptuneSecurity security;
	private IActivityPortletServiceAsync service;
	private PortletConstants constants;
	private Timer refreshTimer;
	
	private FlexTableContainer table;
	
	/**
	 * Constructor
	 * @param security
	 * @param refreshTimeMillis
	 */
	public ActivityPortlet(final NeptuneSecurity security, ActivityPortletConfig config) {
	        super("ACTIVITY");
		this.security = security;
		this.config = config;
		
		service = GWT.create(IActivityPortletService.class);
		constants = GWT.create(PortletConstants.class);

		setTitle(constants.actTitle());
		setDisplayName(constants.actTitle());
		
		table = new FlexTableContainer();
		table.setCellSpacing(0);
		table.setStyleName("portletTable");
		add(table);
		
		this.refreshTimer = new Timer() {
			@Override
			public void run() {
				reload();
			}
		};

		EventHandler handler = new EventHandler();
		addDocumentAttachHandler(handler);
		addDocumentDetachHandler(handler);
	}

	private void reload() {
		service.getData(config.getRecordCount(), new NeptuneAsyncCallback<ActivityPortletData>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				// Clear table
				table.removeAllRows();
			}

			@Override
			public void onNeptuneSuccess(ActivityPortletData result) {
				// Clear table
				table.removeAllRows();
				
				// Call RPC to get table records
				List<ActivityPortletRecord> records = result.getRecords();
				
				// Display a table if there are any records
				if(records != null && !records.isEmpty()) {
					// Create table header
					createTableHeader(table);
					// Iterate over records and populate GWT table
					for(int row = 0; row < records.size(); row++) {
						ActivityPortletRecord rec = records.get(row);
						// Table header has row=0. Use row+1.
						createTableRecord(table, row+1, rec);
					}
				}
				
				update();
			}
		});
	}
	
	private void createTableRecord(FlexTableContainer table, int row, ActivityPortletRecord rec) {
		int colIndex = 0;
		
		// Status
		if(security.isUserInRole("WorkOrderManagement_View")) {
			table.setWidget(row, colIndex, new ImageHyperlink(getImage(rec.getStatusName(), rec.getStatusToolTip()), 
					"Workflow.Work Orders.Search.Id=" + rec.getProcessId()));
		}
		else {
			table.setWidget(row, colIndex, getImage(rec.getStatusName(), rec.getStatusToolTip()));
		}
		table.getCellFormatter().setStyleName(row, colIndex, "portlet-table-icons");

		++colIndex;
		
		// Title
		table.setWidget(row, colIndex, new HyperlinkWidget(rec.getTitle(), "Content.Search.Id=" + rec.getTitleId() + "&Tab=Activity"));
		table.getCellFormatter().setStyleName(row, colIndex, "portlet-table-text");
		
		++colIndex;
		
		// Site name
		if(rec.getSiteName() != null) {
			if(security.isUserInRole("Sites_View")) {
				table.setWidget(row, colIndex, new HyperlinkWidget(rec.getSiteName(), "Sites.View?id=" + rec.getSiteId()));
			}
			else {
				table.setWidget(row, colIndex, new LabelWidget(rec.getSiteName()));
			}
			table.getCellFormatter().setStyleName(row, colIndex, "portlet-table-text");
		}
		
		++colIndex;
		
		// Process name
		table.setWidget(row, colIndex, new LabelWidget(rec.getProcessName()));
		table.getCellFormatter().setStyleName(row, colIndex, "portlet-table-text");
		
		++colIndex;
		
		// Start Time
		String startDate = (rec.getStartTime() == null) ? "N/A" : rec.getStartTime();
		table.setWidget(row, colIndex, new LabelWidget(startDate));
		table.getCellFormatter().setStyleName(row, colIndex, "portlet-table-text");
		
	}
	
	private void createTableHeader(FlexTableContainer table) {
		int row = 0;
		int col = 0;

		// Status
		table.setWidget(row, col, new LabelWidget("Status"));
		table.getCellFormatter().setStyleName(row, col, "portlet-tableHeader");
		table.getCellFormatter().addStyleName(row, col, "activityportlet-tablecolumn"+col+"-width");
		++col;
		
		// Title ID
		table.setWidget(row, col, new LabelWidget("Title ID"));
		table.getCellFormatter().setStyleName(row, col, "portlet-tableHeader");
		table.getCellFormatter().addStyleName(row, col, "activityportlet-tablecolumn"+col+"-width");
		++col;
		
		// Site name
		table.setWidget(row, col, new LabelWidget("Site Name"));
		table.getCellFormatter().setStyleName(row, col, "portlet-tableHeader");
		table.getCellFormatter().addStyleName(row, col, "activityportlet-tablecolumn"+col+"-width");
		++col;
		
		// Process name
		table.setWidget(row, col, new LabelWidget("Process Name"));
		table.getCellFormatter().setStyleName(row, col, "portlet-tableHeader");
		table.getCellFormatter().addStyleName(row, col, "activityportlet-tablecolumn"+col+"-width");
		++col;
		
		// Start Time
		table.setWidget(row, col, new LabelWidget("Start Time"));
		table.getCellFormatter().setStyleName(row, col, "portlet-tableHeader");
		table.getCellFormatter().addStyleName(row, col, "activityportlet-tablecolumn"+col+"-width");
	}
	
	private ImageWidget getImage(String stateName, String toolTip) {
		ImageWidget image = new ImageWidget("images/icon_rect_" + stateName + ".gif");
		if(toolTip == null)
			image.setTitle(getLocalizedStateName(stateName));
		else
			image.setTitle(toolTip);
		return image;
	}
	
	private String getLocalizedStateName(String stateName) {
		return constants.getString("status" + stateName);
	}

	/*
	 * Event Handling
	 */
	private final class EventHandler implements DocumentAttachHandler, DocumentDetachHandler {
		@Override
		public void onAttach(DocumentAttachEvent event) {
			refreshTimer.scheduleRepeating(config.getRefreshRateSecs() * 1000);
			reload();
		}

		@Override
		public void onDetach(DocumentDetachEvent event) {
			refreshTimer.cancel();
		}
	}
}
