/**
 * File Name	:	AdvancedSearchPanel.java
 * Created By	:	Sanjay Khattar (sanjay.khattar@ericsson.com)
 * Date Created	:	Sep 13, 2012 4:48:31 PM
 * Purpose		:	
 *
 * (c) Ericsson Television Inc. 
 */
package com.tandbergtv.cms.portal.content.client.tab.search;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.tandbergtv.cms.portal.content.client.i18n.ContentConstants;
import com.tandbergtv.cms.portal.content.client.rpc.title.UiTitleCriteriaService;
import com.tandbergtv.cms.portal.content.client.rpc.title.UiTitleCriteriaServiceAsync;
import com.tandbergtv.cms.portal.ui.title.client.criteria.CriteriaListingPanel;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.cms.portal.ui.title.client.model.search.SearchOperator;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UIFieldType;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilter;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilterListNode;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleSearch;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleSearchBase;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiColumn;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiSortInfo;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * @author esakhat (sanjay.khattar@ericsson.com)
 *
 */
public class AdvancedSearchPanel extends SearchPanelBase {

	private static final String STYLE_ADVANCED_SEARCH_PANEL = "advanced-search-panel";
	private static final String SPACING_STYLE = "search-criteria-spacing";
	private static final String SAVE_SEARCH_PANEL_STYLE = "save-search-panel";

	private CriteriaListingPanel criteriaListingPanel;
	
	private CheckBoxWidget shareSearch;
	private TextBoxWidget searchName;
	private ButtonWidget saveSearchButton;
	private ButtonWidget copySearchButton;
	private LabelWidget copyIndicator;

	private UITitleSearchBase currentEditSearch;

	private BusyIndicator busyIndicator = new BusyIndicator();

	private ContentConstants contentConstants = (ContentConstants) GWT.create(ContentConstants.class);
	private UiTitleCriteriaServiceAsync criteriaService = GWT.create(UiTitleCriteriaService.class);

	/**
	 * @param neptuneSecurity
	 * @param criteriaMode
	 */
	public AdvancedSearchPanel(SearchColumnPanel searchColumnPanel, NeptuneSecurity neptuneSecurity,
			UiCriteriaMode criteriaMode) {
		super(searchColumnPanel, neptuneSecurity, criteriaMode);

		VerticalContainer mainPanel = new VerticalContainer();
		
		ComplexHeaderPanel complexHeaderPanel = buildComplexHeaderPanel();
		HorizontalContainer saveSearchPanel = new HorizontalContainer();
		criteriaListingPanel = new CriteriaListingPanel(criteriaMode, null, new SearchOperator());
		
		copyIndicator = new LabelWidget("(" + contentConstants.copyIndicatorText() + ")");
		copyIndicator.setVisible(false);

		HorizontalContainer innerHorizontalContainer = new HorizontalContainer();
		
		VerticalContainer innerVerticalContainer = new VerticalContainer();
		
		mainPanel.add(complexHeaderPanel);
		mainPanel.add(innerHorizontalContainer);
		innerHorizontalContainer.add(innerVerticalContainer);
		innerHorizontalContainer.add(searchColumnPanel);
		
		innerVerticalContainer.add(criteriaListingPanel);
		innerVerticalContainer.add(saveSearchPanel);
		innerVerticalContainer.add(copyIndicator);

		mainPanel.addStyleName(STYLE_ADVANCED_SEARCH_PANEL);
		
		mainPanel.setCellHorizontalAlignment(saveSearchPanel, HorizontalContainer.ALIGN_LEFT);

		LabelWidget searchNameLabel = new LabelWidget(contentConstants.searchName());
		saveSearchPanel.add(searchNameLabel);
		saveSearchPanel.setCellVerticalAlignment(searchNameLabel, HasVerticalAlignment.ALIGN_MIDDLE);

		searchName = new TextBoxWidget();
		searchName.setVisibleLength(50);
		searchName.setMaxLength(100);
		searchName.addStyleName(SPACING_STYLE);
		saveSearchPanel.add(searchName);

		shareSearch = new CheckBoxWidget(contentConstants.shareChkBox());
		shareSearch.addStyleName(SPACING_STYLE);
		saveSearchPanel.add(shareSearch);
		saveSearchPanel.setCellVerticalAlignment(shareSearch, HasVerticalAlignment.ALIGN_MIDDLE);

		saveSearchPanel.addStyleName(SAVE_SEARCH_PANEL_STYLE);

		initWidget(mainPanel);

		searchColumnPanel.init(new NeptuneAsyncCallback<Void>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				showErrorMessage(contentConstants.errorLoadingColumns());
			}

			@Override
			public void onNeptuneSuccess(Void result) {
			}
		});
	}
	
	public ButtonWidget buildSearchButton() {
		ButtonWidget searchButton = new ButtonWidget(contentConstants
				.searchButtonName());
		searchButton.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				if (!validate()) {
					return;
				}

				resetMessageListener();

				TitleSearchEvent searchEvent = new TitleSearchEvent(
						buildFilter(), searchColumnPanel.getSelectedColumns());
				fireEvent(searchEvent);
			}

		});
		searchButton.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		
		return searchButton;
	}
	
	
	/*
	 * builds the save button for the advanced section of the search criteria
	 * panel. This button is used to save a search.
	 */
	public ButtonWidget buildSaveSearchButton() {
		ButtonWidget save = new ButtonWidget(contentConstants.saveButton());
		save.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		save.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				if (!validate()) {
					return;
				}

				UITitleFilter filter = buildFilter();
				String searchNameStr = searchName.getValue();

				if (searchNameStr.isEmpty()) {
					showErrorMessage(contentConstants
							.searchNameReq());
					return;
				}

				if (currentEditSearch == null) {
					currentEditSearch = new UITitleSearchBase(-1, "", neptuneSecurity
							.getUsername());
				}
				int id = currentEditSearch.getTitleSearchId();

				// always save selections even if they are the same as user prefs.
				// doing this to avoid callbacks when user changes his user prefs 
				UITitleSearch search;
				if (id <= 0) {
					search = new UITitleSearch(neptuneSecurity.getUsername(), searchNameStr,
							filter, shareSearch.getValue(),
							getSelectedSearchResultsColumns());
				}
				else {
					search = new UITitleSearch(id, neptuneSecurity.getUsername(), searchNameStr,
							filter, shareSearch.getValue(),
							getSelectedSearchResultsColumns());
				}

				busyIndicator.center();
				criteriaService.saveTitleSearch(search,
						new NeptuneAsyncCallback<Integer>() {

							@Override
							public void onNeptuneFailure(Throwable caught) {
								if (caught instanceof TitleSearchNameUniqueException) {
									showErrorMessage(
											contentConstants.errSearchNameUnique());
								} else {
									showErrorMessage(
											contentConstants.errWhileSaving());
								}
								busyIndicator.hide();
							}

							@Override
							public void onNeptuneSuccess(Integer titleSearchId) {
								clearCriteria();
								searchTabPanel.showListView();
								showInfoMessage(contentConstants
										.succSearchSave());
								if (busyIndicator.isShowing()) {
									busyIndicator.hide();
								}
							}

						});
			}

		});
		return save;
	}

	/*
	 * builds the copy button for the advanced section of the search criteria
	 * panel. This button is used to copy a search.
	 */
	public ButtonWidget buildCopySearchButton() {
		final ButtonWidget copy = new ButtonWidget(contentConstants.copyButton());
		copy.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		copy.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				if (!validate()) {
					return;
				}
				
				int titleSearchId = -1;
				
				if (currentEditSearch != null) {
					titleSearchId = currentEditSearch.getTitleSearchId();
				}
				
				if (titleSearchId == -1) {
					// if it is -1 set the current edit search to an empty.
					currentEditSearch = new UITitleSearchBase(-1, "", neptuneSecurity
							.getUsername());

					// manage buttons for new rule
					managePersistanceButton(-1, neptuneSecurity.getUsername());
					if (busyIndicator.isShowing()) {
						busyIndicator.hide();
					}
				}
				else {
					UITitleFilter filter = buildFilter();
					String searchNameStr = searchName.getValue();
					currentEditSearch = new UITitleSearch(neptuneSecurity.getUsername(), searchNameStr,
							filter, shareSearch.getValue(),
							getSelectedSearchResultsColumns());
					managePersistanceButton(0, neptuneSecurity.getUsername());
				}

				copy.setVisible(false);
				copyIndicator.setVisible(true);
				
				Scheduler.get().scheduleDeferred(new ScheduledCommand() {

					@Override
					public void execute() {
						searchName.selectAll();
						searchName.setFocus(true);
					}
					
				});
			}

		});
		return copy;
	}

	public boolean validate() {
		if (criteriaListingPanel == null)
			return false;
		else 
			try {
				searchColumnPanel.validateSelections();
				return this.criteriaListingPanel.validate();
			} catch(SelectionValidationException e) {
				searchColumnPanel.showErrorMessage(e.getMessage());
				return false;
			}		
	}

	/*
	 * Returns a UIFilter representing the criteria in the advanced panel on
	 * this search criteria panel.
	 */
	private UITitleFilter buildFilter() {
		// if there is no criteria build a get everything
		if (criteriaListingPanel.isEmpty()) {
			return getAllTitlesFilter();
		}

		UITitleFilter filter = new UITitleFilter();
		filter.setCriteriaMode(criteriaMode);
		UITitleFilterListNode rootNode = criteriaListingPanel.getCriteria();

		filter.setRootNode(rootNode);

		setFilterSortInfo(filter);
		filter.setIncludeCount(true);
		return filter;
	}

	private UITitleFilter getAllTitlesFilter() {
		UITitleFilter filter = new UITitleFilter();
		setFilterSortInfo(filter);
		filter.setCriteriaMode(criteriaMode);
		filter.setRootNode(null);
		filter.setIncludeCount(true);
		return filter;
	}

	private void setFilterSortInfo(UITitleFilter filter) {
		if(getSelectedSortColumn().getFieldType().equals(UIFieldType.ASSET)) {
			filter.getSortInfo().setSortFieldType(UIFieldType.ASSET);
			filter.getSortInfo().setSortColumnSection(getSelectedSortColumn().getSectionName());
		} else {
			filter.getSortInfo().setSortFieldType(UIFieldType.PMM_TITLE);
		}
		filter.getSortInfo().setSortColumnName(getSelectedSortColumn().getName());
		filter.getSortInfo().setSortOrder(getSelectedSortOrder());
	}
	
	public void loadCriteria(int titleSearchId) {
		// clear crap out
		shareSearch.setValue(false);
		searchName.setText("");
		if (copyIndicator != null) {
			copyIndicator.setVisible(false);
		}
		criteriaListingPanel.resetCriteria();
		if (titleSearchId == -1) {
			// if it is -1 set the current edit search to an empty.
			currentEditSearch = new UITitleSearchBase(-1, "", neptuneSecurity
					.getUsername());

			// manage buttons for new rule
			managePersistanceButton(-1, neptuneSecurity.getUsername());
			if (busyIndicator.isShowing()) {
				busyIndicator.hide();
			}
			return;
		}

		if (!busyIndicator.isShowing()) {
			busyIndicator.center();
		}

		criteriaService.getTitleSearch(titleSearchId,
				new NeptuneAsyncCallback<UITitleSearch>() {

					@Override
					public void onNeptuneFailure(Throwable caught) {
						if (caught instanceof SearchNotFoundException) {
							showErrorMessage(contentConstants
									.searchNotFound());
						} else {
							showErrorMessage(contentConstants
									.errorLoadCrit());
						}
						busyIndicator.hide();
					}

					@Override
					public void onNeptuneSuccess(UITitleSearch result) 
					{
						boolean isSharedSearch = result.isShare();
						String ownerUser = result.getUserName();
						String currentUser = neptuneSecurity.getUsername();
						
						if((ownerUser.equals(currentUser)) || isSharedSearch) 
						{
							searchName.setText(result.getName());
							shareSearch.setValue(isSharedSearch);
							criteriaListingPanel.setFilter(result.getTitleFilter().getRootNode());
							UiColumn sortCol = buildUiColumn(result.getTitleFilter().getSortInfo());
							searchColumnPanel.setSelections(result.getDisplayColumns(), sortCol, 
									result.getTitleFilter().getSortInfo().getSortOrder());
							managePersistanceButton(result.getTitleSearchId(), ownerUser);
							currentEditSearch = result;
						}
						else {
							showErrorMessage(contentConstants.notAuthorized());
						}
						busyIndicator.hide();
					}
				});
	}
	
	public void clearCriteria() {
		loadCriteria(-1);
	}

	private UiColumn buildUiColumn(UiSortInfo sortInfo) {
		UiColumn uic = new UiColumn();
		uic.setName(sortInfo.getSortColumnName());
		uic.setSectionName(sortInfo.getSortColumnSection());
		return uic;
	}

	/*
	 * Sets the visibilty of the save copy and delete buttons in the advanced
	 * view.
	 */
	private void managePersistanceButton(int titleSearchId, String searchOwner) {
		boolean ruleBelongsToMe = neptuneSecurity.getUsername().equals(
				searchOwner);
		boolean isNewRule = titleSearchId == -1;

		if (saveSearchButton != null) {
			saveSearchButton.setVisible(ruleBelongsToMe);
		}
		if (copySearchButton != null) {
			copySearchButton.setVisible(!isNewRule);
		}
	}
	
	private ComplexHeaderPanel buildComplexHeaderPanel() {
		ComplexHeaderPanel complexHeaderPanel = new ComplexHeaderPanel();
		complexHeaderPanel.addHeaderDescription(contentConstants.advancedSearchHeaderDescription());
		complexHeaderPanel.addHeaderCaution(contentConstants.advancedSearchCaution());
		complexHeaderPanel.addHeaderTip(contentConstants.advancedSearchTip());
		
		return complexHeaderPanel;
	}

	/**
	 * @return the saveSearchButton
	 */
	public ButtonWidget getSaveSearchButton() {
		return saveSearchButton;
	}

	/**
	 * @param saveSearchButton the saveSearchButton to set
	 */
	public void setSaveSearchButton(ButtonWidget saveSearchButton) {
		this.saveSearchButton = saveSearchButton;
	}

	/**
	 * @return the copySearchButton
	 */
	public ButtonWidget getCopySearchButton() {
		return copySearchButton;
	}

	/**
	 * @param copySearchButton the copySearchButton to set
	 */
	public void setCopySearchButton(ButtonWidget copySearchButton) {
		this.copySearchButton = copySearchButton;
	}

}
