/**
 * File Name	:	SavedSearchListPanel.java
 * Created By	:	Sanjay Khattar (sanjay.khattar@ericsson.com)
 * Date Created	:	Sep 19, 2012 3:32:35 PM
 * Purpose		:	
 *
 * (c) Ericsson Television Inc. 
 */
package com.tandbergtv.cms.portal.content.client.tab.search;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.FlexTable;
import com.tandbergtv.cms.portal.content.client.i18n.ContentConstants;
import com.tandbergtv.cms.portal.content.client.i18n.ContentMessages;
import com.tandbergtv.cms.portal.content.client.rpc.title.UiTitleCriteriaService;
import com.tandbergtv.cms.portal.content.client.rpc.title.UiTitleCriteriaServiceAsync;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UIFieldType;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilter;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleSearch;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleSearchBase;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiColumn;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HTMLWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * @author esakhat (sanjay.khattar@ericsson.com)
 *
 */
public class SavedSearchListPanel extends SearchPanelBase {

	private static final String STYLE_NAME_SAVED_SEARCHES_TABLE = "saved-searches-table";
	private static final String STYLE_NAME_HEADER_ROW = "saved-searches-table-header";
	private static final String STYLE_NAME_DATA_ROW = "saved-searches-table-data";
	private static final String STYLE_NAME_CHECKBOX_CELL = "saved-searches-cell-checkbox";
	private static final String STYLE_NAME_SEARCH_ID_CELL = "saved-searches-cell-search-id";
	private static final String STYLE_NAME_SEARCH_NAME_CELL = "saved-searches-cell-search-name";
	private static final String STYLE_NAME_SHARED_CELL = "saved-searches-cell-shared";
	private static final String STYLE_NAME_SEARCH_OWNER_CELL = "saved-searches-cell-search-owner";
	private static final String STYLE_NAME_GO_BUTTON_CELL = "saved-searches-cell-button-go";
	private static final String STYLE_NAME_SAVED_SEARCH_HYPERLINK = "saved-search-hyperlink";

	private static final String ROLE_NAME_ADMIN = "Admin";

	private SimpleContainer mainContainer;
	private FlexTable flexTable;
	private List<CheckBoxWidget> checkBoxesList = new ArrayList<CheckBoxWidget>();
	private CheckBoxWidget allCB;
	
	private Set<Integer> selectedIds = new HashSet<Integer>();

	private BusyIndicator busyIndicator = new BusyIndicator();

	private ContentConstants contentConstants = (ContentConstants) GWT.create(ContentConstants.class);
	private ContentMessages messages = (ContentMessages) GWT.create(ContentMessages.class);
	
	private UiTitleCriteriaServiceAsync criteriaService = GWT.create(UiTitleCriteriaService.class);
	
	public SavedSearchListPanel(SearchColumnPanel searchColumnPanel,
			NeptuneSecurity neptuneSecurity, UiCriteriaMode criteriaMode) {
		super(searchColumnPanel, neptuneSecurity, criteriaMode);
		
		mainContainer = new SimpleContainer();
		
		VerticalContainer verticalContainer = new VerticalContainer();
		mainContainer.setWidget(verticalContainer);
		
		ComplexHeaderPanel complexHeaderPanel = buildComplexHeaderPanel();
		verticalContainer.add(complexHeaderPanel);

		flexTable = new FlexTable();
		verticalContainer.add(flexTable);
		
		initWidget(mainContainer);

		refreshTable();
	}
	
	public void refresh() {
		resetMessageListener();
		refreshTable();
	}

	public void refreshTable() {
		flexTable.removeAllRows();
		selectedIds.clear();
		checkBoxesList = new ArrayList<CheckBoxWidget>();
		if (!busyIndicator.isShowing()) {
			busyIndicator.center();
		}
		criteriaService.getTitleSearches(0, 0, neptuneSecurity.getUsername(), criteriaMode,
				new NeptuneAsyncCallback<List<UITitleSearchBase>>() {

					@Override
					public void onNeptuneFailure(Throwable caught) {
						showErrorMessage(contentConstants
								.errLoadingSavedSearches());
						busyIndicator.hide();
					}

					@Override
					public void onNeptuneSuccess(
							List<UITitleSearchBase> searches) {
						populateTable(searches);
						
						if (busyIndicator.isShowing()) {
							busyIndicator.hide();
						}
						
					}
				});
		
	}
	
	public void deleteSelected() {
		resetMessageListener();
		if (selectedIds.size() > 0) {
			String msg = messages.confirmDelSelectedSavedSearch();
			if (Window.confirm(msg)) {
				if (!busyIndicator.isShowing()) {
					busyIndicator.center();
				}
				criteriaService.deleteTitleSearches(selectedIds,
						new NeptuneAsyncCallback<Void>() {

							@Override
							public void onNeptuneFailure(Throwable caught) {
								showErrorMessage(contentConstants
										.errWhileDeling());
								busyIndicator.hide();
							}

							@Override
							public void onNeptuneSuccess(Void result) {
								refreshTable();
								showInfoMessage(contentConstants
										.successDeleteSearches());
								if (busyIndicator.isShowing()) {
									busyIndicator.hide();
								}
							}

						});
			}
		}
		else {
			showErrorMessage(contentConstants.noSearchSelected());
		}
	}
	
	private void populateTable(List<UITitleSearchBase> searches) {
		flexTable.clear();
		flexTable.addStyleName(STYLE_NAME_SAVED_SEARCHES_TABLE);
		
		flexTable.getRowFormatter().addStyleName(0, STYLE_NAME_HEADER_ROW);
		
		allCB = new CheckBoxWidget();
		allCB.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
			@Override
			public void onValueChange(ValueChangeEvent<Boolean> event) {
				if (checkBoxesList != null) {
					for (CheckBoxWidget cb : checkBoxesList) {
						cb.setValue(event.getValue(), true);
					}
				}
			}});
		
		flexTable.setWidget(0, 0, allCB);
		flexTable.setText(0, 1, contentConstants.searchListIDColumnHeader());
		flexTable.setText(0, 2, contentConstants.searchListNameColumnHeader());
		flexTable.setText(0, 3, contentConstants.searchListSharedColumnHeader());
		flexTable.setText(0, 4, contentConstants.searchListOwnerColumnHeader());
		flexTable.setText(0, 5, contentConstants.searchListGoColumnHeader());
		
		flexTable.getCellFormatter().addStyleName(0, 0,
				STYLE_NAME_CHECKBOX_CELL);

		int rowIndex = 1;
		// we start out by adding an empty row
		UITitleSearchBase empty = new UITitleSearchBase(-1,
				contentConstants.showAllSearch(), neptuneSecurity
						.getUsername());
		populateTableRow(empty, rowIndex);
		
		for (UITitleSearchBase search : searches) {
			rowIndex++;
			populateTableRow(search, rowIndex);			
		}
	}
	
	private void populateTableRow(final UITitleSearchBase search, int rowNumber) {
		flexTable.getRowFormatter().addStyleName(rowNumber, STYLE_NAME_DATA_ROW);
		
		final int titleSearchId = search.getTitleSearchId();
		boolean isShared = search.isShare();
		String owner = search.getUserName();
		boolean isCurrentUserAdmin = neptuneSecurity.isUserInRole(ROLE_NAME_ADMIN);
		boolean isDeletable = false;
		if (isCurrentUserAdmin || getCurrentUser().equalsIgnoreCase(owner)) {
			isDeletable = true;
		}
		
		if ((titleSearchId > 0) && isDeletable) {
			CheckBoxWidget cb = new CheckBoxWidget();
			flexTable.setWidget(rowNumber, 0, cb);
			flexTable.getCellFormatter().addStyleName(rowNumber, 0,
					STYLE_NAME_CHECKBOX_CELL);
			checkBoxesList.add(cb);
			cb.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
				@Override
				public void onValueChange(ValueChangeEvent<Boolean> event) {
					if (event.getValue()) {
						selectedIds.add(titleSearchId);
					} else {
						allCB.setValue(false, false);
						selectedIds.remove(titleSearchId);
					}
				}
			});
		}
		
		String strTitleSearchId = String.valueOf(titleSearchId);
		if (titleSearchId <= 0) {
			strTitleSearchId = "";
		}
		
		LabelWidget labelId = new LabelWidget(strTitleSearchId);
		labelId.addStyleName(STYLE_NAME_SAVED_SEARCH_HYPERLINK);
		flexTable.setWidget(rowNumber, 1, labelId);
		flexTable.getCellFormatter().addStyleName(rowNumber, 1, STYLE_NAME_SEARCH_ID_CELL);
		labelId.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				goToSavedSearch(titleSearchId);			
			}
		});

		LabelWidget labelName = new LabelWidget(search.getName());
		flexTable.setWidget(rowNumber, 2, labelName);
		flexTable.getCellFormatter().addStyleName(rowNumber, 2, STYLE_NAME_SEARCH_NAME_CELL);
		if (titleSearchId > 0) {
			labelName.addStyleName(STYLE_NAME_SAVED_SEARCH_HYPERLINK);
			labelName.addClickHandler(new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					goToSavedSearch(titleSearchId);			
				}
			});
		}

		HTMLWidget htmlShared = new HTMLWidget("<b>&#x2713;</b>");
		if ((titleSearchId == -1) || isShared) {
			flexTable.setWidget(rowNumber, 3, htmlShared);
			flexTable.getCellFormatter().addStyleName(rowNumber, 3, STYLE_NAME_SHARED_CELL);
		}

		LabelWidget labelOwner = new LabelWidget(search.getUserName());
		flexTable.setWidget(rowNumber, 4, labelOwner);
		flexTable.getCellFormatter().addStyleName(rowNumber, 4, STYLE_NAME_SEARCH_OWNER_CELL);

		ButtonWidget goButton =  new ButtonWidget(contentConstants.goButton());
		goButton.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		goButton.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				resetMessageListener();
				if (search.getTitleSearchId() == -1) {
					if (validate()) {
						TitleSearchEvent searchEvent = new TitleSearchEvent(
								getAllTitlesFilter(),
								getSelectedSearchResultsColumns());
						fireEvent(searchEvent);
					}
				} else {
					getFilterAndFireSearch(search);
				}
			}
			
		});
		
		flexTable.setWidget(rowNumber, 5, goButton);
		flexTable.getCellFormatter().addStyleName(rowNumber, 5, STYLE_NAME_GO_BUTTON_CELL);		
	}
	
	private void goToSavedSearch(int titleSearchId) {
		
		searchTabPanel.goToSavedSearch(titleSearchId);

	}
		
	private UITitleFilter getAllTitlesFilter() {
		UITitleFilter filter = new UITitleFilter();
		setFilterSortInfo(filter);
		filter.setCriteriaMode(criteriaMode);
		filter.setRootNode(null);
		filter.setIncludeCount(true);
		return filter;
	}

	private void setFilterSortInfo(UITitleFilter filter) {
		if(getSelectedSortColumn().getFieldType().equals(UIFieldType.ASSET)) {
			filter.getSortInfo().setSortFieldType(UIFieldType.ASSET);
			filter.getSortInfo().setSortColumnSection(getSelectedSortColumn().getSectionName());
		} else {
			filter.getSortInfo().setSortFieldType(UIFieldType.PMM_TITLE);
		}
		filter.getSortInfo().setSortColumnName(getSelectedSortColumn().getName());
		filter.getSortInfo().setSortOrder(getSelectedSortOrder());
	}

	/*
	 * gets the filter for a search that already exists fresh from the database
	 * and fires a search event for it.
	 */
	private void getFilterAndFireSearch(UITitleSearchBase search) {
		busyIndicator.center();
		criteriaService.getTitleSearch(search.getTitleSearchId(),
				new NeptuneAsyncCallback<UITitleSearch>() {

					@Override
					public void onNeptuneFailure(Throwable caught) {
						if (caught instanceof SearchNotFoundException) {
							showErrorMessage(contentConstants
									.searchNotFound());
						} else {
							showErrorMessage(contentConstants
									.errRetrievingSearch());
						}
						busyIndicator.hide();
					}

					@Override
					public void onNeptuneSuccess(UITitleSearch result) {
						busyIndicator.hide();
						UITitleFilter filter = result.getTitleFilter();
						filter.setIncludeCount(result.getTitleFilter().isCountIncluded());
						List<UiColumn> queryColumns = searchColumnPanel.getUiColumnsForSavedDisplayColumns(result.getDisplayColumns());
						TitleSearchEvent searchEvent = new TitleSearchEvent(filter, queryColumns);
						fireEvent(searchEvent);
					}
				});
	}

	private ComplexHeaderPanel buildComplexHeaderPanel() {
		ComplexHeaderPanel complexHeaderPanel = new ComplexHeaderPanel();
		complexHeaderPanel.addHeaderDescription(contentConstants.savedSearchHeaderDescription());
		complexHeaderPanel.addHeaderTip(contentConstants.savedSearchTip());
		
		return complexHeaderPanel;
	}

}
