package com.tandbergtv.cms.portal.content.client.tab.search;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.i18n.ContentConstants;
import com.tandbergtv.cms.portal.content.client.i18n.ContentMessages;
import com.tandbergtv.cms.portal.content.client.rpc.title.SearchColumnPreference;
import com.tandbergtv.cms.portal.content.client.rpc.title.SearchPreferenceDisplayColumn;
import com.tandbergtv.cms.portal.content.client.rpc.title.UiTitleCriteriaService;
import com.tandbergtv.cms.portal.content.client.rpc.title.UiTitleCriteriaServiceAsync;
import com.tandbergtv.cms.portal.content.client.tab.list.assettable.SearchResultColumnProvider;
import com.tandbergtv.cms.portal.content.client.tab.list.assettable.SearchResultColumnProvider.SearchColumns;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UISortOrder;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiColumn;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiSortInfo;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWithHiddenNameWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.orderabletable.OrderableTable;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.orderabletable.TableRow;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.IMessageEmitter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.resizablecontainer.IMessageListener;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * This panel contains all widgets for user to select search results columns and
 * their sorting order.
 * 
 * @author spuranik
 * 
 */
public class SearchColumnPanel extends VerticalContainer implements IMessageEmitter {

	private static final String STYLE_PREFERENCES_PANEL = "preferences-panel";
	
	private static final UISortOrder DEFAULT_SORT_ORDER = UISortOrder.DESCENDING;

	private static int HEADER_ROW = 1;
	private static String SEARCH_COL_NAME = "Columns";

	// column names for which re-ordering is not supported
	private final List<String> noOrderColumns = new ArrayList<String>(Arrays
			.asList(SearchSpecialColumns.ID_COLUMN));
	
	// table which shows the available columns in the search results page 
	OrderableTable displayColTable;
	
	// container which has the sortBy column selection list box and order icon
	HorizontalContainer sortByContainer;
	ListBoxWidget<UiColumn> sortByListBox;
	ListBoxWidget<UISortOrder> sortOrderListBox;

	// button that saves the selection in user preferences
	ButtonWidget saveAsPreferencesButton;
	
	// list of available search results display columns
	private List<UiColumn> searchResultColumns;
	
	// list of available sort columns
	private List<UiColumn> sortColumns;
	
	private String defaultInternalSortColumnName;
	
	private ContentConstants constants = (ContentConstants) GWT.create(ContentConstants.class);
	private ContentMessages messages = (ContentMessages) GWT.create(ContentMessages.class);
	
	private UiTitleCriteriaServiceAsync criteriaService = GWT.create(UiTitleCriteriaService.class);
	
	// selected cols from user prefs
	private SearchColumnPreference prefs;
	
	private static String MISSING_DISPLAY_COL_DELIMITER = ",";
	// since id is always the first item in the display cols, following the same
	// order when selecting the default search column
	private static int ID_SORT_POSITION = 0; 
	
	private UiCriteriaMode criteriaMode;
	
	private IMessageListener messageListener;

	private boolean hasUserDefaultPreferences;
	
	private boolean hasInstructions;
	
	public SearchColumnPanel(UiCriteriaMode criteriaMode, boolean hasUserDefaultPreferences, boolean hasInstructions) {
		this.criteriaMode = criteriaMode;
		this.hasUserDefaultPreferences = hasUserDefaultPreferences;
		this.hasInstructions = hasInstructions;
	}
	
	/**
	 * Init this panel which includes populating all widgets with thier data 
	 * 
	 * @param initCallback
	 */
	public void init(final NeptuneAsyncCallback<Void> initCallback) {
		// fetch the column info first
		final NeptuneAsyncCallback<Void> callback = new NeptuneAsyncCallback<Void>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				initCallback.onFailure(caught);
				Window.alert("Failed to initialize Search Column Table. Reason: "
						+ ((caught != null) ? caught.getMessage() : ""));
			}

			@Override
			public void onNeptuneSuccess(Void result) {
				// init all widgets now that we have the data to populate in the widgets
				initWidgets();
				initCallback.onSuccess(null);
			}
		};
		// before initing the widgets get the user preferences
		getUserPreferences(criteriaMode, callback);
	}

	/**
	 * Gets the user preferences for this user so the selected columns can be
	 * set based on his preferences if any
	 * 
	 * @param callback
	 */
	private void getUserPreferences(UiCriteriaMode criteriaMode, final NeptuneAsyncCallback<Void> callback) {
		criteriaService.getSearchColumnPreferences(criteriaMode, new NeptuneAsyncCallback<SearchColumnPreference>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(SearchColumnPreference prefs) {
				setPrefs(prefs);
				getColumns(callback);				 
			} 
		});
	}

	private void getColumns(final NeptuneAsyncCallback<Void> callback) {
		SearchResultColumnProvider columnProvider = new SearchResultColumnProvider(criteriaMode);
		columnProvider.getSearchColumns(new NeptuneAsyncCallback<SearchColumns>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(SearchColumns result) {
				setColumns(result);
				callback.onNeptuneSuccess(null);
			}
		});
	}

	private void setColumns(SearchColumns cols) {
		setSearchResultColumns(cols.getDisplayColumns());
		setSortColumns(cols.getSortColumns());
		setDefaultInternalSortColumnName(cols.getDefaultInternalSortColumnName());
	}
	
	/**
	 *  Creates and inits all required widgets.
	 *  
	 */
	private void initWidgets() {
		this.clear();

		createTable();
		
		VerticalContainer chVerticalContainer = new VerticalContainer();
		
		if (hasInstructions) {
			ComplexHeaderPanel complexHeaderPanel = buildComplexHeaderPanel();
			chVerticalContainer.add(complexHeaderPanel);
			this.add(chVerticalContainer);
		}
		
		VerticalContainer prefsVerticalContainer = new VerticalContainer();
		this.add(prefsVerticalContainer);
		
		prefsVerticalContainer.add(displayColTable);
		
		buildSortByPanel();
		prefsVerticalContainer.add(sortByContainer);
		
		this.addStyleName(STYLE_PREFERENCES_PANEL);
		
		HorizontalContainer buttonsContainer = new HorizontalContainer();
		prefsVerticalContainer.add(buttonsContainer);

		// save as preferences button
		saveAsPreferencesButton = new ButtonWidget(constants.saveColumnsAsPreferences());
		saveAsPreferencesButton.addStyleDependentName(StyleNames.DO_BUTTON_STYLE);
		saveAsPreferencesButton.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				try {
					saveColumnInfo();
				} catch (SelectionValidationException e) {
					showErrorMessage(e.getMessage());
				}
			}
		});
		buttonsContainer.add(saveAsPreferencesButton);
		saveAsPreferencesButton.setVisible(hasUserDefaultPreferences);
		
		// notify the user about any missing columns if he had any preferences saved
		if(prefs != null) {
			List<UiColumn> selPrefColumns = new ArrayList<UiColumn>();
			for(SearchPreferenceDisplayColumn sc : prefs.getSelectedColumns()) {
				UiColumn c = new UiColumn();
				c.setInternalName(sc.getInternalName());
				selPrefColumns.add(c);
			}
			UiColumn selectedSortCol = new UiColumn();
			selectedSortCol.setInternalName(prefs.getSortColumn().getInternalName());
			
			UiColumn missingSortCol = null;
			List<UiColumn> missingCols = findMissingDisplayColumns(selPrefColumns);
			if(isSortColumnMissing(selectedSortCol)) {
				missingSortCol = selectedSortCol;
			}

			notifyMissingSelections(missingCols, missingSortCol);
			// remove missing selections from ui model or else when the user
			// saves his preferences again, they will be passed back to the
			// server
			removeMissingFromPrefsSelections(missingCols, missingSortCol);
		}
		setSelectionsFromPrefsOrDefault();
		setSortOrderFromPrefs();
	}

	private void removeMissingFromPrefsSelections(List<UiColumn> missingCols, UiColumn missingSortCol) {
		if (missingCols != null) {
			// notify about missing display cols
			for (UiColumn missing : missingCols) {
				for (SearchPreferenceDisplayColumn prefCol : prefs.getSelectedColumns()) {
					if (prefCol.getInternalName().equals(missing.getInternalName())) {
						prefs.getSelectedColumns().remove(prefCol);
						break;
					}
				}
			}
		}
		
		// notify about missing sort column
		if(missingSortCol != null) {
			// reassign sort to default
			String idInternalName = sortByListBox.getItem(ID_SORT_POSITION).getInternalName();
			SearchPreferenceDisplayColumn defaultSortCol = new SearchPreferenceDisplayColumn(idInternalName);
			prefs.setSortColumn(defaultSortCol);
		}
	}

	private void createTable() {
		// TODO: check permissions
		displayColTable = new OrderableTable("", true, true, noOrderColumns.size());
		displayColTable.addColumn(constants.searchResultsColumnName(), SEARCH_COL_NAME);		
	}

	private void showSuccessMessage(String message) {
		showInfoMessage(message);
	}

	/**
	 * go thru available list of columns and see if the given set of cols is
	 * present in it. If not return those cols.
	 * 
	 * @param displayCols
	 * @return
	 */
	private List<UiColumn> findMissingDisplayColumns(List<UiColumn> displayCols) {
		List<UiColumn> missingDisplayColumns =  new ArrayList<UiColumn>();
		for(UiColumn col : displayCols) {
			boolean foundMatchingResultCol = false;
			for(UiColumn uic : getSearchResultColumns()) { 
				if(uic.getInternalName().equals(col.getInternalName())) {
					foundMatchingResultCol = true;
					break;
				}
			}
			// add it to the list of cols which were in the user prefs but
			// not found in the list of available display cols
			if(!foundMatchingResultCol) {
				missingDisplayColumns.add(col);
			}
		}
		return missingDisplayColumns;		
	}
	
	private boolean isSortColumnMissing(UiColumn sortCol) {
		boolean sortColIsMissing = true;
		for (int i = 0; i < sortByListBox.getItemCount(); i++) {
			if(sortByListBox.getItem(i).equals(sortCol)) {
				sortColIsMissing = false;
				break;
			}
		}
		return sortColIsMissing;
	}

	/**
	 * If the user had selected this col previously and saved it in his
	 * preferences use that or else see if the config dictates it to be selected
	 * by default.
	 * 
	 * @param c
	 * @return
	 */
	private boolean isSelected(UiColumn c) {
		String internalName = c.getInternalName();
		if(prefs != null && prefs.getSelectedColumns() != null) {
			for(SearchPreferenceDisplayColumn dc : prefs.getSelectedColumns()) {
				if(internalName.equals(dc.getInternalName())) {
					return true;
				}
			}
			return false;
		} else {
			return c.isSelectedByDefault();
		}
	}

	private void saveColumnInfo() throws SelectionValidationException {
		validateSelections();
		
		// prepare list of selected columns
		List<SearchPreferenceDisplayColumn> selectedColumns = new ArrayList<SearchPreferenceDisplayColumn>();

		String internalColumnName = "";
		for(int i=0; i < displayColTable.getRowCount(); i++) {
			TableRow row = displayColTable.getTableRow(i);
			Widget column = displayColTable.getWidget(i + HEADER_ROW, SEARCH_COL_NAME);
			if(column instanceof LabelWithHiddenNameWidget) {
				internalColumnName = ((LabelWithHiddenNameWidget)column).getInternalText();
			}
			if (row.isChecked()) {
				SearchPreferenceDisplayColumn c = new SearchPreferenceDisplayColumn(internalColumnName);
				selectedColumns.add(c);
			}			
		}

		// prepare sort info
		SearchPreferenceDisplayColumn sortColumn = new SearchPreferenceDisplayColumn(
				sortByListBox.getSelectedItem().getInternalName());
		UISortOrder sortOrder = getSelectedSortOrder();
		
		final SearchColumnPreference preference = new SearchColumnPreference(selectedColumns, sortColumn, sortOrder.ordinal());
		criteriaService.saveSearchColumnsAsPreference(preference, criteriaMode, new AsyncCallback<Void>() {

			@Override
			public void onFailure(Throwable caught) {
				showErrorMessage(constants.preferencesNotSaved() + caught.getLocalizedMessage());
			}

			@Override
			public void onSuccess(Void result) {
				setPrefs(preference);
				showSuccessMessage(constants.preferencesSaved());
			}
		});
	} 
	
	public void validateSelections() throws SelectionValidationException {
		// check if the sort column is selected for display
		if(!isSortcolumnSelected()) {
			throw new SelectionValidationException(constants.sortColumnValidationError());
		}
	}

	/**
	 * Some columns cannot be re-ordered. Their position is fixed. For such
	 * columns, do not show the draggable icon
	 * 
	 * @param column
	 * @return
	 */
	private boolean isDragAllowed(UiColumn column) {
		return !(noOrderColumns.contains(column.getName()));
	}

	private void buildSortByPanel() {
		// sort by label
		LabelWidget sortByLabel = new LabelWidget("Sort By: ");
		
		// sort column drop down
		sortByListBox = new ListBoxWidget<UiColumn>();
		for(UiColumn c : this.getSortColumns()) {
			sortByListBox.insertItem(c.getDisplayName(), c, sortByListBox.getItemCount());
		}
		setSelectedSortColumn();
 		
		// container which has the above 2 widgets
		sortByContainer = new HorizontalContainer();
		sortByContainer.add(sortByLabel);
		sortByContainer.add(sortByListBox);
		sortByContainer.setSpacing(3);
		
		sortOrderListBox = new ListBoxWidget<UISortOrder>();
		sortOrderListBox.insertItem(UISortOrder.ASCENDING.name(), UISortOrder.ASCENDING, 0);
		sortOrderListBox.insertItem(UISortOrder.DESCENDING.name(), UISortOrder.DESCENDING, 1);
		sortOrderListBox.setSelectedItem(DEFAULT_SORT_ORDER);
		sortByContainer.add(sortOrderListBox);
	}

	private void setSelectedSortColumn() {
		for(UiColumn c : this.getSortColumns()) {
			if(isSelectedForSort(c)) {
				sortByListBox.setSelectedItem(c);
			}
		}
	}

	private void setSortOrderFromPrefs() {
		if(prefs != null && prefs.getSortOrder() != -1) {
			int intSortOrder = prefs.getSortOrder();
			
			switch (intSortOrder) {
				case 0:
					sortOrderListBox.setSelectedItem(UISortOrder.ASCENDING);
					break;
				case 1:
					sortOrderListBox.setSelectedItem(UISortOrder.DESCENDING);		
					break;
			}
		}
		else {
			sortOrderListBox.setSelectedItem(DEFAULT_SORT_ORDER);					
		}
		
	}

	/**
	 * Determines whether the column is used as a sort column in the user prefrences 
	 * 
	 * @param c
	 * @return
	 */
	private boolean isSelectedForSort(UiColumn c) {
		if(prefs != null && prefs.getSortColumn() != null) {
			return this.prefs.getSortColumn().getInternalName().equals(c.getInternalName());	
		}
		return false;
	}

	public List<UiColumn> getSearchResultColumns() {
		return searchResultColumns;
	}

	public void setSearchResultColumns(List<UiColumn> searchResultColumns) {
		this.searchResultColumns = searchResultColumns;
	}

	public List<UiColumn> getSortColumns() {
		return sortColumns;
	}

	public void setSortColumns(List<UiColumn> sortColumns) {
		this.sortColumns = sortColumns;
	}

	public void setPrefs(SearchColumnPreference prefs) {
		if (prefs.getSelectedColumns().size() > 0
				&& prefs.getSortColumn() != null) {
			this.prefs = prefs;
		}
	}

	/**
	 * Returns a list of ui cols the user has selected  
	 * 
	 * @return
	 */
	public List<UiColumn> getSelectedColumns() {
		List<UiColumn> selectedCols = new ArrayList<UiColumn>();
		
		String internalColumnName = "";
		for(int i=0; i < displayColTable.getRowCount(); i++) {
			TableRow row = displayColTable.getTableRow(i);
			Widget column = displayColTable.getWidget(i + HEADER_ROW, SEARCH_COL_NAME);
			if(column instanceof LabelWithHiddenNameWidget) {
				internalColumnName = ((LabelWithHiddenNameWidget)column).getInternalText();
			}
			
			if (row.isChecked()) {
				selectedCols.add(getColumn(internalColumnName));
			}
		}
		return selectedCols;
	}	
	
	/**
	 * Given the internal name finds the corresponding complete ui column object
	 * 
	 * @param internalColumnName
	 * @return
	 */
	private UiColumn getColumn(String internalColumnName) {
		for(UiColumn c : searchResultColumns) {
			if(c.getInternalName().equals(internalColumnName)) {
				return c;
			}
		}
		return null;
	}

	public String getDefaultInternalSortColumnName() {
		return defaultInternalSortColumnName;
	}

	public void setDefaultInternalSortColumnName(
			String defaultInternalSortColumnName) {
		this.defaultInternalSortColumnName = defaultInternalSortColumnName;
	}
	
	public UiColumn getSelectedSortColumn() {
		return this.sortByListBox.getSelectedItem();
	}
	
	public UISortOrder getSelectedSortOrder() {
		UISortOrder sortOrder = sortOrderListBox.getSelectedItem();
		return sortOrder;
	}
	
	@Override
	public void registerMessageListener(IMessageListener messageListener) {
		this.messageListener = messageListener;
	}

	public void showInfoMessage(String infoMsg) {
		if (messageListener != null) {
			messageListener.setInfoMessage(infoMsg);
		}
	}
	
	public void showErrorMessage(String errorMessage) {
		if (messageListener != null) {
			messageListener.setErrorMessage(errorMessage);
		}
	}
	
	public void resetMessageListener() {
		if (messageListener != null) {
			messageListener.reset();
		}
	}

	/**
	 * checks if the column selected to sort by is selected for display
	 * 
	 * @return
	 */
	private boolean isSortcolumnSelected() {
		String sortColDisplayName = this.sortByListBox.getSelectedItem().getDisplayName();
		for(int i=0; i < displayColTable.getRowCount(); i++) {
			TableRow row = displayColTable.getTableRow(i);
			Widget column = displayColTable.getWidget(i + HEADER_ROW, SEARCH_COL_NAME);
			if(column instanceof LabelWithHiddenNameWidget) {
				String columnName = ((LabelWithHiddenNameWidget)column).getText();
				if(columnName.equals(sortColDisplayName)) {
					return row.isChecked();
				}
			}
		}
		return false;
	}
	
	public void setSelections(List<UiColumn> displayCols, UiColumn sortCol,
			UISortOrder sortOrder) {
		this.displayColTable.clear();
		if (displayCols.isEmpty()) {
			// this would happen for searches which have been created prior to
			// this feature set selections from user prefs
			setSelectionsFromPrefsOrDefault();
			setSortOrderFromPrefs();
			setSelectedSortColumn();
		} else {
			UiColumn missingSortCol = null;
			List<UiColumn> missingCols = findMissingDisplayColumns(displayCols);
			boolean sortColIsMissing = isSortColumnMissing(sortCol);
			if(sortColIsMissing) {
				missingSortCol = sortCol;
			}
			notifyMissingSelections(missingCols, missingSortCol);
			removeDisplayColumnsFromSelection(missingCols, displayCols);
			
			List<UiColumn> cols = reorderDisplayCols(displayCols);

			for (int i = 0; i < cols.size(); i++) {
				UiColumn c = cols.get(i);
				Widget label = new LabelWithHiddenNameWidget(c
						.getInternalName(), c.getDisplayName());
				try {
					boolean dragAllowed = isDragAllowed(c);
					// title id row cannot be unchecked.
					boolean chbkEnabled = c.getInternalName().equals(SearchSpecialColumns.ID_COLUMN) ? false : true; 
					displayColTable.addItem(displayColTable.getRowCount(),
							SEARCH_COL_NAME, label,
							TableConstants.STYLE_DATACELL_TEXT, dragAllowed, c.isSelected(), chbkEnabled);
				} catch (Exception e) {
					String msg = "Error populating table: ";
					showErrorMessage(msg + e.getMessage());
					throw new RuntimeException("Error populating table", e);
				}
			}
			
			// sort column
			if(!sortColIsMissing) {
				this.sortByListBox.setSelectedItem(sortCol);
			}

			sortOrderListBox.setSelectedItem(sortOrder);
		}				
	}
	
	private void removeDisplayColumnsFromSelection(List<UiColumn> missingCols,
			List<UiColumn> displayCols) {
		for (UiColumn mCol : missingCols) {
			for (UiColumn dCol : displayCols) {
				if (mCol.getInternalName().equalsIgnoreCase(dCol.getInternalName())) {
					displayCols.remove(dCol);
					break;
				}
			}
		}
	}

	private void notifyMissingSelections(List<UiColumn> missingCols, UiColumn sortCol) {
			if(missingCols != null) {
			StringBuilder mdCols = new StringBuilder();
			// if there are missing cols build and display notification message 
			if(missingCols.size() > 0) {
				for(UiColumn missingCol : missingCols) {
					if(mdCols.length() > 0) {
						mdCols.append(MISSING_DISPLAY_COL_DELIMITER);
					}
					mdCols.append(missingCol.getInternalName());
				}
				showErrorMessage(messages
						.missingSelectedSearchDisplayColumns(mdCols.toString()));
			}
		}
		
		// notify about missing sort column
		if(sortCol != null) {
				showErrorMessage(messages
						.missingSelectedSortColumn(sortCol.getInternalName()));
				this.sortByListBox.setSelectedIndex(ID_SORT_POSITION);
		}
	}

	private List<UiColumn> reorderDisplayCols(List<UiColumn> displayCols) {
		List<UiColumn> uiCols = new ArrayList<UiColumn>();
		// add special cols first
		addSpecialCols(uiCols);
		// select them if there selected in saved search
		selectSpecialCols(uiCols, displayCols);
		// then add all the selected cols, excluding the special columns
		for (int i = 0; i < displayCols.size(); i++) {
			if(!noOrderColumns.contains(displayCols.get(i).getInternalName())) {
				uiCols.add(displayCols.get(i));
			}
		}
		
		// now add the ones which were not selected excluding the special cols 
		for(UiColumn c : getSearchResultColumns()) {
			boolean selected = false;
			for(UiColumn selCol : displayCols) {
				if(c.getInternalName().equals(selCol.getInternalName())) {
					// Above the selected column was added to the list but it does not 
					// have a display name selected flag set. So do that here..yes its ugly! 
					selCol.setDisplayName(c.getDisplayName());
					selCol.setSelected(true);
					selected = true;
					break;
				}
			}
			// if this column was not one of the selected cols and is not a special column, add this col
			if(!selected && !noOrderColumns.contains(c.getInternalName())) {
				// if the column was selected prior to loading a saved search
				// where this col is not selected, make sure it is unchecked now
				c.setSelected(false);
				uiCols.add(c);
			}
		}
		return uiCols;
	}	

	private void selectSpecialCols(List<UiColumn> uiCols, List<UiColumn> displayCols) {
		for(UiColumn c : uiCols) {
			boolean selected = false;
			for(UiColumn dc : displayCols) {
				if(c.getInternalName().equals(dc.getInternalName())) {
					// if the col being added in the ui list is found in the
					// display col list, means it has been selected
					selected = true;
					c.setSelected(true);
					break;
				}
				if(!selected) {
					c.setSelected(false);
				}
			}
		}
	}

	/**
	 * inserts the special cols to the incoming list if they are available in
	 * the list of selectable display cols
	 * 
	 * @param uiCols
	 */
	private void addSpecialCols(List<UiColumn> uiCols) {
		// is id in the list of display columns
		for(UiColumn c : getSearchResultColumns()) {
			if(c.getInternalName().equals(SearchSpecialColumns.ID_COLUMN)) {
				uiCols.add(c);
				break;
			}
		}
	}

	public void setSelectionsFromPrefsOrDefault() {
		List<UiColumn> selectedCols = new ArrayList<UiColumn>();
		if(prefs == null) {
			selectedCols = getSearchResultColumns();
		} else {
			for (int i = 0; i < prefs.getSelectedColumns().size(); i++) {
				UiColumn c = new UiColumn();
				c.setInternalName(prefs.getSelectedColumns().get(i).getInternalName());
				selectedCols.add(c);
			}
		}
		List<UiColumn> reorderedDisplayColumns = reorderDisplayCols(selectedCols);
		
		for (int i = 0; i < reorderedDisplayColumns.size(); i++) {
			UiColumn c = reorderedDisplayColumns.get(i);
			Widget label = new LabelWithHiddenNameWidget(c.getInternalName(), c.getDisplayName());
			try {
				boolean dragAllowed = isDragAllowed(c);
				boolean isSelected = isSelected(c);
				// title id row cannot be unchecked.
				boolean chbkEnabled = c.getInternalName().equals(SearchSpecialColumns.ID_COLUMN) ? false : true;
				displayColTable.addItem(displayColTable.getRowCount(),
						SEARCH_COL_NAME, label, TableConstants.STYLE_DATACELL_TEXT, dragAllowed, isSelected, chbkEnabled);
				
			} catch(Exception e) {
				String msg = "Error populating table: ";
				showErrorMessage(msg + e.getMessage());
				throw new RuntimeException("Error populating table", e);
			}
		}
	}
	
	public void clearMessages() {
		resetMessageListener();
	}

	/**
	 * Finds the display column for every column psased in. This is because the
	 * saved search column only have the internal name.
	 * 
	 * @param savedCols
	 * @return
	 */
	public List<UiColumn> getUiColumnsForSavedDisplayColumns(List<UiColumn> savedCols) {
		List<UiColumn> displayCols = new ArrayList<UiColumn>();
		if(savedCols.isEmpty()) {
			for(UiColumn dc : getSearchResultColumns()) {
				if(isSelected(dc)) {
					displayCols.add(dc);
				}
			}
		} else  {
			for(int i=0; i < savedCols.size(); i++) {
				for(UiColumn c : searchResultColumns) {
					if(c.getInternalName().equalsIgnoreCase(savedCols.get(i).getInternalName())) {
						displayCols.add(c);
						break;
					}
				}
			}
		}
		return displayCols;
	}
	
	public UiColumn getUiColumnForSavedSortColumn(UiSortInfo savedCol) {
		for (UiColumn c : sortColumns) {

			if (c.getName().equalsIgnoreCase(savedCol.getSortColumnName())) {
				// title sort condition
				if (c.getSectionName() == null
						&& savedCol.getSortColumnSection() == null) {
					return c;
				} else {
					// asset types have to match as well
					if (c.getSectionName() != null && savedCol.getSortColumnSection() != null) {
						if (c.getSectionName().equalsIgnoreCase(savedCol.getSortColumnSection())) {
							return c;
						}
					}
				}
			}
		}
		return null;
	}
	
	public void clearSelections() {
		this.displayColTable.clear();
	}

	private ComplexHeaderPanel buildComplexHeaderPanel() {
		ComplexHeaderPanel complexHeaderPanel = new ComplexHeaderPanel();
		complexHeaderPanel.addHeaderDescription(constants.prefrencesHeaderDescription());
		for (String headerSubDescription : constants.prefrencesHeaderSubDescriptions()) {
			complexHeaderPanel.addHeaderSubDescription(headerSubDescription);
		}
		String headerNote = constants.prefrencesNote();
		if (UiCriteriaMode.SERIES.equals(criteriaMode)) {
			headerNote = constants.prefrencesNoteSeries();
		}
		complexHeaderPanel.addHeaderNote(headerNote);
		return complexHeaderPanel;
	}

}
