package com.tandbergtv.cms.portal.content.client.tab.search;

import static com.google.gwt.user.client.ui.HasHorizontalAlignment.ALIGN_CENTER;
import static com.google.gwt.user.client.ui.HasVerticalAlignment.ALIGN_TOP;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.i18n.ContentConstants;
import com.tandbergtv.cms.portal.content.client.model.NoPartnersException;
import com.tandbergtv.cms.portal.content.client.model.UiInvalidQwikSrchException;
import com.tandbergtv.cms.portal.content.client.tab.list.ResultsLoadedHandler;
import com.tandbergtv.cms.portal.content.client.tab.list.TitleListPanel;
import com.tandbergtv.cms.portal.content.client.tab.list.TitleListViewInput;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilter;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UITitleFilterQuickSearch;
import com.tandbergtv.cms.portal.ui.title.client.model.search.UiSortInfo;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;

/**
 * Panel which is displayed when the user clicks the 'Search' option under
 * 'Content' tab.
 * 
 * @author spuranik
 */
public class TitleSearchPanel extends SimpleContainer {

	/* Model / Properties */
	private final NeptuneSecurity security;
	private final TitleListViewInput titleListViewInput;
	private Long requestKey;
	private String action;
	private String listOfAssetIds;
	private String tab;
	private Integer searchId;
	private AnchorChangeListener anchorChangeNotifier;
	
	/* Widgets */
	private SimpleContainer searchPanel;
	private SimpleContainer criteriaPanelWrapper;
	private SearchTabPanel searchTabPanel;
	private SimpleContainer searchResultPanelWrapper;

	private HeaderPanel headerPanel;
	
	private ContentConstants myConstants = (ContentConstants) GWT.create(ContentConstants.class);
	private UiCriteriaMode criteriaMode;

	/**
	 * Create the new Title Search Panel using the provided security information 
	 * @param security The security information
	 * @param specificationName 
	 */
	public TitleSearchPanel(NeptuneSecurity security, UiCriteriaMode criteriaMode) {
		this.security = security;
		this.criteriaMode = criteriaMode;
		titleListViewInput = new TitleListViewInput(0, 0, 0);
		initialize();
	}

	private void initialize() {
		/* Build the search panel and the search result panel */
		searchPanel = new SimpleContainer();
		searchResultPanelWrapper = new SimpleContainer();

		/* Build the contents of the search panel */
		VerticalContainer containerPanel = new VerticalContainer();
		containerPanel.setVerticalAlignment(ALIGN_TOP);
		containerPanel.setHorizontalAlignment(ALIGN_CENTER);
		
		containerPanel.setSpacing(5);

		containerPanel.setWidth("100%");
		
		/* Add the heading label */ 
		if(criteriaMode == UiCriteriaMode.EPISODE) {
			headerPanel = new HeaderPanel(myConstants.advancedSearchLabel());
		} else if(criteriaMode == UiCriteriaMode.SERIES) {
			headerPanel = new HeaderPanel(myConstants.seriesAdvancedSearchLabel());
		}
		
		containerPanel.add(headerPanel);
		
		/* Add the search criteria panel */
		criteriaPanelWrapper = new SimpleContainer();
		criteriaPanelWrapper.setWidget(new LabelWidget());
		containerPanel.add(criteriaPanelWrapper);

		/* Set the contents of the search panel and search result panel */
		searchPanel.setWidget(containerPanel);
		searchResultPanelWrapper.setWidget(new LabelWidget(""));
		
		/* Set the contents of this widget to be a blank label */
		this.setWidget(searchPanel);
	}

	/**
	 * Reset the state and contents of this panel, causing the view to return to the default search
	 * page and transient state maintained being cleared.
	 */
	public void reset() {
		/* Reset the transient state maintained */
		this.titleListViewInput.resetInfo();
		this.requestKey = null;
		this.action = null;
		this.listOfAssetIds = null;

		/* Rebuild the search view and go to the search page */
		showSearchView();
	}
 
	/**
	 * Update the view to return the search form view. If the reset flag is set, reset the view to
	 * show a initial / reset search form
	 */
	public void showSearchView() {
		searchResultPanelWrapper.clear();
		
		searchTabPanel = new SearchTabPanel(security, criteriaMode);
		searchTabPanel.addHeightOffsettingWidget(headerPanel);
		searchTabPanel.setOffsetHeight(20);
		searchTabPanel.addTitleSearchHandler(getSearchHandler());
		
		criteriaPanelWrapper.setWidget(searchTabPanel);
		
		this.setWidget(searchPanel);
	}

	public void showPopulatedSearchPanel() {
		
		this.setWidget(searchPanel);
		
	}
	/**
	 * Update the view to return the search result view without changing the previous view state
	 */
	public void showSearchResultsView() {
		this.setWidget(searchResultPanelWrapper);
	}

	/**
	 * Determine if the search results view is showing 
	 */
	boolean isShowingSearchResultsView() {
		return (this.getWidget() != null && this.getWidget() == searchResultPanelWrapper);
	}

	/**
	 * Set the new widget that serves as the view for the search results page
	 */
	void setSearchResultsView(Widget widget) {
		searchResultPanelWrapper.setWidget(widget);
	}

	/**
	 * Get the search result view
	 * 
	 * @return The search result view
	 */
	Widget getSearchResultsView() {
		return searchResultPanelWrapper.getWidget();
	}

	private TitleSearchHandler getSearchHandler() {
		TitleSearchHandler searchHandler = new TitleSearchHandler() {

			@Override
			public void search(TitleSearchEvent event) {
				UITitleFilter filter = event.getTitleFilter();
				boolean isQuickSearch = false;
				if(filter.getRootNode() instanceof UITitleFilterQuickSearch)
					isQuickSearch = true;
				if (!security.isUserInRole("AssetManagement_View")) {
					Window.alert("User is not authorized to search for assets.");
					throw new RuntimeException("User is not authorized to search for assets.");
				}
				
				ResultsLoadedHandler handler = new ResultsLoadedHandler() {

					@Override
					public void resultsFailedToLoad(Throwable caught) {
						if (caught instanceof NoPartnersException) {
							searchTabPanel.getResizableContainer().setErrorMessage(myConstants.noParntersErrMsg());
						} else if(caught instanceof UiInvalidQwikSrchException) {
							searchTabPanel.getResizableContainer().setErrorMessage(myConstants.invalidQuickSearch());
						} else {
							showSearchResultsView();
							getAnchorChangeNotifier().anchorChanged("Results");
						}
					}

					@Override
					public void resultsLoaded() {
						showSearchResultsView();
						getAnchorChangeNotifier().anchorChanged("Results");
					}
					
				};

				UiSortInfo sInfo = event.getTitleFilter().getSortInfo();
				boolean isSeries=false;
				if (criteriaMode == UiCriteriaMode.SERIES )
					isSeries = true;
				TitleListPanel titleListPanel = new TitleListPanel(
						getAnchorChangeNotifier(), filter,
						getTitleListViewInput(), getRequestKey(), getAction(),
						getListOfAssetIds(), null, isQuickSearch, isSeries,
						handler, event.getDisplayColumns(),
						searchTabPanel.getSortColumn(sInfo),
						sInfo.getSortOrder(), searchResultPanelWrapper);
				
				SearchResultsPanel searchResultsPanel = new SearchResultsPanel(titleListPanel, isSeries);
				searchResultPanelWrapper.setWidget(searchResultsPanel);
				
				titleListPanel.setSearchResultsPanel(searchResultsPanel);
				
				/* Listen for view cancel */
				titleListPanel.addViewCancelHandler(new ViewCancelHandler() {
					@Override
					public void onCancel(ViewCancelEvent event) {
						showPopulatedSearchPanel();
						
					}
				});
				
			}
			
		};
		return searchHandler;
	}

	/**
	 * @return the requestKey
	 */
	public Long getRequestKey() {
		return requestKey;
	}

	/**
	 * @param requestKey the requestKey to set
	 */
	public void setRequestKey(Long requestKey) {
		this.requestKey = requestKey;
	}

	/**
	 * @return the action
	 */
	public String getAction() {
		return action;
	}

	/**
	 * @param action the action to set
	 */
	public void setAction(String action) {
		this.action = action;
	}
	
	/**
	 * @return the listOfAssetIds
	 */
	public String getListOfAssetIds() {
		return listOfAssetIds;
	}

	/**
	 * @param listOfAssetIds the listOfAssetIds to set
	 */
	public void setListOfAssetIds(String listOfAssetIds) {
		this.listOfAssetIds = listOfAssetIds;
	}

	/**
	 * @return the tab
	 */
	public String getTab() {
		return tab;
	}

	/**
	 * @param tab the tab to set
	 */
	public void setTab(String tab) {
		this.tab = tab;
	}

	/**
	 * @return the searchId
	 */
	public Integer getSearchId() {
		return searchId;
	}

	/**
	 * @param searchId the searchId to set
	 */
	public void setSearchId(Integer searchId) {
		this.searchId = searchId;
	}

	/**
	 * @return the titleListViewInput
	 */
	public TitleListViewInput getTitleListViewInput() {
		return titleListViewInput;
	}

	/**
     * @return the security
     */
    public NeptuneSecurity getSecurity() {
	    return security;
    }

	/**
     * @return the anchorChangeNotifier
     */
    public AnchorChangeListener getAnchorChangeNotifier() {
	    return anchorChangeNotifier;
    }
    
    /**
     * @param anchorChangeNotifier the anchorChangeNotifier to set
     */
    public void setAnchorChangeNotifier(AnchorChangeListener anchorChangeNotifier) {
	    this.anchorChangeNotifier = anchorChangeNotifier;
    }
}
