/*
 * Created on Aug 14, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.tab.search;

import java.util.HashMap;
import java.util.Map;

import com.google.gwt.user.client.History;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.tab.list.TitleListPanel;
import com.tandbergtv.cms.portal.content.client.title.view.TitleView;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.series.SeriesTitleView;
import com.tandbergtv.cms.portal.ui.title.client.criteria.UiCriteriaMode;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.LazyView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;

/**
 * The lazy panel view that contains the title search form
 * 
 * @author Vijay Silva
 */
class TitleSearchView extends LazyView {

	private TitleSearchPanel widget = null;
	private AnchorChangeListener anchorChangeNotifier;
	private UiCriteriaMode criteriaMode;

	/* Anchors recognized for parsing */
	private static final String SCHEDULE_ID = "scheduleId";
	private static final String TITLE_LIST_ID = "titleListId";
	private static final String LICENSE_ID = "licenseId";
	private static final String ACTION = "action";
	private static final String REQUEST_KEY = "request_key";
	private static final String TITLE_ID = "titleId";
	private static final String LIST_OF_ASSET_IDS = "listOfAssetIds";

	/* The anchor for content search */
	private static final String CONTENT_SEARCH_ANCHOR = "Content.Search";
	private static final String RETURN_ANCHOR = "returnAnchor";
	
	/**
	 * Constructor
	 */
	public TitleSearchView(UiCriteriaMode criteriaMode) {
		this.criteriaMode = criteriaMode;
	}

	@Override
	protected Widget createWidget() {
		widget = new TitleSearchPanel(ClientAuthorizationManager.getSecurity(), criteriaMode);
		if (anchorChangeNotifier != null)
			widget.setAnchorChangeNotifier(anchorChangeNotifier);

		return widget;
	}

	/* (non-Javadoc)
	 * @see com.tandbergtv.cms.portal.content.client.view.HasAnchor#getAnchor()
	 */
	@Override
	public String getAnchor() {
	    if (widget == null)
	    	return null;

	    /* Check if the title view is showing, and return that anchor */
	    if (widget.isShowingSearchResultsView()) {
	    	Widget searchResultWidget = widget.getSearchResultsView();
	    	if (searchResultWidget instanceof TitleView) {
	    		return ((TitleView) searchResultWidget).getAnchor();
	    	} else {
	    		return TitleListPanel.RESULTS_PAGE_ANCHOR;
	    	}
	    }

	    Map<String, String> anchorTokens = new HashMap<String, String>();
	    if (widget.getTitleListViewInput().getScheduleId() != 0) {
	    	long scheduleId = widget.getTitleListViewInput().getScheduleId();
	    	anchorTokens.put(SCHEDULE_ID, String.valueOf(scheduleId));
	    }
	    
	    if (widget.getTitleListViewInput().getTitleListId() != 0) {
	    	long titleListId = widget.getTitleListViewInput().getTitleListId();
	    	anchorTokens.put(TITLE_LIST_ID, String.valueOf(titleListId));
	    }
	    
	    if (widget.getTitleListViewInput().getLicenseId() != 0) {
            long licenseId = widget.getTitleListViewInput().getLicenseId();
            anchorTokens.put(LICENSE_ID, String.valueOf(licenseId));
        }
	    
	    /* Cannot have an action without a request key */
	    if (widget.getRequestKey() != null) {
			anchorTokens.put(REQUEST_KEY, widget.getRequestKey().toString());

			if (widget.getAction() != null) {
				anchorTokens.put(ACTION, widget.getAction());
			}

			if (widget.getListOfAssetIds() != null) {
				anchorTokens.put(LIST_OF_ASSET_IDS, widget.getListOfAssetIds());
			}
		}
	    
	    return new AnchorTokenizer().buildAnchor(anchorTokens);
	}

	/*
	 * Set the input that will be rendered by the search panel
	 */
	@Override
	public void setWidgetAnchor(String anchor) {
		//TODO: add handling for bulk edit anchors
		widget.reset();

		/* Remove the leading '?' */
		if (anchor != null && anchor.startsWith("?"))
			anchor = anchor.substring(1);

		/* Build the anchor tokens */
		AnchorTokenizer tokenizer = new AnchorTokenizer();
		Map<String, String> anchorTokens = tokenizer.parseAnchor(anchor);

		/* If using 'titleId', transform the anchor (and update tokens) accepted by Title View */
		if (anchorTokens.containsKey(TITLE_ID)) {
			anchor = transformAnchor(tokenizer, anchorTokens);
		}

		/* Anchor with Title ID always shows Title View */
		if (anchorTokens.containsKey(TitleViewInput.TITLE_ID_ANCHOR_KEY)) 
		{
			String returnAnchor = anchorTokens.get(RETURN_ANCHOR);
			TitleViewEventHandler titleViewEventHandler = new TitleViewEventHandler(returnAnchor);
			
			TitleView titleView;
			if (criteriaMode == UiCriteriaMode.SERIES)
				titleView = new SeriesTitleView();
			else
			    titleView = new TitleView();
				
			titleView.addViewCancelHandler(titleViewEventHandler);
			titleView.addViewAnchorChangeHandler(titleViewEventHandler);
			titleView.setAnchor(anchor);
			widget.setSearchResultsView(titleView);
			widget.showSearchResultsView();
		}
		/* Show the title search view */
		else {
			Long scheduleId = getLongValue(anchorTokens.get(SCHEDULE_ID));
			Long titleListId = getLongValue(anchorTokens.get(TITLE_LIST_ID));
			Long licenseId = getLongValue(anchorTokens.get(LICENSE_ID));
			String action = anchorTokens.get(ACTION);
			Long requestKey = getLongValue(anchorTokens.get(REQUEST_KEY));
			String listOfAssetIds = anchorTokens.get(LIST_OF_ASSET_IDS);
			
			/* Set the widget inputs */
			if (scheduleId != null) {
				widget.getTitleListViewInput().setScheduleId(scheduleId);
			}
			if (titleListId != null) {
				widget.getTitleListViewInput().setTitleListId(titleListId);
			}
			if (licenseId != null) {
                widget.getTitleListViewInput().setLicenseId(licenseId);
			}

			/* Cannot have an action without a request key */
			widget.setRequestKey(requestKey);
			if (requestKey != null) {
				widget.setAction(action);
				widget.setListOfAssetIds(listOfAssetIds);
			}
		}
	}

	/*
	 * Transform the anchor to a title view friendly anchor, and update the anchor map
	 */
	private String transformAnchor(AnchorTokenizer tokenizer, Map<String, String> anchorTokens) {
		renameMapKey(anchorTokens, TITLE_ID, TitleViewInput.TITLE_ID_ANCHOR_KEY);
		renameMapKey(anchorTokens, ACTION, TitleViewInput.ACTION_ANCHOR_KEY);
		renameMapKey(anchorTokens, REQUEST_KEY, TitleViewInput.REQUESTKEY_ANCHOR_KEY);
		renameMapKey(anchorTokens, LIST_OF_ASSET_IDS, TitleViewInput.ASSET_IDS_ANCHOR_KEY);
		return tokenizer.buildAnchor(anchorTokens);
	}

	/*
	 * Rename key in the map
	 */
	private void renameMapKey(Map<String, String> map, String oldKey, String newKey) {
		if (map.containsKey(oldKey)) {
			String value = map.remove(oldKey);
			map.put(newKey, value);
		}
	}
	
	/*
	 * Get the long value from the string, if possible
	 */
	private Long getLongValue(String longValue) {
		if (longValue != null && longValue.trim().length() > 0) {
			try {
				return new Long(longValue);
			} catch (RuntimeException e) {
				// TODO: If there is a failure parsing, should this show an error?
			}
		}

		return null;
	}

	/**
	 * Get the rendered title view. If this lazy widget has not be 'initialized', the title view
	 * will not be set (null).
	 * 
	 * @return The title view
	 */
	public TitleSearchPanel getTitleSearchPanel() {
		return widget;
	}

	/**
	 * Set the reverse anchor change listener used to notify the framework to update the anchor
	 * 
	 * @param anchorChangeNotifier the anchor change listener
	 */
	void setAnchorChangeNotifier(AnchorChangeListener anchorChangeListener) {
		this.anchorChangeNotifier = anchorChangeListener;
		if (widget != null)
			widget.setAnchorChangeNotifier(anchorChangeNotifier);
	}

	/*
	 * Event Listener for the title view cancel event
	 */
	private final class TitleViewEventHandler implements ViewCancelHandler, ViewAnchorChangeHandler {

		private String returnAnchor;
		
		public TitleViewEventHandler(String returnAnchor) {
			if (returnAnchor == null || returnAnchor.isEmpty())
				this.returnAnchor = CONTENT_SEARCH_ANCHOR;
			else
				this.returnAnchor = returnAnchor;
		}

		@Override
		public void onCancel(ViewCancelEvent event) {
			/* Set the new anchor */
			History.newItem(returnAnchor);
		}

		@Override
		public void onAnchorChange(ViewAnchorChangeEvent event) {
			delegateViewAnchorChangeEvent(event);
		}
	}
}
