/*
 * Created on Apr 21, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.cms.portal.content.client.title.model.compare;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.user.client.rpc.IsSerializable;

/**
 * UI Model for the IAssetDifference model class
 * 
 * @author Vijay Silva
 */
public class UIAssetDifference implements IsSerializable {

	/* Properties */
	private Long asset1Id;
	private Long asset2Id;
	private Map<String, String> metadata = new HashMap<String, String>();
	private String assetType;
	private String assetTypeDisplayName;
	private UIAssetDifference parentDifference;
	private List<UIAssetDifference> childDifferences = new ArrayList<UIAssetDifference>();
	private List<UIFieldDifference> fieldDifferences = new ArrayList<UIFieldDifference>();

	/* The Asset Type for files */
	private static final String FILE_ASSET_TYPE = "FILE";

	/**
	 * Constructor
	 */
	public UIAssetDifference() {
	}

	/**
	 * @return the ID of asset1 being compared
	 */
	public Long getAsset1Id() {
		return asset1Id;
	}

	/**
	 * @param asset1Id the ID for asset1 being compared
	 */
	public void setAsset1Id(Long asset1Id) {
		this.asset1Id = asset1Id;
	}

	/**
	 * @return The ID of asset2 being compared
	 */
	public Long getAsset2Id() {
		return asset2Id;
	}

	/**
	 * @param asset2Id The ID of asset2Id being compared
	 */
	public void setAsset2Id(Long asset2Id) {
		this.asset2Id = asset2Id;
	}

	/**
	 * @return The common metadata associated with the assets being compared
	 */
	public Map<String, String> getMetadata() {
		return metadata;
	}

	/**
	 * @param metadata The common metadata to set
	 */
	public void setMetadata(Map<String, String> metadata) {
		if (metadata == null) {
			metadata = new HashMap<String, String>();
		}

		this.metadata = metadata;
	}

	/**
	 * @return The asset type of the assets compared
	 */
	public String getAssetType() {
		return assetType;
	}

	/**
	 * @param assetType the asset type to set
	 */
	public void setAssetType(String assetType) {
		this.assetType = assetType;
	}

	/**
	 * @return The display name of the asset type of the assets compared
	 */
	public String getAssetTypeDisplayName() {
		return assetTypeDisplayName;
	}

	/**
	 * @param assetTypeDisplayName the asset type display name to set
	 */
	public void setAssetTypeDisplayName(String assetTypeDisplayName) {
		this.assetTypeDisplayName = assetTypeDisplayName;
	}

	/**
	 * @return The parent asset difference
	 */
	public UIAssetDifference getParentDifference() {
		return parentDifference;
	}

	/**
	 * @param parentDifference The parent asset difference to set
	 */
	public void setParentDifference(UIAssetDifference parentDifference) {
		this.parentDifference = parentDifference;
	}

	/**
	 * Get the child differences. Use this only to iterate through the differences. To add / remove
	 * child differences, use the {@link #addChildDifference(UIAssetDifference)} and
	 * {@link #removeChildDifference(UIAssetDifference)} methods.
	 * 
	 * @return the child differences.
	 */
	public List<UIAssetDifference> getChildDifferences() {
		return childDifferences;
	}

	/**
	 * Add child difference to the list, updating the parent of the child difference
	 * 
	 * @param childDifference the child asset difference to add
	 */
	public void addChildDifference(UIAssetDifference childDifference) {
		childDifference.setParentDifference(this);
		this.childDifferences.add(childDifference);
	}

	/**
	 * Remove the child difference
	 * 
	 * @param childDifference The child difference to remove
	 */
	public void removeChildDifference(UIAssetDifference childDifference) {
		if (this.childDifferences.remove(childDifference)) {
			childDifference.setParentDifference(null);
		}
	}

	/**
	 * Removes all child differences from this asset difference
	 */
	public void removeAllChildDifferences() {
		for (UIAssetDifference childDifference : this.childDifferences) {
			childDifference.setParentDifference(null);
		}

		this.childDifferences.clear();
	}

	/**
	 * @return the field differences
	 */
	public List<UIFieldDifference> getFieldDifferences() {
		return fieldDifferences;
	}

	/**
	 * Add a new field difference to the list of field differences
	 * 
	 * @param fieldDifference the fieldDifference to add
	 */
	public void addFieldDifference(UIFieldDifference fieldDifference) {
		fieldDifference.setAssetDifference(this);
		this.fieldDifferences.add(fieldDifference);
	}

	/**
	 * Remove the field difference from the maintained list of differences
	 * 
	 * @param fieldDifference The field difference to remove
	 */
	public void removeFieldDifference(UIFieldDifference fieldDifference) {
		if (this.fieldDifferences.remove(fieldDifference)) {
			fieldDifference.setAssetDifference(null);
		}
	}

	/**
	 * Remove all maintained field differences
	 */
	public void removeAllFieldDifferences() {
		for (UIFieldDifference fieldDifference : this.fieldDifferences) {
			fieldDifference.setAssetDifference(null);
		}

		this.fieldDifferences.clear();
	}

	/**
	 * Determine if this asset difference is for a file
	 * 
	 * @return true if this asset difference is for a file, false otherwise
	 */
	public boolean isFileAsset() {
		return FILE_ASSET_TYPE.equals(this.assetType);
	}

	/**
	 * Get the ancestor difference that is not a file difference
	 * 
	 * @return The ancestor difference that is not a file difference
	 */
	public UIAssetDifference getAncestorDifference() {
		/* If parent is for a file, get ancestor of the parent */
		UIAssetDifference parent = getParentDifference();
		if (parent != null && parent.isFileAsset()) {
			return parent.getAncestorDifference();
		}

		return parent;
	}

	/**
	 * Determine the difference type for this asset difference
	 * 
	 * @return The asset difference type
	 */
	public UIAssetDifferenceType getDifferenceType() {
		UIAssetDifferenceType differenceType = null;

		/* Check if the first asset ID is present */
		if (getAsset1Id() == null) {
			differenceType = UIAssetDifferenceType.ASSET1_NOT_DEFINED;
		}
		/* Check if the second asset ID is present */
		else if (getAsset2Id() == null) {
			differenceType = UIAssetDifferenceType.ASSET2_NOT_DEFINED;
		}
		/* Check if field differences exist */
		else if (fieldDifferences.size() > 0) {
			differenceType = UIAssetDifferenceType.METADATA_DIFFERENT;
		}
		/* Default, there are differences in the children */
		else {
			differenceType = UIAssetDifferenceType.NO_DIFFERENCE;
		}

		return differenceType;
	}
}
