/*
 * Created on Feb 19, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.model.metadata.asset;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.user.client.rpc.IsSerializable;
import com.tandbergtv.cms.portal.content.client.title.model.ICopyable;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetDefinition;

/**
 * UI Model object that represents an asset
 * 
 * @author Vijay Silva
 */
public abstract class UIAsset implements IsSerializable, ICopyable {

	/* The name of the 'fake' root field */
	private static final String ROOT_FIELD_NAME = UIAssetDefinition.ROOT_FIELD_NAME;

	/* Properties */
	private Long id;
	private UIAsset parentAsset;
	private Long originalAssetId;
	private Integer originalAssetRevision;
	private boolean active = true;
	private UIComplexField rootField;
	private boolean thumbnailAvailable;
	private List<UIAssetFile> files = new ArrayList<UIAssetFile>();

	/**
	 * Constructor - Creates a new root asset
	 */
	public UIAsset() {
		rootField = new UIComplexField();
		rootField.setParentAsset(this);
		rootField.setName(ROOT_FIELD_NAME);
		this.thumbnailAvailable = false;
	}

	/**
	 * @return the id
	 */
	public Long getId() {
		return id;
	}

	/**
	 * @param id the id to set
	 */
	public void setId(Long id) {
		this.id = id;
	}

	/**
	 * @return the originalAssetId
	 */
	public Long getOriginalAssetId() {
		return originalAssetId;
	}

	/**
	 * @param originalAssetId the originalAssetId to set
	 */
	public void setOriginalAssetId(Long originalAssetId) {
		this.originalAssetId = originalAssetId;
	}

	/**
	 * @return the originalAssetRevision
	 */
	public Integer getOriginalAssetRevision() {
		return originalAssetRevision;
	}

	/**
	 * @param originalAssetRevision the originalAssetRevision to set
	 */
	public void setOriginalAssetRevision(Integer originalAssetRevision) {
		this.originalAssetRevision = originalAssetRevision;
	}

	/**
	 * Get the root asset
	 * 
	 * @return The root asset
	 */
	public UIAsset getRootAsset() {
		return (parentAsset != null) ? parentAsset.getRootAsset() : this;
	}

	/**
	 * @return the parent asset
	 */
	public UIAsset getParentAsset() {
		return parentAsset;
	}

	/**
	 * @param parentAsset the parent asset to set
	 */
	public void setParentAsset(UIAsset parentAsset) {
		this.parentAsset = parentAsset;
	}

	/**
	 * Get the asset type name for this asset.
	 * 
	 * @return the asset type
	 */
	public abstract String getAssetType();

	/**
	 * @return true if the asset if active, false otherwise
	 */
	public boolean isActive() {
		return active;
	}

	/**
	 * @param active The active flag value for the asset
	 */
	public void setActive(boolean active) {
		this.active = active;
	}
	
	/**
	 * @return true if the thumbnail is available, false otherwise
	 */
    public boolean isThumbnailAvailable() {
    	return thumbnailAvailable;
    }

	/**
     * @param thumbnailAvailable Set the flag indicating if the thumbnail is available
     */
    public void setThumbnailAvailable(boolean thumbnailAvailable) {
    	this.thumbnailAvailable = thumbnailAvailable;
    }

	/**
	 * The root field (always exists and cannot be removed from the asset tree)
	 * 
	 * @return the root field
	 */
	public UIComplexField getRootField() {
		return this.rootField;
	}

	/**
	 * @return the files
	 */
	public List<UIAssetFile> getFiles() {
		return files;
	}

	/**
	 * Add a new file to the list of files for this asset
	 * 
	 * @param file
	 */
	public void addFile(UIAssetFile file) {
		file.setParentAsset(this);
		if (!files.contains(file))
			files.add(file);
	}

	/**
	 * Remove the file from the list of files for this Asset
	 * 
	 * @param file The file to remove
	 * @return true if the file was removed, false otherwise
	 */
	public boolean removeFile(UIAssetFile file) {
		boolean result = files.remove(file);

		if (result)
			file.setParentAsset(null);

		return result;
	}

	/**
	 * Create a copy of this asset
	 * 
	 * @see com.tandbergtv.cms.portal.content.client.title.model.ICopyable#createCopy()
	 */
	public abstract UIAsset createCopy();

	/**
	 * Copies all the properties of this asset to the copy instance provided
	 * 
	 * @param copy The instance to copy all current properties to
	 */
	protected void copyProperties(UIAsset copy) {
		copy.setId(this.getId());
		copy.setActive(this.isActive());
		copy.setOriginalAssetId(this.getOriginalAssetId());
		copy.setOriginalAssetRevision(this.getOriginalAssetRevision());
		copy.setThumbnailAvailable(this.isThumbnailAvailable());

		/* Copy all the fields */
		copy.rootField = this.rootField.createCopy();

		/* Copy all the files */
		for (UIAssetFile file : this.getFiles()) {
			copy.addFile(file.createCopy());
		}
	}

	/**
	 * The accept method used for the visitor pattern
	 * 
	 * @param visitor The visitor
	 */
	public abstract void accept(IUIAssetVisitor visitor);
	
	public void clearFields()
	{
		getRootField().getChildren().removeAll();
		if(this instanceof UIGroupAsset)
		{
			List<UIAsset> assets = ((UIGroupAsset)this).getChildren().getAll();
			if(assets != null)
			{
				for(UIAsset asset: assets)
				{
					asset.clearFields();
				}
			}
		}
	}
}
