/*
 * Created on Feb 19, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.model.metadata.asset;

/**
 * UI Model object that represents a complex field. A Complex field has one or more child fields
 * which could be other complex fields or simple fields.
 * 
 * @author Vijay Silva
 */
public class UIComplexField extends UIField {

	/* Properties */
	private IUIFieldMap children = new UIFieldMap();

	/**
	 * Default constructor
	 */
	public UIComplexField() {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setParentAsset(UIAsset parentAsset) {
		super.setParentAsset(parentAsset);

		/* Set the parent asset for all the children as well */
		for (UIField child : children.getAll()) {
			child.setParentAsset(parentAsset);
		}
	}

	/**
	 * @return the children
	 */
	public IUIFieldMap getChildren() {
		return children;
	}

	/**
	 * Add a new field to the list of child fields. If there are more than one field with the same
	 * name, this new field is added to the end of the list of the fields with same name.
	 * 
	 * @param field The field to add
	 */
	public void addField(UIField field) {
		this.updateFieldToAdd(field);
		children.addField(field);
	}

	/**
	 * Add a new field to the list of child fields. The field is added at the specified index in the
	 * list of fields with the same name. If the index is invalid, a runtime exception is thrown.
	 * 
	 * @param field The field to add
	 * @param index The 0-based index at which to add the field. The index must be a value between 0
	 *        and the number of fields with the same name.
	 */
	public void addField(UIField field, int index) {
		this.updateFieldToAdd(field);
		children.addField(field, index);
	}

	/*
	 * Update the properties of the child field being added to this field
	 */
	private void updateFieldToAdd(UIField field) {
		field.setParentAsset(this.getParentAsset());
		field.setParentField(this);
	}

	/**
	 * Remove an existing field from child fields
	 * 
	 * @param field the field to remove
	 * @return true if removed, false otherwise
	 */
	public boolean removeField(UIField field) {
		boolean removed = children.removeField(field);

		if (removed) {
			field.setParentAsset(null);
			field.setParentField(null);
		}

		return removed;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public UIComplexField createCopy() {
		UIComplexField copy = new UIComplexField();

		/* Copy the properties */
		this.copyComplexFieldProperties(copy);

		return copy;
	}

	/*
	 * Copy all properties to the copy instance
	 */
	protected void copyComplexFieldProperties(UIComplexField copy) {
		super.copyProperties(copy);

		/* Copy all child fields */
		for (UIField child : this.getChildren().getAll()) {
			copy.addField(child.createCopy());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(IUIFieldVisitor visitor) {
		visitor.visit(this);
	}
}
