/*
 * Created on Feb 24, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.model.metadata.asset;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.user.client.rpc.IsSerializable;

/**
 * Maintains the collections of fields using the field name as the key.
 * 
 * @author Vijay Silva
 */
public class UIFieldMap implements IUIFieldMap, IsSerializable {

	/* Properties */
	private Map<String, List<UIField>> fieldMap = new HashMap<String, List<UIField>>();

	/**
	 * Constructor
	 */
	public UIFieldMap() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<UIField> getAll() {
		List<UIField> allFields = new ArrayList<UIField>();

		/* Add all fields */
		for (List<? extends UIField> fields : this.fieldMap.values()) {
			if (fields != null)
				allFields.addAll(fields);
		}

		return allFields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeAll() {
		this.fieldMap.clear();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getFieldCount() {
		int count = 0;

		for (List<? extends UIField> fields : this.fieldMap.values()) {
			count += (fields != null) ? fields.size() : 0;
		}

		return count;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<UIField> getFields(String name) {
		return this.getOrCreateList(name);
	}

	/*
	 * Attempts to validate the type of the fields
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <FieldType extends UIField> List<FieldType> getFields(String name,
	        Class<FieldType> fieldType) {
		List<FieldType> typedFields = null;

		List<UIField> fields = getFields(name);
		UIField firstField = (fields.size() > 0) ? fields.get(0) : null;
		if (firstField == null || firstField.getClass().equals(fieldType)) {
			typedFields = (List<FieldType>) fields;
		}

		return typedFields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public UIField getField(String name) {
		List<UIField> fields = this.fieldMap.get(name);
		return (fields != null && fields.size() > 0) ? fields.get(0) : null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getFieldCount(String fieldName) {
		List<? extends UIField> fields = fieldMap.get(fieldName);
		return (fields != null) ? fields.size() : 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addField(UIField field) {
		String fieldName = field.getName();
		List<UIField> fields = this.getOrCreateList(fieldName);
		validateFieldType(field, fields);
		fields.add(field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addField(UIField field, int index) {
		String fieldName = field.getName();
		List<UIField> fields = this.getOrCreateList(fieldName);
		validateFieldType(field, fields);
		fields.add(index, field);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean removeField(UIField field) {
		List<UIField> fields = fieldMap.get(field.getName());
		return (fields != null) ? fields.remove(field) : false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeFields(String fieldName) {
		fieldMap.remove(fieldName);
	}

	/*
	 * Attempts the get the list from the field map. If the list does not exist, creates and adds
	 * the list to the field map.
	 */
	private List<UIField> getOrCreateList(String key) {
		List<UIField> fieldList = fieldMap.get(key);

		if (fieldList == null) {
			fieldList = new ArrayList<UIField>();
			fieldMap.put(key, fieldList);
		}

		return fieldList;
	}

	/* Check if the new field matches the type of the existing fields */
	private void validateFieldType(UIField field, List<UIField> fields) {
		if (fields.size() > 0) {
			UIField firstField = fields.get(0);
			if (!firstField.getClass().equals(field.getClass())) {
				String fieldClass = field.getClass().getName();
				String expectedClass = firstField.getClass().getName();
				throw new RuntimeException("Cannot add field of type: " + fieldClass
				        + " to list of fields of type: " + expectedClass + " for field: "
				        + field.getName());
			}
		}
	}
}
