/*
 * Created on Feb 19, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.model.metadata.asset;

import java.util.List;

/**
 * UI Model object that represents a 'Group' asset
 * 
 * @author Vijay Silva
 */
public class UIGroupAsset extends UIAsset {

	/* The Properties */
	private String assetType;
	private IUIAssetMap children = new UIAssetMap();
	private Long seriesDocumentId;	
	private Long seasonAssetId;

	/**
	 * Constructor
	 */
	public UIGroupAsset() {
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getAssetType() {
		return assetType;
	}

	/**
	 * @param assetType the assetType to set
	 */
	public void setAssetType(String assetType) {
		this.assetType = assetType;
	}

	/**
	 * @return the children
	 */
	public IUIAssetMap getChildren() {
		return children;
	}

	/**
	 * @param children the children to set
	 */
	public void setChildren(IUIAssetMap children) {
		if (children != null) {
			for (UIAsset asset : children.getAll()) {
				asset.setParentAsset(this);
			}
		}

		this.children = (children != null) ? children : new UIAssetMap();
	}

	/**
	 * Get all child item assets with matching asset type
	 * 
	 * @param assetType The asset type
	 * @return The child item assets with matching asset type or empty list if no matches exist
	 * @throws RuntimeException thrown if the assets with matching asset type are not item assets
	 */
	public List<UIItemAsset> getChildItems(String assetType) {
		return this.children.getItems(assetType);
	}

	/**
	 * Get all child group assets with matching asset type
	 * 
	 * @param assetType The asset type
	 * @return The child group assets with matching asset type or empty list if no matches exist
	 * @throws RuntimeException thrown if the assets with matching asset type are not group assets
	 */
	public List<UIGroupAsset> getChildGroups(String assetType) {
		return this.children.getGroups(assetType);
	}

	/**
	 * Add a new child asset to the children of this group
	 * 
	 * @param asset The child asset
	 */
	public void addChild(UIAsset asset) {
		if (asset != null) {
			asset.setParentAsset(this);
			children.addAsset(asset);
		}
	}

	/**
	 * Remove a child asset from this group
	 * 
	 * @param asset The child asset
	 */
	public boolean removeChild(UIAsset asset) {
		boolean removed = children.removeAsset(asset);
		if (removed) {
			asset.setParentAsset(null);
		}
		return removed;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public UIGroupAsset createCopy() {
		UIGroupAsset copy = new UIGroupAsset();

		/* Copy the properties */
		this.copyProperties(copy);

		return copy;
	}

	/**
	 * Copy all properties of this item asset to the copy instance provided
	 * 
	 * @param copy The instance to copy properties to
	 */
	protected void copyProperties(UIGroupAsset copy) {
		/* Copy all properties of the parent type */
		super.copyProperties(copy);

		/* Copy of properties of this group type */
		copy.setAssetType(this.getAssetType());
		for (UIAsset child : this.getChildren().getAll()) {
			copy.addChild(child.createCopy());
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(IUIAssetVisitor visitor) {
		visitor.visit(this);
	}
	
	/**
	 * @return the seriesDocumentId
	 */
	public Long getSeriesDocumentId() {
		return seriesDocumentId;
	}

	/**
	 * @param seriesDocumentId the seriesDocumentId to set
	 */
	public void setSeriesDocumentId(Long seriesDocumentId) {
		this.seriesDocumentId = seriesDocumentId;
	}

	/**
	 * @return the seasonAssetId
	 */
	public Long getSeasonAssetId() {
		return seasonAssetId;
	}

	/**
	 * @param seasonAssetId the seasonAssetId to set
	 */
	public void setSeasonAssetId(Long seasonAssetId) {
		this.seasonAssetId = seasonAssetId;
	}
	
}
